#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

""" BBF YANG Notifications Example

This script script subscribes to notifications from the Netconf Server and
listens for BBF onu-presence-state-change change events.

For reference, an example of a raw onu-presence-state-change event is shown below.

<notification xmlns="urn:ietf:params:xml:ns:netconf:notification:1.0">
  <eventTime>2021-01-28T14:19:57Z</eventTime>
  <interfaces-state xmlns="urn:ietf:params:xml:ns:yang:ietf-interfaces">
    <interface>
      <name>channeltermination.1/0/1</name>
      <type xmlns:bbf-xponift="urn:bbf:yang:bbf-xpon-if-type">bbf-xponift:channel-termination</type>
      <channel-termination xmlns="urn:bbf:yang:bbf-xpon">
        <onu-presence-state-change xmlns="urn:bbf:yang:bbf-xpon-onu-state">
          <detected-serial-number>BFWS00123193</detected-serial-number>
          <last-change>2021-01-28T14:19:57.545976Z</last-change>
          <onu-presence-state xmlns:bbf-xpon-onu-types="urn:bbf:yang:bbf-xpon-onu-types">bbf-xpon-onu-types:onu-present</onu-presence-state>
          <onu-id>1</onu-id>
          <detected-registration-id></detected-registration-id>
          <v-ani-ref>vani-BFWS00123193</v-ani-ref>
        </onu-presence-state-change>
      </channel-termination>
    </interface>
  </interfaces-state>
</notification>

"""

import argparse
import sys
import json
from lxml import etree
import ncclient
from ncclient import manager
import os
from getpass import getpass
from datetime import datetime, timezone
import time
from netconf_driver import NetconfDriver

if __name__ == '__main__':

    # Example: ./config_add_ctag_svc.py --onu ALPHe30cadcf --olt_tag 200 --onu_tag 2

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-f", "--filter", action="store", dest="filter", default=None, required=False, help="Subscription filter.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    # Connect to the Netconf Server
    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)

    # Subscribe to notifications from the Netconf Server
    nc.subscribe()

    # Listen for notification events (type CTRL-C to terminate)
    while True:
        # Check for connection failures
        if not nc.is_connected():
            print("ERROR: Connection to server failed.")
            nc.subscribe(stream="NETCONF", filter=args.filter)
            time.sleep(1)
        else:
            # Get the next notification
            notification = nc.get_notification()
            if notification:
                if args.verbose:
                    print(etree.tostring(notification.notification_ele, pretty_print=True, encoding="unicode").rstrip())

                NSMAP = {
                    'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
                    'ncEvent' : "urn:ietf:params:xml:ns:netconf:notification:1.0",
                    'if' : "urn:ietf:params:xml:ns:yang:ietf-interfaces",
                    'bbf-xpon' : "urn:bbf:yang:bbf-xpon",
                    'bbf-xpon-onu-s' : "urn:bbf:yang:bbf-xpon-onu-state",
                    }


                # Get the etree node for the event notification
                #import pdb; pdb.set_trace()
                event_node = notification.notification_ele

                # Get the event time
                event_time = ""
                if event_node is not None:
                    node = event_node.find("ncEvent:eventTime", namespaces=NSMAP)
                    if node is not None:
                        event_time = node.text

                # Convert the UTC time to local time
                if event_time:
                    try:
                        event_utc_time = datetime.strptime(event_time, "%Y-%m-%dT%H:%M:%SZ")
                        event_time = event_utc_time.replace(tzinfo=timezone.utc).astimezone(tz=None).strftime("%Y-%m-%d %H:%M:%S")
                    except ValueError:
                        # Ignore date/time conversion failures
                        pass


                # Get the interface name
                olt_if_name = ""
                if event_node is not None:
                    node = event_node.find("if:interfaces-state/if:interface/if:name", namespaces=NSMAP)
                    if node is not None:
                        olt_if_name = node.text

                # Get the ONU Serial Number
                onu_serial_number = ""
                if event_node is not None:
                    node = event_node.find("if:interfaces-state/if:interface/bbf-xpon:channel-termination/bbf-xpon-onu-s:onu-presence-state-change/bbf-xpon-onu-s:detected-serial-number", namespaces=NSMAP)
                    if node is not None:
                        onu_serial_number = node.text

                # Get the ONU ID
                onu_id = ""
                if event_node is not None:
                    node = event_node.find("if:interfaces-state/if:interface/bbf-xpon:channel-termination/bbf-xpon-onu-s:onu-presence-state-change/bbf-xpon-onu-s:onu-id", namespaces=NSMAP)
                    if node is not None:
                        onu_id = node.text

                # Get the V-ANI
                vani_name = ""
                if event_node is not None:
                    node = event_node.find("if:interfaces-state/if:interface/bbf-xpon:channel-termination/bbf-xpon-onu-s:onu-presence-state-change/bbf-xpon-onu-s:v-ani-ref", namespaces=NSMAP)
                    if node is not None:
                        vani_name = node.text

                # Get the ONU state
                onu_presence_state = ""
                if event_node is not None:
                    node = event_node.find("if:interfaces-state/if:interface/bbf-xpon:channel-termination/bbf-xpon-onu-s:onu-presence-state-change/bbf-xpon-onu-s:onu-presence-state", namespaces=NSMAP)
                    if node is not None:
                        onu_presence_state = node.text

                # Only print BBF ONU Presence State Change Events
                if onu_presence_state:
                    print(f"\n{event_time} ONU Presence State Change Event")
                    print(f"  OLT PON = {olt_if_name}")
                    print(f"  ONU Serial Number = {onu_serial_number}")
                    print(f"  ONU ID = {onu_id}")
                    print(f"  V-ANI = {vani_name}")
                    print(f"  State = {onu_presence_state}")
