#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2021 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

""" Configure an ONU to OLT Inventory.

This Tibit Yang script will add a new ONU to the OLT inventory..
The ONU will receive a ONU ID and an Alloc ID, but it will not be configured with service or other details.
The default SLA will be Max.

Example - Add ONU ALPHe30cadcf to OLT 70:b3:d5:52:37:24

  ./add_ctag_service/edit_config_add_ctag_svc.py \
      --olt 70:b3:d5:52:37:24 \
      --onu ALPHe30cadcf \

usage: edit_config_provision_onu.py [--help] [-h HOST] [-p PORT]
                                   --olt OLT
                                   --onu ONU 

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --olt OLT             OLT MAC Address (e.g., 70:b3:d5:52:37:24) (default:
                        None)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
"""

import argparse
import itertools
from lxml import etree
import os
import sys
import getpass

sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from common.netconf_driver import NetconfDriver

def get_onu_ids(nc, options):
    """
    Get a dictionary of ONU ID values configured in ONU Inventory for an OLT.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        onu_ids: A dictionary mapping ONU serial number to ONU ID configured in ONU inventory
    """

    # Send a Netconf <get> request to retrieve the ONU Inventory for an OLT
    OLT_CFG_ONU_INVENTORY = '''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:olt xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                <tibitcntlr:olt>
                    <name>{{OLT}}</name>
                    <tibitcntlr:onus/>
                </tibitcntlr:olt>
                </tibitcntlr:olt>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=OLT_CFG_ONU_INVENTORY, options=options, message="/tibit-pon-controller-db:olt/tibit-pon-controller-db:olt/tibit-pon-controller-db:onus")

    # Parse the Netconf response and build a dictionary mapping ONU serial number
    # to ONU IDs from the XML response data.
    NSMAP = {
        'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
        'tibit' : "urn:com:tibitcom:ns:yang:controller:db",
        }
    root = etree.fromstring(rsp_xml)
    onu_ids = {}
    for onu in root.findall(f"nc:data/tibit:olt/tibit:olt/tibit:onus", namespaces=NSMAP):
        onu_serial_number = onu.find("tibit:id", namespaces=NSMAP)
        if onu_serial_number is not None:
            onu_serial_number = onu_serial_number.text
            onu_id = onu.find("tibit:alloc-id-omcc", namespaces=NSMAP)
            if onu_id is not None:
                onu_id = onu_id.text
                onu_ids[onu_serial_number] = int(onu_id)

    return onu_ids

def get_next_onu_id(nc, options):
    """
    Get the next free ONU ID that has _not_ been configured in the ONU Inventory.

    If the list of empty, return the first valid ONU ID value '1'. Otherwise find the first
    unused value starting from from 1 to 128.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        next_onu_id: The next available ONU ID value or 'None'.
    """
    next_onu_id = None
    onu_ids = get_onu_ids(nc, options)
    if len(onu_ids.values()) == 0:
        next_onu_id = 1
    else:
        for value in range(1,129):
            if value not in onu_ids.values():
                next_onu_id = value
                break

    return next_onu_id

def get_onu_id_from_serial_number(nc, serial_number, options):
    """
    Get the configured ONU ID value for an ONU by serial number.

    Args:
        nc: Netconf driver object.
        serial_number: ONU Vendor-specific serial number
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        onu_id: The ONU ID configured for this ONU or 'None'.
    """
    onu_ids = get_onu_ids(nc, options)
    onu_id = None
    if serial_number in onu_ids:
        onu_id = onu_ids[serial_number]

    return onu_id

def get_alloc_ids(nc, options):
    """
    Get a dictionary of Alloc ID values per ONU configured in ONU Inventory for an OLT.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        alloc_ids: A dictionary of a list Alloc IDs for each ONU configured in ONU inventory
    """

    # Send a Netconf <get> request to retrieve the ONU Inventory for an OLT
    OLT_CFG_ONU_INVENTORY = '''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:olt xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                <tibitcntlr:olt>
                    <name>{{OLT}}</name>
                    <tibitcntlr:onus/>
                </tibitcntlr:olt>
                </tibitcntlr:olt>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=OLT_CFG_ONU_INVENTORY, options=options, message="/tibit-pon-controller-db:olt/tibit-pon-controller-db:olt/tibit-pon-controller-db:onus")

    # Parse the Netconf response and build a list of Alloc IDs from the XML response data.
    # Each ONU can be configured with multiple Alloc IDs. Therefore, build a dictionary with the
    # format { ONU : [] } that contains a list of Alloc IDs for each ONU entry in the dictionary.
    NSMAP = {
        'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
        'tibit' : "urn:com:tibitcom:ns:yang:controller:db",
        }
    root = etree.fromstring(rsp_xml)
    alloc_ids = {}
    for onu in root.findall(f"nc:data/tibit:olt/tibit:olt/tibit:onus", namespaces=NSMAP):
        onu_serial_number = onu.find("tibit:id", namespaces=NSMAP)
        if onu_serial_number is not None:
            onu_serial_number = onu_serial_number.text
            alloc_ids[onu_serial_number] = []
            for alloc_id in onu.findall("tibit:olt-service/tibit:unicast-id", namespaces=NSMAP):
                alloc_id = alloc_id.text
                alloc_ids[onu_serial_number].append(int(alloc_id))

    return alloc_ids

def get_next_alloc_id(nc, options):
    """
    Get the next free Alloc ID that has _not_ been configured in the ONU Inventory.

    If the list of empty, return the first valid Alloc ID value '1154'. Otherwise find the first
    unused value starting from from 1154 to 1534.

    Args:
        nc: Netconf driver object.
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        next_alloc_id: The next available Alloc ID value or 'None'.
    """
    next_alloc_id = None
    alloc_ids = get_alloc_ids(nc, options)
    alloc_ids = alloc_ids.values()
    alloc_ids = list(itertools.chain.from_iterable(alloc_ids))
    if len(alloc_ids) == 0:
        next_alloc_id = 1154
    else:
        # Need to exclude values reserved by HW: 0x4FF, 0x5FF (already excluded from the range above)
        restricted_values = [0x4FF, 0x5FF]
        for value in range(1154,1535):
            if value not in alloc_ids:
                if value not in restricted_values:
                    next_alloc_id = value
                    break

    return next_alloc_id

def get_alloc_id_from_serial_number(nc, serial_number, options):
    """
    Get the configured Alloc ID value for an ONU by serial number.

    Note, this routine assumes only one Alloc ID is configured for each ONU. As a result,
    only the first Alloc ID value is returned.

    Args:
        nc: Netconf driver object.
        serial_number: ONU Vendor-specific serial number
        options: Netconf request options ({{VAR}}=value) dictionary.

    Returns:
        alloc_id: The Alloc ID configured for this ONU or 'None'.
    """
    alloc_ids = get_alloc_ids(nc, options)
    alloc_id = None
    if serial_number in alloc_ids:
        if len(alloc_ids[serial_number]) > 0:
            alloc_id = alloc_ids[serial_number][0]

    return alloc_id

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., 70:b3:d5:52:37:24)")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")

    parser.parse_args()
    args = parser.parse_args()

    if args.user:
        user = args.user
    else:
        user = getpass.getuser()

    passwd = getpass.getpass(prompt="Please enter the password for "+user+": ")
    nc = NetconfDriver(host=args.host, port=args.port, user=user, passwd=passwd, verbose=False)
    if nc:

        options = {
            "{{OLT}}" : args.olt,
            "{{ONU}}" : args.onu,
        }


        # Get the ONU ID for this ONU
        onu_id = get_onu_id_from_serial_number(nc, args.onu, options)
        if onu_id is None:
            onu_id = get_next_onu_id(nc, options)
        options['{{ONU_ID}}'] = onu_id

        # Get the Alloc ID for this ONU
        alloc_id = get_alloc_id_from_serial_number(nc, args.onu, options)
        if alloc_id is None:
            alloc_id = get_next_alloc_id(nc, options)
        options['{{ALLOC_ID}}'] = alloc_id

        data_xml = ''' <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760"> 
                     <edit-config>
                       <target>
                         <running />
                       </target>
                       <config>
                          <olt xmlns=\"urn:com:tibitcom:ns:yang:controller:db\">
                              <olt>
                                 <name>{{OLT}}</name>
                                 <device-id>{{OLT}}</device-id>
                                 <!-- ONU Inventory -->
                                 <onus>
                                    <id>{{ONU}}</id>
                                    <alloc-id-omcc>{{ONU_ID}}</alloc-id-omcc>
                                     <olt-service>
                                          <id>0</id>
                                          <unicast-id>{{ALLOC_ID}}</unicast-id>
                                     </olt-service>
                                 </onus>
                               </olt>
                           </olt>
                       </config>
                     </edit-config>
                  </rpc>'''

        # Configure the ONU ID and Alloc ID in the ONU inventory
        nc.edit_config(data_xml=data_xml,options=options,message="Provision an ONU at inventory")
        # Display a summary of what was configured
        print(f"\nProvisioned ONU {options['{{ONU}}']} to OLT {options['{{OLT}}']}")
        print(f" PON Alloc ID:  "+str(alloc_id))
        print(f" PON ONU ID:    "+str(onu_id))
