"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#

Connector for using MongoDB calls in Django API
"""
import copy
import pymongo
import json
import gridfs
import base64
import datetime
import inspect
import os
import io

from bson.binary import Binary
from PIL import Image
from pymongo import ReturnDocument
from resizeimage import resizeimage
from rest_framework import status
from pymongo.errors import *

from api.settings import IN_PRODUCTION

from database_manager import database_manager
from utils.tools import get_nested_value


class MongoDBConnector:

    def __init__(self, database_id: str):
        """Initialize the MongoDBConnector with database host info"""
        self.database_id = database_id
        self.database = database_manager.get_all_databases()[database_id][0]
        self.user_database = database_manager.user_database

        self.development_databases_files = "databases.json"
        self.development_logo_directory = "../src/assets/images/pictures/"
        self.development_footer_text = "../src/assets/text-files/"
        self.production_databases_files = "/var/www/html/api/databases.json"
        self.production_logo_directory = "/var/www/html/tibit-ponmgr/assets/images/pictures/"
        self.production_footer_text = "/var/www/html/tibit-ponmgr/assets/text-files/"
        self.database_seed_base_dir = "/var/www/html/api/databaseSeedFiles/"  # instead of having two distinct production and development variables, like above, the value of this variable is modified to be production or development

        if IN_PRODUCTION:
            with open("/var/www/html/api/user_database.json"):
                pass
            with open("/var/www/html/api/user_migration.json"):
                pass
            self.database_seed_base_dir = "/var/www/html/api/databaseSeedFiles/"
        else:
            with open("user_database.json"):
                pass
            with open("user_migration.json"):
                pass
            self.database_seed_base_dir = "databaseSeedFiles/"

    def seed_database(self, to_include, user_email):
        """Insert default documents and files into database collections"""
        response = [status.HTTP_200_OK, "Database was successfully initialized.", None, {}]

        try:
            database_manager.get_database(self.database_id)
            # Insert default alarm configuration documents
            self.initialize_alarm_configs_collection(to_include, "CNTL-ALARM-CFG")
            self.initialize_alarm_configs_collection(to_include, "OLT-ALARM-CFG")
            self.initialize_alarm_configs_collection(to_include, "ONU-ALARM-CFG")

            # Insert default device configuration documents
            self.initialize_device_configs_collection(to_include, "CNTL-CFG")
            self.initialize_device_configs_collection(to_include, "OLT-CFG")
            self.initialize_device_configs_collection(to_include, "ONU-CFG")
            self.initialize_device_configs_collection(to_include, "SWI-CFG")

            # Insert default/global PON Automation configuration documents
            self.initialize_pon_auto_configs_collection(to_include, "CNTL-AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "OLT-AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "ONU-AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "SWI-AUTO-CFG")
            self.initialize_pon_auto_configs_collection(to_include, "AUTO-CFG")

            # Insert default SLA, Downstream mapping, and Service configuration documents
            self.initialize_sla_configs_collection(to_include)
            self.initialize_ds_map_configs_collection(to_include)
            self.initialize_service_configs_collection(to_include)

            # Upload default firmware and picture files
            self.initialize_olt_firmware_bucket(to_include)
            self.initialize_onu_firmware_bucket(to_include)
            self.initialize_picture_bucket(to_include)

            response[2] = to_include.keys()
        except Exception as err:
            self.status_log("EXCEPTION (seed_database)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not seed database due to {type(err).__name__}::{sys.exc_info()[1]}", None, None]

        action_data = {"old": "none", "new": "seed files", "collection": "databaseSeedFiles/", "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    def set_seed_pon_mode(self, json_seed):
        directory = self.database_seed_base_dir + "documents/deviceConfigs/olts/"
        response = [status.HTTP_200_OK, "PON Mode Seed was successfully set.", None, {}]
        try:
            if directory != "":
                for filename in os.listdir(directory):
                    if filename.lower().endswith(".json"):
                        with open(directory + filename, 'r') as json_file:
                            new_doc = json.load(json_file)
                            old_doc = copy.deepcopy(new_doc)
                        if hasattr(new_doc, "OLT"):
                            new_doc['OLT']['PON Mode'] = json_seed
                        with open(directory + filename, 'w') as json_file:
                            json_file.write(json.dumps(new_doc, indent=4))
                        return [status.HTTP_200_OK, "PON Mode Seed was successfully set.", new_doc, old_doc]
        except Exception as err:
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not change seed pon mode due to {type(err).__name__}::{sys.exc_info()[1]}", None, None]
        return response

    def initialize_alarm_configs_collection(self, collections_to_include, collection):
        """
        Insert all default alarm configuration documents into the specified database collection.
        :param collections_to_include: list of collections and documents to be initialized
        :param collection: the alarm configuration collection to initialize. (CNTL-ALARM-CFG, OLT-ALARM-CFG, or ONU-ALARM-CFG)
        :return: None
        """
        directory = ""

        # Set directory
        if collection == "CNTL-ALARM-CFG":
            directory = self.database_seed_base_dir + "documents/alarmConfigs/controllers/"
        if collection == "OLT-ALARM-CFG":
            directory = self.database_seed_base_dir + "documents/alarmConfigs/olts/"
        if collection == "ONU-ALARM-CFG":
            directory = self.database_seed_base_dir + "documents/alarmConfigs/onus/"

        if directory != "":
            for filename in os.listdir(directory):
                # Strip filename
                doc_id = filename[14:-5]
                if collection == "CNTL-ALARM-CFG":
                    doc_id = doc_id[1:]
                # Insert document if to be included
                if collection in collections_to_include and "-ALARM-CFG" in filename and filename.lower().endswith(
                        ".json"):
                    with open(directory + filename) as json_file:
                        self.database[collection].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_device_configs_collection(self, collections_to_include, collection):
        """
        Insert all default device configuration documents into the specified database collection.
        :param collections_to_include: list of collections and documents to be initialized
        :param collection: the device configuration collection to initialize. (CNTL-CFG, OLT-CFG, ONU-CFG, or SWI-CFG)
        :return: None
        """
        directory = ""

        # Set directory
        if collection == "CNTL-CFG":
            directory = self.database_seed_base_dir + "documents/deviceConfigs/controllers/"
        if collection == "OLT-CFG":
            directory = self.database_seed_base_dir + "documents/deviceConfigs/olts/"
        if collection == "ONU-CFG":
            directory = self.database_seed_base_dir + "documents/deviceConfigs/onus/"
        if collection == "SWI-CFG":
            directory = self.database_seed_base_dir + "documents/deviceConfigs/switches/"

        if directory != "":
            for filename in os.listdir(directory):
                # Strip filename
                doc_id = filename[8:-5]
                if collection == "CNTL-CFG":
                    doc_id = doc_id[1:]
                # Insert document if to be included
                if collection in collections_to_include and "-CFG" in filename and filename.lower().endswith(".json"):
                    with open(directory + filename) as json_file:
                        self.database[collection].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_pon_auto_configs_collection(self, collections_to_include, collection):
        """
        Insert all default/global pon automation configuration documents into the specified database collection.
        :param collections_to_include: list of collections and documents to be initialized
        :param collection: the device configuration collection to initialize. (CNTL-AUTO-CFG, OLT-AUTO-CFG, ONU-AUTOCFG, SWI-AUTO-CFG, or AUTO-CFG)
        :return: None
        """
        directory = ""

        # Set directory
        if collection == "CNTL-AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/controllers/"
        if collection == "OLT-AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/olts/"
        if collection == "ONU-AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/onus/"
        if collection == "SWI-AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/switches/"
        if collection == "AUTO-CFG":
            directory = self.database_seed_base_dir + "documents/ponAutoConfigs/automation/"

        if directory != "":
            for filename in os.listdir(directory):
                # Strip filename
                if collection == "AUTO-CFG":
                    doc_id = filename[9:-5]
                else:
                    doc_id = filename[13:-5]
                    if collection == "CNTL-AUTO-CFG":
                        doc_id = doc_id[1:]
                # Insert document if to be included
                if collection in collections_to_include and "-CFG" in filename and filename.lower().endswith(".json"):
                    with open(directory + filename) as json_file:
                        self.database[collection].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_sla_configs_collection(self, documents_to_include):
        """
        Insert all default SLA configuration documents into the SLA-CFG collection.
        :param documents_to_include: list of which SLA documents to initialize
        :return: None
        """
        directory = self.database_seed_base_dir + "documents/slaConfigs/"
        for filename in os.listdir(directory):
            # Strip filename
            doc_id = filename[8:-5]
            # Insert document if to be included
            if doc_id in documents_to_include and "SLA-CFG" in filename and filename.lower().endswith(".json"):
                with open(directory + filename) as json_file:
                    self.database["SLA-CFG"].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)


    def initialize_ds_map_configs_collection(self, documents_to_include):
        """
        Insert all default DS-MAP-CFG configuration documents into the DS-MAP-CFG collection.
        :param documents_to_include: list of which DS-MAP-CFG documents to initialize
        :return: None
        """
        directory = self.database_seed_base_dir + "documents/dsMapConfigs/"
        for filename in os.listdir(directory):
            # Strip filename
            doc_id = filename[11:-5]
            # Insert document if to be included
            if doc_id in documents_to_include and "DS-MAP-CFG" in filename and filename.lower().endswith(".json"):
                with open(directory + filename) as json_file:
                    self.database["DS-MAP-CFG"].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_service_configs_collection(self, documents_to_include):
        """
        Insert all default Service configuration documents into the SRV-CFG collection.
        :param documents_to_include: list of which Service configuration documents to initialize
        :return: None
        """
        directory = self.database_seed_base_dir + "documents/srvConfigs/"
        for filename in os.listdir(directory):
            # Strip document
            doc_id = filename[4:-9]
            # Insert document if to be included
            if doc_id in documents_to_include and "SRV-" in filename and filename.lower().endswith(".json"):
                with open(directory + filename) as json_file:
                    self.database["SRV-CFG"].replace_one({"_id": doc_id}, json.load(json_file), upsert=True)

    def initialize_olt_firmware_bucket(self, files_to_include):
        """
        Upload all default OLT firmware files into the OLT-FIRMWARE bucket.
        :param files_to_include: list of which files to initialize
        :return: None
        """
        fs = gridfs.GridFSBucket(db=self.database, bucket_name="OLT-FIRMWARE")
        directory = self.database_seed_base_dir + "firmwares/olts/"
        for filename in os.listdir(directory):
            # Strip filename
            file_id = os.path.splitext(filename)[0]
            # Exclude unexpected files
            if filename in files_to_include.keys():
                # Delete old file if it exists
                if fs.find({"_id": file_id}).count() > 0:
                    fs.delete(file_id=file_id)
                if self.database['OLT-FIRMWARE.chunks'].find({'files_id': filename}).count() > 0:
                    self.database['OLT-FIRMWARE.chunks'].delete_many({'files_id': filename})

                # Upload new file
                up = fs.open_upload_stream_with_id(file_id=file_id, filename=filename,
                                                   metadata=files_to_include[filename])
                up.write(Binary(open(directory + filename, mode='rb').read()))
                up.close()

    def initialize_onu_firmware_bucket(self, files_to_include):
        """
        Upload all default ONU firmware files into the ONU-FIRMWARE bucket.
        :param files_to_include: list of which files to initialize
        :return: None
        """
        fs = gridfs.GridFSBucket(db=self.database, bucket_name="ONU-FIRMWARE")
        directory = self.database_seed_base_dir + "firmwares/onus/"
        for filename in os.listdir(directory):
            # Exclude unexpected files
            if filename in files_to_include.keys():
                # Delete old file if it exists
                if fs.find({"_id": filename}).count() > 0:
                    fs.delete(file_id=filename)
                if self.database['ONU-FIRMWARE.chunks'].find({'files_id': filename}).count() > 0:
                    self.database['ONU-FIRMWARE.chunks'].delete_many({'files_id': filename})

                # Upload new file
                up = fs.open_upload_stream_with_id(file_id=filename, filename=filename,
                                                   metadata=files_to_include[filename])
                up.write(Binary(open(directory + filename, mode='rb').read()))
                up.close()

    def initialize_picture_bucket(self, files_to_include):
        """
        Upload all default device pictures into the PICTURES bucket.
        Scales all pictures to a maximum height of 200 pixels.
        :param files_to_include: list of which files to initialize
        :return: None
        """
        fs = gridfs.GridFSBucket(db=self.database, bucket_name="PICTURES")
        directory = self.database_seed_base_dir + "pictures/"
        for device_directory in os.listdir(directory):
            if device_directory in ["controllers", "olts", "onus", "switches"]:
                for filename in os.listdir(directory + device_directory):
                    # Strip filename for file_id
                    file_id = os.path.splitext(filename)[0]
                    if filename in files_to_include.keys():
                        # Delete old picture if it exists
                        if fs.find({"_id": file_id}).count() > 0:
                            fs.delete(file_id=file_id)
                        if self.database['PICTURES.chunks'].find({'files_id': filename}).count() > 0:
                            self.database['PICTURES.chunks'].delete_many({'files_id': filename})

                        # Read file
                        image_bytes = io.BytesIO()
                        binary = Binary(open(directory + device_directory + "/" + filename, mode='rb').read())

                        with Image.open(io.BytesIO(binary)) as image:
                            # Shrink to height of 200 if greater than 200 pixels tall
                            if image.size[1] > 400:
                                file_data = resizeimage.resize_height(image, 400)
                                file_data.save(image_bytes, format='PNG')
                                image_bytes = image_bytes.getvalue()
                            else:
                                image_bytes = binary

                        # Upload new picture
                        up = fs.open_upload_stream_with_id(file_id=file_id, filename=filename,
                                                           metadata=files_to_include[filename])
                        up.write(image_bytes)
                        up.close()

    def get_seed_documents_info(self):
        """Returns the filenames and collections for all files in the documents directory"""
        data = []
        for collection_type in os.listdir(self.database_seed_base_dir + "documents/"):
            if collection_type in ["deviceConfigs", "alarmConfigs", "srvConfigs", "slaConfigs", 'dsMapConfigs', 'ponAutoConfigs']:
                # Determine collection to insert to and filename
                for device_type in os.listdir(self.database_seed_base_dir + "documents/" + collection_type):
                    collection = ""
                    if collection_type == "slaConfigs":
                        if device_type.lower().endswith('.json'):
                            filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type
                            if "SLA-CFG" in filename:
                                data.append((filename, "SLA-CFG"))
                    elif collection_type == "dsMapConfigs":
                        if device_type.lower().endswith('.json'):
                            filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type
                            if "DS-MAP-CFG" in filename:
                                data.append((filename, "DS-MAP-CFG"))
                    elif collection_type == "srvConfigs":
                        if device_type.lower().endswith('.json'):
                            filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type
                            if "SRV-" in filename:
                                data.append((filename, "SRV-CFG"))
                    elif collection_type == "alarmConfigs":
                        if device_type in ["olts", "onus", "controllers"]:
                            for file in os.listdir(
                                    self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/"):
                                filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/" + file
                                if "-CFG" in filename:
                                    if device_type == "controllers":
                                        collection = "CNTL-ALARM-CFG"
                                    elif device_type == "olts":
                                        collection = "OLT-ALARM-CFG"
                                    elif device_type == "onus":
                                        collection = "ONU-ALARM-CFG"
                                    data.append((filename, collection))
                    elif collection_type == "deviceConfigs":
                        if device_type in ["olts", "onus", "controllers", "switches"]:
                            for file in os.listdir(
                                    self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/"):
                                filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/" + file
                                if "-CFG" in filename:
                                    if device_type == "controllers":
                                        collection = "CNTL-CFG"
                                    elif device_type == "olts":
                                        collection = "OLT-CFG"
                                    elif device_type == "onus":
                                        collection = "ONU-CFG"
                                    elif device_type == "switches":
                                        collection = "SWI-CFG"
                                    data.append((filename, collection))
                    elif collection_type == "ponAutoConfigs":
                        if device_type in ["olts", "onus", "controllers", "switches", "automation"]:
                            for file in os.listdir(
                                    self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/"):
                                filename = self.database_seed_base_dir + "documents/" + collection_type + "/" + device_type + "/" + file
                                if "-CFG" in filename:
                                    if device_type == "controllers":
                                        collection = "CNTL-AUTO-CFG"
                                    elif device_type == "olts":
                                        collection = "OLT-AUTO-CFG"
                                    elif device_type == "onus":
                                        collection = "ONU-AUTO-CFG"
                                    elif device_type == "switches":
                                        collection = "SWI-AUTO-CFG"
                                    elif device_type == "automation":
                                        collection = "AUTO-CFG"
                                    data.append((filename, collection))
        return data

    """
    Stored databases
    """

    # Get the list of database information
    def get_databases(self, user_email):
        response = [status.HTTP_200_OK, "Retrieved list of databases."]

        try:
            try:
                with open(self.production_databases_files) as json_file:
                    response[1] = json.load(json_file)
            except FileNotFoundError:
                with open(self.development_databases_files) as json_file:
                    response[1] = json.load(json_file)

        except Exception as err:
            self.status_log("EXCEPTION (get_databases)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get databases due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "databases.json", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Add a database to the list of databases
    def add_database(self, db_json, user_email):
        response = [status.HTTP_201_CREATED, "Added database to server."]

        try:
            db_id = list(db_json.keys())[0]
            param_validity = "OK"
            if param_validity == "OK":
                try:
                    with open(self.production_databases_files, 'r') as json_file:
                        json_obj = json.load(json_file)
                        response[1] = json_obj
                        old_keys = list(json_obj.keys())
                except FileNotFoundError:
                    with open(self.development_databases_files, 'r') as json_file:
                        json_obj = json.load(json_file)
                        response[1] = json_obj
                        old_keys = list(json_obj.keys())
                if db_id not in old_keys:
                    try:
                        with open(self.production_databases_files, 'w') as json_file:
                            json_obj.update(db_json)
                            json.dump(json_obj, json_file, indent=4, sort_keys=True)
                    except FileNotFoundError:
                        with open(self.development_databases_files, 'w') as json_file:
                            json_obj.update(db_json)
                            json.dump(json_obj, json_file, indent=4, sort_keys=True)
                else:
                    response = [status.HTTP_409_CONFLICT,
                                f"Could not add database {db_id} because a database with this ID already exists.::Database ID must be unique."]
            else:
                try:
                    with open(self.production_databases_files, 'r') as json_file:
                        json_obj = json.load(json_file)
                except FileNotFoundError:
                    with open(self.development_databases_files, 'r') as json_file:
                        json_obj = json.load(json_file)
                response = [status.HTTP_400_BAD_REQUEST,
                            f"Could not add database {db_id} due to one or more database parameters being invalid.::{param_validity}"]
        except Exception as err:
            self.status_log("EXCEPTION (add_database)", sys.exc_info()[1])
            json_obj = "ERROR"
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not add database due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "databases.json", "old": response[1], "new": json_obj, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_201_CREATED)

        return response

    def edit_database(self, db_json, user_email):
        response = [status.HTTP_200_OK, "Database edited."]

        try:
            db_id = list(db_json.keys())[0]
            param_validity = "OK"
            if param_validity == "OK":
                try:
                    with open(self.production_databases_files, 'r') as json_file:
                        json_obj = json.load(json_file)
                        response[1] = json_obj
                        old_keys = list(json_obj.keys())
                except FileNotFoundError:
                    with open(self.development_databases_files, 'r') as json_file:
                        json_obj = json.load(json_file)
                        response[1] = json_obj
                        old_keys = list(json_obj.keys())
                if db_id not in old_keys:
                    response = [status.HTTP_409_CONFLICT,
                                f"Could not edit database {db_id} because a database with this ID does not exist.::Database entry not found."]
                else:
                    try:
                        with open(self.production_databases_files, 'w') as json_file:
                            json_obj[db_id] = db_json[db_id]
                            json.dump(json_obj, json_file, indent=4, sort_keys=True)
                    except FileNotFoundError:
                        with open(self.development_databases_files, 'w') as json_file:
                            json_obj[db_id] = db_json[db_id]
                            json.dump(json_obj, json_file, indent=4, sort_keys=True)
            else:
                try:
                    with open(self.production_databases_files, 'r') as json_file:
                        json_obj = json.load(json_file)
                except FileNotFoundError:
                    with open(self.development_databases_files, 'r') as json_file:
                        json_obj = json.load(json_file)
                response = [status.HTTP_400_BAD_REQUEST,
                            f"Could not edit database {db_id} due to one or more database parameters being invalid.::{param_validity}"]
        except Exception as err:
            self.status_log("EXCEPTION (edit_database)", sys.exc_info()[1])
            json_obj = "ERROR"
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not edit database due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "databases.json", "old": response[1], "new": json_obj, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    # Delete a database from the list of databases
    def delete_database(self, db_id, user_email):
        response = [status.HTTP_200_OK, "Deleted database."]

        try:

            try:
                with open(self.production_databases_files, 'r') as json_file:
                    json_obj = json.load(json_file)
                    response[1] = json_obj
            except FileNotFoundError:
                with open(self.development_databases_files, 'r') as json_file:
                    json_obj = json.load(json_file)
                    response[1] = json_obj
            try:
                with open(self.production_databases_files, 'w') as json_file:
                    if db_id == "Default":
                        response = [status.HTTP_400_BAD_REQUEST,
                                    "The Default database entry may not be deleted.::Cannot delete the Default database."]
                    elif db_id in json_obj and len(json_obj) > 1:
                        del json_obj[db_id]
                    else:
                        response = [status.HTTP_400_BAD_REQUEST,
                                    "Cannot delete the final active database reference.::Cannot delete the final active database reference."]
                    json.dump(json_obj, json_file, indent=4, sort_keys=True)
            except FileNotFoundError:
                with open(self.development_databases_files, 'w') as json_file:
                    if db_id == "Default":
                        response = [status.HTTP_400_BAD_REQUEST,
                                    "The Default database entry may not be deleted.::Cannot delete the Default database."]
                    elif db_id in json_obj and len(json_obj) > 1:
                        del json_obj[db_id]
                    else:
                        response = [status.HTTP_400_BAD_REQUEST,
                                    "Cannot delete the final active database reference.::Cannot delete the final active database reference."]
                    json.dump(json_obj, json_file, indent=4, sort_keys=True)

        except Exception as err:
            self.status_log("EXCEPTION (delete_database)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete database due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "databases.json", "old": response[1], "new": json_obj, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    """
    Database Seeding
    """

    def get_seed_files(self, user_email: str):
        """Gives the names and metadata of all files to seed a database with

        Args:
            user_email: The email of the user requesting the information

        Returns:
            Response object containing the filenames and metadata for all files to seed a database with
        """

        response = [status.HTTP_200_OK,
                    {
                        "pictures": [], "oltFirmwares": [], "onuFirmwares": [],
                        "CFG": [], "ALARM-CFG": [], "SLA-CFG": [], "DS-MAP-CFG": [],
                        "SRV-CFG": [], "AUTO-CFG": []
                    }]

        try:
            with open(self.database_seed_base_dir + "metadata.json", "r") as json_file:
                metadata_file = json.load(json_file)
            for directory in os.listdir(self.database_seed_base_dir):
                if directory == "firmwares" or directory == "pictures":
                    current_dir = self.database_seed_base_dir + directory
                    for device_directory in os.listdir(current_dir):
                        current_dir = self.database_seed_base_dir + directory + "/" + device_directory
                        if os.path.isdir(current_dir):
                            for filename in os.listdir(current_dir):
                                file_to_be = directory + "/" + device_directory + "/" + filename
                                if file_to_be in metadata_file:
                                    value = {
                                        "filename": filename,
                                        "metadata": metadata_file[file_to_be]
                                    }
                                    if directory == "firmwares":
                                        if device_directory == "olts":
                                            response[1]["oltFirmwares"].append(value)
                                        else:
                                            response[1]["onuFirmwares"].append(value)
                                    else:
                                        response[1]["pictures"].append(value)
            for (filename, collection) in self.get_seed_documents_info():
                doc_id = str(filename)[str(filename).rfind("/") + 1:]

                if collection == "SLA-CFG":
                    response[1][collection].append({"collection": collection, "docId": doc_id[8:-5]})
                if collection == "DS-MAP-CFG":
                    response[1][collection].append({"collection": collection, "docId": doc_id[11:-5]})
                elif collection == "SRV-CFG":
                    response[1][collection].append({"collection": collection, "docId": doc_id[4:-9]})
                elif "ALARM" in collection:
                    response[1]["ALARM-CFG"].append(
                        {"collection": collection, "docId": doc_id[doc_id.rfind("-") + 1:-5]})
                elif collection == "SWI-CFG" or collection == "CNTL-CFG" or collection == "OLT-CFG" or collection == "ONU-CFG":
                    response[1]["CFG"].append({"collection": collection, "docId": doc_id[doc_id.rfind("-") + 1:-5]})
                elif collection == "SWI-AUTO-CFG" or collection == "CNTL-AUTO-CFG"\
                        or collection == "OLT-AUTO-CFG" or collection == "ONU-AUTO-CFG" or collection == "AUTO-CFG":
                    response[1]["AUTO-CFG"].append({"collection": collection, "docId": doc_id[doc_id.rfind("-") + 1:-5]})
        except Exception as err:
            self.status_log("EXCEPTION (get_seed_files)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get seed files due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "databaseSeedFiles/", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Delete devices
    """

    # Remove all data for the specified ONU from database
    def delete_onu_all(self, onu_id, user_email, delete_from_olt_inv):
        response = [status.HTTP_200_OK, f"All ONU {onu_id} data was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["ONU-CFG"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["STATS-ONU-{}".format(onu_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SYSLOG-ONU-{}".format(onu_id.replace(":", ""))]
            collection.drop()
            collection = self.database["ONU-MIB-CUR-STATE"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["ONU-MIB-RST-STATE"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["CPE-STATE"]
            collection.delete_many({"ONU.ID": onu_id})
            collection = self.database["ONU-AUTO-CFG"]
            collection.delete_one({"_id": onu_id})
            collection = self.database["ONU-AUTO-STATE"]
            collection.delete_one({"_id": onu_id})

            # Removes from parent OLT-CFG
            if (delete_from_olt_inv == "true"):
                collection = self.database["OLT-CFG"]
                querystring = "ONUs." + onu_id
                collection.update_many({querystring: {'$exists': 1}},
                                       {"$unset": {querystring: ""}, "$inc": {"OLT.CFG Change Count": 1}})

        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_all)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all ONU {onu_id} data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {
            "collection": f"ONU-STATE, CPE-STATE, ONU-CFG, STATS-ONU-{onu_id.replace(':', '')}, SYSLOG-ONU-{onu_id.replace(':', '')}, ONU-MIB-CUR-STATE, ONU-MIB-RST-STATE",
            "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified ONU from database
    def delete_onu_logs(self, onu_id, user_email):
        response = [status.HTTP_200_OK, "All ONU {} logs were deleted.".format(onu_id)]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SYSLOG-ONU-{}".format(onu_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_logs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete ONU {onu_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-ONU-{}".format(onu_id.replace(":", "")),
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified ONU from database
    def delete_onu_stats(self, onu_id, user_email):
        response = [status.HTTP_200_OK, "All ONU {} statistics were deleted.".format(onu_id)]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["STATS-ONU-{}".format(onu_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete ONU {onu_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU-{}".format(onu_id.replace(":", "")),
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified OLT from database
    def delete_olt_all(self, olt_id, user_email):
        response = [status.HTTP_200_OK, f"All OLT {olt_id} data was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            collection.delete_one({"_id": olt_id})
            collection = self.database["OLT-CFG"]
            collection.delete_one({"_id": olt_id})
            collection = self.database["STATS-OLT-{}".format(olt_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SYSLOG-OLT-{}".format(olt_id.replace(":", ""))]
            collection.drop()
            collection = self.database["OLT-AUTO-CFG"]
            collection.delete_one({"_id": olt_id})
            collection = self.database["OLT-AUTO-STATE"]
            collection.delete_one({"_id": olt_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_all)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all OLT {olt_id} data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {
            "collection": f"OLT-STATE, OLT-CFG, STATS-OLT-{olt_id.replace(':', '')}, SYSLOG-OLT-{olt_id.replace(':', '')}",
            "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all logs for the specified OLT from database
    def delete_olt_logs(self, olt_id, user_email):
        response = [status.HTTP_200_OK, f"All OLT {olt_id} logs were deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SYSLOG-OLT-{}".format(olt_id.replace(":", "").replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_logs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete OLT {olt_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"SYSLOG-OLT-{olt_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all stats for the specified OLT from database
    def delete_olt_stats(self, olt_id, user_email):
        response = [status.HTTP_200_OK, f"All OLT {olt_id} statistics were deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["STATS-OLT-{}".format(olt_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete OLT {olt_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"STATS-OLT-{olt_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified CNTL from database
    def delete_cntl(self, cntl_id, user_email):
        response = [status.HTTP_200_OK, f"All PON Controller {cntl_id} data was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            collection.delete_one({"_id": cntl_id})
            collection = self.database["CNTL-CFG"]
            collection.delete_one({"_id": cntl_id})
            collection = self.database["STATS-CNTL-{}".format(cntl_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SYSLOG-CNTL-{}".format(cntl_id.replace(":", ""))]
            collection.drop()
            collection = self.database["CNTL-AUTO-CFG"]
            collection.delete_one({"_id": cntl_id})
            collection = self.database["CNTL-AUTO-STATE"]
            collection.delete_one({"_id": cntl_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_cntl_all)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all PON Controller {cntl_id} data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"CNTL-STATE, CNTL-CFG, SYSLOG-CNTL-{cntl_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified CNTL from database
    def delete_cntl_logs(self, cntl_id, user_email):
        response = [status.HTTP_200_OK, f"All PON Controller {cntl_id} logs were deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SYSLOG-CNTL-{}".format(cntl_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_cntl_logs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete PON Controller {cntl_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"SYSLOG-CNTL-{cntl_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all stats for the specified PON Controller from database
    def delete_cntl_stats(self, cntl_id, user_email):
        response = [status.HTTP_200_OK, f"All PON Controller {cntl_id} statistics were deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["STATS-CNTL-{}".format(cntl_id.replace(":", ""))]
            collection.drop()
        except Exception as err:
            self.status_log("EXCEPTION (delete_cntl_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete PON Controller {cntl_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"STATS-CNTL-{cntl_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Remove all data for the specified Switch from database
    def delete_switch_all(self, switch_id, user_email):
        response = [status.HTTP_200_OK, f"All Switch {switch_id} data was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            collection.delete_one({"_id": switch_id})
            collection = self.database["STATS-SWI-{}".format(switch_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SYSLOG-SWI-{}".format(switch_id.replace(":", ""))]
            collection.drop()
            collection = self.database["SWI-AUTO-CFG"]
            collection.delete_one({"_id": switch_id})
            collection = self.database["SWI-AUTO-STATE"]
            collection.delete_one({"_id": switch_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_switch_all)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete all Switch {switch_id} data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"SWI-CFG, SYSLOG-SWI-{switch_id.replace(':', '')}",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all image names
    """

    def get_image_names(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["PICTURES.files"]
            names = collection.find({}, {"metadata": 1})
            for name in names:
                response[1].append(name)
        except Exception as err:
            self.status_log("EXCEPTION (get_image_names)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get image names due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all image names that are in use by a device
    """

    def get_device_image_names(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            names = collection.distinct("SWI.Picture")
            for name in names:
                response[1].append(name)

            collection = self.database["CNTL-CFG"]
            names = collection.distinct("CNTL.Picture")
            for name in names:
                response[1].append(name)

            collection = self.database["OLT-CFG"]
            names = collection.distinct("OLT.Picture")
            for name in names:
                response[1].append(name)

            collection = self.database["ONU-CFG"]
            names = collection.distinct("ONU.Picture")
            for name in names:
                response[1].append(name)
        except Exception as err:
            self.status_log("EXCEPTION (get_device_image_names)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get image names due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get first 10 devices that are using the given image name
    """

    def get_devices_from_image(self, user_email, image_name, device_type):
        response = [status.HTTP_200_OK, []]
        try:
            database_manager.get_database(self.database_id)
            if device_type == 'OLT':
                collection = self.database["OLT-CFG"]
                names = collection.find({"OLT.Picture": image_name}, {'_id': 1, 'OLT.Name': 1}).limit(10)
                for name in names:
                    response[1].append({'_id': name['_id'], 'name': name['OLT']['Name']})
            if device_type == 'ONU':
                collection = self.database["ONU-CFG"]
                names = collection.find({"ONU.Picture": image_name}, {'_id': 1, 'ONU.Name': 1}).limit(10)
                for name in names:
                    response[1].append({'_id': name['_id'], 'name': name['ONU']['Name']})
            if device_type == 'CNTL':
                collection = self.database["CNTL-CFG"]
                names = collection.find({"CNTL.Picture": image_name}, {'_id': 1, 'CNTL.Name': 1}).limit(10)
                for name in names:
                    response[1].append({'_id': name['_id'], 'name': name['CNTL']['Name']})
            if device_type == 'SWI':
                collection = self.database["SWI-CFG"]
                names = collection.find({"SWI.Picture": image_name}, {'_id': 1, 'SWI.Name': 1}).limit(10)
                for name in names:
                    response[1].append({'_id': name['_id'], 'name': name['SWI']['Name']})
        except Exception as err:
            self.status_log("EXCEPTION (get_devices_from_image)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get device names due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Delete CPE based on ID
    """

    def delete_cpe(self, cpe_id, user_email):
        response = [status.HTTP_200_OK, f"Deleted CPE {cpe_id}."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CPE-STATE"]
            query = {"_id": cpe_id}
            collection.delete_one(query)
        except Exception as err:
            self.status_log("EXCEPTION (delete_cpe)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete CPE {cpe_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CPE-STATE",
                       "deleted": response[0] == status.HTTP_200_OK,
                       "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Delete ALL CPEs based on ONU
    """

    def delete_all_cpes_for_onu(self, onu_id, user_email):
        response = [status.HTTP_200_OK, f"Deleted all CPEs for ONU {onu_id}"]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CPE-STATE"]
            collection.delete_many({"ONU.ID": onu_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_all_cpes_for_onu)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete CPEs for ONU {onu_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CPE-STATE",
                       "deleted": response[0] == status.HTTP_200_OK,
                       "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Delete ALL *EXPIRED* CPEs based on ONU
    """

    def delete_all_expired_cpes_for_onu(self, onu_id, user_email):
        response = [status.HTTP_200_OK, f"Deleted all expired CPEs for ONU {onu_id}"]
        expired_cpes_to_be_deleted = []

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CPE-STATE"]
            cpes = collection.find({"ONU.ID": onu_id}, {"DHCP": 1, "DHCPV6": 1})
            for cpe in cpes:
                # If the DHCPv6 is not present, we only need to check if v4 is expired
                if "DHCP" in cpe and "DHCPV6" not in cpe and "State" in cpe["DHCP"]:
                    if cpe['DHCP']['State'] == 'EXPIRED':
                        expired_cpes_to_be_deleted.append(cpe["_id"])
                # If the DHCPv4 is not present, we only need to check if v6 is expired
                elif "DHCP" not in cpe and "DHCPV6" in cpe and "State" in cpe["DHCPV6"]:
                    if cpe['DHCPV6']['State'] == 'EXPIRED':
                        expired_cpes_to_be_deleted.append(cpe["_id"])
                # If DHCPv6 is present, we need to make sure they are both expired
                elif "DHCP" in cpe and "DHCPV6" in cpe and "State" in cpe["DHCP"] and "State" in cpe["DHCPV6"]:
                    if cpe['DHCP']['State'] == 'EXPIRED' and cpe['DHCPV6']['State'] == 'EXPIRED':
                        expired_cpes_to_be_deleted.append(cpe["_id"])
            collection.delete_many({"_id": {"$in": expired_cpes_to_be_deleted}})
        except Exception as err:
            self.status_log("EXCEPTION (delete_all_expired_cpes_for_onu)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete expired CPEs for ONU {onu_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CPE-STATE",
                       "deleted": response[0] == status.HTTP_200_OK,
                       "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Get ALL CPEs based on ONU
    """

    def get_all_cpes_for_onu(self, onu_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CPE-STATE"]
            cpe_cursor = collection.find({"ONU.ID": onu_id}).sort("_id", 1)
            for cpe in cpe_cursor:
                response[1].append(cpe)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cpes_for_onu)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get CPEs for ONU {onu_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CPE-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get FW Filenames
    """

    def get_olt_fw_filenames(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-FIRMWARE.files"]
            names = collection.find({}, {"metadata": 1, "filename": 1})
            for name in names:
                response[1].append(name)
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_fw_filenames)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT firmware filenames due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-FIRMWARE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_fw_filenames(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-FIRMWARE.files"]
            names = collection.find({}, {"metadata": 1, "filename": 1})
            for name in names:
                response[1].append(name)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_fw_filenames)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU firmware filenames due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get a device name
    """

    def get_onu_name(self, onu_id, user_email):
        response = [status.HTTP_200_OK, f"Retrieved ONU {onu_id} name."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            response[1] = collection.find_one({"_id": onu_id}, {"_id": 1, "ONU.Name": 1})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_name)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} name due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olt_name(self, olt_id, user_email):
        response = [status.HTTP_200_OK, f"Retrieved OLT {olt_id} name."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-CFG"]
            response[1] = collection.find_one({"_id": olt_id}, {"_id": 1, "OLT.Name": 1})
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_name)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} name due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_cntl_name(self, cntl_id, user_email):
        response = [status.HTTP_200_OK, f"Retrieved PON Controller {cntl_id} name."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            response[1] = collection.find_one({"_id": cntl_id}, {"_id": 1, "CNTL.Name": 1})
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_name)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {cntl_id} name due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    User methods
    """

    def create_user(self, user_data):
        response = [status.HTTP_201_CREATED, "Created new user."]

        try:
            # Set users email as document id
            collection = self.database["USERS"]
            result = collection.insert_one(user_data)
            if result.inserted_id is None:
                response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                            "Failed to create new user due to Inserted ID is None.::Inserted ID is None."]
        except Exception as err:
            self.status_log("EXCEPTION (create_user)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Failed to create new user due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    def authenticate_user(self, user_data):
        response = [status.HTTP_200_OK, user_data]

        try:
            collection = self.database["USERS"]
            if collection.find({"_id": user_data["_id"], "password": user_data.password}).count() == 0:
                response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                            "Failed to authenticate user due to user not found.::User not found."]
        except Exception as err:
            self.status_log("EXCEPTION (authenticate_user)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Failed to authenticate user due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    def user_exists(self, email_address):
        response = [status.HTTP_200_OK, "User exists."]

        try:
            collection = self.database["USERS"]
            if collection.find({"_id": email_address}).count() == 0:
                response[1] = "User does not exists."
        except Exception as err:
            self.status_log("EXCEPTION (user_exists)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Failed to check if user exists user due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    def get_minimum_password_requirements(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            collection = self.user_database["tibit_settings"]
            expiration = collection.find_one({"_id": "Default"}, {"Password Requirements": 1, "_id": 0})

            if expiration is not None:
                response[1] = expiration['Password Requirements']
        except Exception as err:
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            self.status_log("EXCEPTION (get_minimum_password_requirements)", sys.exc_info()[1])

        action_data = {"collection": "tibit_settings", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def set_minimum_password_requirements(self, password_requirements_object, user_email):
        response = [status.HTTP_200_OK, 'OK', None, {}]

        try:
            collection = self.user_database["tibit_settings"]
            update_document = {"$set": {"Password Requirements": password_requirements_object}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            self.status_log("EXCEPTION (set_minimum_password_requirements)", sys.exc_info()[1])

        action_data = {"collection": "tibit_settings", "old": None, "new": password_requirements_object,
                       "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)
        return response

    """
    Unattached devices for tree
    """

    def get_all_unattached_controllers(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            cfg_collections = self.database["CNTL-CFG"]
            cntl_cfg_ids = cfg_collections.distinct("_id", {})
            state_collections = self.database["CNTL-STATE"]

            index = 0
            while index < len(cntl_cfg_ids):
                cntl_state = state_collections.find_one({"_id": cntl_cfg_ids[index]},
                                                        {"_id": 1, "Time": 1, "CNTL.Version": 1})
                if cntl_state is None:
                    cntl_cfg = cfg_collections.find_one({"_id": cntl_cfg_ids[index]}, {"_id": 1, "CNTL.Name": 1})
                    response[1].append({'_id': cntl_cfg_ids[index], 'name': cntl_cfg['CNTL']['Name']})

                index += 1

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_controllers)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all unattached Switches
    def get_all_unattached_switches(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            cfg_collection = self.database["SWI-CFG"]
            swi_cfg_ids = cfg_collection.distinct("_id", {})
            collection = self.database["OLT-STATE"]
            index = 0
            while index < len(swi_cfg_ids):
                cfg_state = collection.find_one({"Switch.Chassis ID": swi_cfg_ids[index]}, {"_id": 1})
                if cfg_state is None:
                    swi_cfg = cfg_collection.find_one({"_id": swi_cfg_ids[index]})
                    response[1].append({'_id': swi_cfg_ids[index], 'name': swi_cfg['SWI']['Name']})
                index += 1

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_switches)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all unattached OLTs
    # Pre-Provisioned: INV:!primary, CFG, !STATE, offline
    # Uninventoried w/ no state: !INV, CFG, !STATE, offline
    def get_all_unattached_olts(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            all_olts = list(self.database["OLT-CFG"].aggregate([
                {
                    "$group": {"_id": "$_id", "Name": {"$first": "$OLT.Name"}}
                    # $first has to be there for some reason to get the name
                }
            ]))

            for olt in all_olts:
                id = olt["_id"]
                if id != "Default":
                    x = self.database["CNTL-STATE"].find({'$or': [
                        {"OLTs.Primary": id},
                        {"OLTs.Primary Missing": id},
                        {"OLTs.Primary Free": id},
                        {"OLTs.Secondary": id},
                        {"OLTs.Secondary Missing": id},
                        {"OLTs.Secondary Free": id},
                        {"OLTs.Unspecified": id},
                        {"OLTs.Unspecified Free": id},
                    ]}, {"_id": 1}).count()
                    unattached_cntl = x == 0

                    y = self.database["SWI-CFG"].find({f"OLTs.{id}": {"$exists": True}}, {"_id": 1}).count()
                    if y == 0:
                        olt_state = self.database["OLT-STATE"].find_one({"_id": id}, {"Switch.Chassis ID": 1})
                        if olt_state and "Switch" in olt_state and olt_state["Switch"]["Chassis ID"] != "":
                            chassis = olt_state["Switch"]["Chassis ID"]
                            s = self.database["SWI-CFG"].find({"_id": chassis}, {"_id": 1}).count()
                            unattached_switch = s == 0
                        else:
                            unattached_switch = True
                    else:
                        unattached_switch = False

                    if unattached_cntl or unattached_switch:
                        response[1].append({'_id': id, 'name': olt['Name']})
                else:
                    response[1].append({'_id': id, 'name': olt['Name']})
            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_olts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached OLTs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG, CNTL-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all unattached ONUs
    def get_all_unattached_onus(self, user_email):
        response = [status.HTTP_200_OK, []]
        try:
            database_manager.get_database(self.database_id)
            all_onus = list(self.database["ONU-CFG"].aggregate([
                {
                    "$group": {"_id": "$_id", "Name": {"$first": "$ONU.Name"}}
                    # $first has to be there for some reason to get the name
                }
            ]))

            configured_controllers = list(self.database["CNTL-STATE"].aggregate([
                {"$addFields": {"result": {"$objectToArray": "$System Status"}}},
                {"$unwind": "$result"},
                {"$project": {"result.v.ONUs": 1}},
                {"$addFields": {"onus": {"$objectToArray": "$result.v.ONUs"}}},
                {"$project": {"onus.k": 1}},
            ]))

            for onu in all_onus:
                id = onu["_id"]
                unattached = True
                configured_olts = self.database["OLT-CFG"].find({f"ONUs.{id}": {"$exists": True}}, {"_id": 1}).count()
                if configured_olts == 0:
                    for system in configured_controllers:
                        for system_onu in system["onus"]:
                            if 'k' in system_onu and system_onu['k'] == id:
                                unattached = False
                else:
                    unattached = False

                if unattached:
                    response[1].append({'_id': id, 'Name': onu['Name']})
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unattached_onus)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unattached ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG, ONU-CFG, CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all onus using a specified sla
    def get_all_sla_onu_usage(self, sla, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            onu_state_collection = self.database["ONU-STATE"]

            for i in range(8):
                service = "OLT-Service " + str(i) + ".SLA-CFG.Source"
                onu_cursor = onu_state_collection.find({service: sla},
                                                       {"_id": 1, "OLT.MAC Address": 1, "CNTL.MAC Address": 1})
                for onu in onu_cursor:

                    # Getting switch mac from olt state
                    olt_state_collection = self.database["OLT-STATE"]
                    olt = olt_state_collection.find_one({"_id": onu["OLT"]["MAC Address"]}, {"Switch.Chassis ID": 1})
                    onu["Switch"] = {}
                    onu["Switch"]["Chassis ID"] = olt["Switch"]["Chassis ID"]

                    if onu["_id"] not in response[1]:
                        response[1].append(onu)

        except Exception as err:
            self.status_log("EXCEPTION (get_all_sla_onu_usage)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs using SLA {sla} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all onus using a specified service config
    def get_all_srv_onu_usage(self, srv, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            onu_state_collection = self.database["ONU-STATE"]

            onu_cursor = onu_state_collection.find({"ONU.SRV-CFG": srv},
                                                   {"_id": 1, "OLT.MAC Address": 1, "CNTL.MAC Address": 1})
            for onu in onu_cursor:
                # Getting switch mac from olt state
                olt_state_collection = self.database["OLT-STATE"]
                olt = olt_state_collection.find_one({"_id": onu["OLT"]["MAC Address"]}, {"Switch.Chassis ID": 1})
                onu["Switch"] = {}
                onu["Switch"]["Chassis ID"] = olt["Switch"]["Chassis ID"]

                response[1].append(onu)

        except Exception as err:
            self.status_log("EXCEPTION (get_all_srv_onu_usage)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs using SLA {srv} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active CNTL MAC Addresses
    def get_all_controllers(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection CNTL_STATE
            collection = self.database["CNTL-STATE"]
            cntl_ids = collection.distinct("_id", {})
            collection = self.database["CNTL-CFG"]
            index = 0
            while index < len(cntl_ids):
                response[1].append({"_id": "", "name": ""})
                response[1][index]["_id"] = cntl_ids[index]
                name = collection.find_one({"_id": cntl_ids[index]}, {"CNTL.Name": 1})
                if name is None:
                    response[1][index]["name"] = ""
                else:
                    response[1][index]["name"] = name["CNTL"]["Name"]
                index += 1

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntls)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active CNTLs with their versions
    def get_all_controllers_with_versions(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection CNTL_STATE
            collection = self.database["CNTL-CFG"]
            cntl_ids = collection.distinct("_id", {})
            index = 0
            while index < len(cntl_ids):
                response[1].append({"_id": "", "name": "", "version": "", "type": "Controller"})
                response[1][index]["_id"] = cntl_ids[index]
                version = collection.find_one({"_id": cntl_ids[index]}, {"CNTL.CFG Version": 1})
                response[1][index]["version"] = version["CNTL"]["CFG Version"].split("-")[0]
                name = collection.find_one({"_id": cntl_ids[index]}, {"CNTL.Name": 1})
                if name is None:
                    response[1][index]["name"] = ""
                else:
                    response[1][index]["name"] = name["CNTL"]["Name"]
                index += 1
        except Exception as err:
            self.status_log("EXCEPTION (get_all_controllers_with_versions)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active OLT MAC Addresses
    def get_all_olts(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            olt_ids = collection.find({}, {"OLT.PON Mode": 1})
            collection = self.database["OLT-CFG"]
            response[1] = []
            for olt in olt_ids:
                olt_data = {"_id": "", "Name": "", "PON Mode": ""}
                olt_id = olt["_id"]
                olt_data["_id"] = olt_id
                olt_data["PON Mode"] = olt["OLT"]["PON Mode"]
                document = collection.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if document is None:
                    olt_data["Name"] = ""
                else:
                    olt_data["Name"] = document["OLT"]["Name"]
                response[1].append(olt_data)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of OLTs with their versions
    def get_all_olts_with_versions(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-CFG"]
            olt_ids = collection.distinct("_id", {})
            index = 0
            while index < len(olt_ids):
                response[1].append({"_id": "", "name": "", "version": ""})
                response[1][index]["_id"] = olt_ids[index]
                name = collection.find_one({"_id": olt_ids[index]}, {"OLT.Name": 1})
                if name is None:
                    response[1][index]["name"] = ""
                else:
                    response[1][index]["name"] = name["OLT"]["Name"]
                version = collection.find_one({"_id": olt_ids[index]}, {"CNTL.CFG Version": 1})
                response[1][index]["version"] = version["CNTL"]["CFG Version"].split("-")[0]
                index += 1
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_with_versions)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all OLTs under the specified Switch
    def get_all_olts_for_switch(self, switch_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            olts = collection.find({"Switch.Chassis ID": switch_id},
                                   {"_id": 1, "Switch.Port ID": 1, "CNTL.MAC Address": 1, "Switch.Chassis ID": 1})
            for olt in olts:
                olt_id = olt["_id"]
                port = olt["Switch"]["Port ID"]
                switch = olt["Switch"]["Chassis ID"]
                controller = olt["CNTL"]["MAC Address"]
                online = False

                # Get OLTs Name if it exists
                olt_cfg_coll = self.database["OLT-CFG"]
                name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                    name = name_obj["OLT"]["Name"]
                else:
                    name = ""

                # Check if OLT is online
                cntl_state_coll = self.database["CNTL-STATE"]
                system_status_obj = cntl_state_coll.find_one({"_id": controller}, {"System Status": 1})
                if system_status_obj is not None and "System Status" in system_status_obj and olt_id in \
                        system_status_obj["System Status"].keys():
                    online = True
                response[1].append(
                    {"_id": olt_id, "port": port, "cntl": controller, "name": name, "switch": switch, "online": online})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['port'], i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_for_switch)", err)
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs for Switch {switch_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0])

        return response

    # Retrieve an array of all OLTs in the inventory of the specified switch
    def get_all_olts_in_switch_inventory(self, switch_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all olts inventoried on switch
            inventoried_olt_macs = []
            collection = self.database["SWI-CFG"]
            olt_mac_list = collection.find({"_id": switch_id}, {"OLTs": 1})
            if olt_mac_list.count() > 0 and 'OLTs' in olt_mac_list[0]:
                for olt in olt_mac_list[0]['OLTs'].keys():
                    inventoried_olt_macs.append(olt)

            # Getting state files for all inventoried OLTS
            state_olt_macs = []
            collection = self.database["OLT-STATE"]
            olts = collection.find({"_id": {"$in": inventoried_olt_macs}},
                                   {"_id": 1, "Switch.Port ID": 1, "CNTL.MAC Address": 1, "Switch.Chassis ID": 1})
            for olt in olts:
                port = ""
                switch = ""
                if "Switch" in olt:
                    if "Port ID" in olt["Switch"]:
                        port = olt["Switch"]["Port ID"]

                    if "Chassis ID" in olt["Switch"]:
                        switch = olt["Switch"]["Chassis ID"]

                state_olt_macs.append(olt["_id"])
                olt_id = olt["_id"]
                controller = olt["CNTL"]["MAC Address"]
                online = False

                # Get OLTs Name if it exists
                olt_cfg_coll = self.database["OLT-CFG"]
                name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                    name = name_obj["OLT"]["Name"]
                else:
                    name = ""

                # Check if OLT is online
                cntl_state_coll = self.database["CNTL-STATE"]
                system_status_obj = cntl_state_coll.find_one({"_id": controller}, {"System Status": 1})
                if system_status_obj is not None and "System Status" in system_status_obj and olt_id in \
                        system_status_obj["System Status"].keys():
                    online = True

                response[1].append(
                    {"_id": olt_id, "port": port, "cntl": controller, "name": name, "switch": switch, "online": online})

            # Appending inventoried OLTS that do not have a state file
            for olt in inventoried_olt_macs:
                if olt not in state_olt_macs:
                    olt_id = olt
                    port = ""
                    switch = ""
                    controller = ""
                    online = False

                    # Get OLTs Name if it exists
                    olt_cfg_coll = self.database["OLT-CFG"]
                    name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                    if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                        name = name_obj["OLT"]["Name"]
                    else:
                        name = ""
                    response[1].append({"_id": olt_id, "port": port, "cntl": controller, "name": name, "switch": switch,
                                        "online": online})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['port'], i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_for_switch)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs in Switch {switch_id} inventory due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_all_olts_for_controller(self, controller_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            olt_state_coll = self.database["OLT-STATE"]
            if controller_id == "Default":
                olts = olt_state_coll.find({"CNTL.MAC Address": ""}, {"_id": 1})
            else:
                olts = olt_state_coll.find({"CNTL.MAC Address": controller_id},
                                           {"_id": 1, "Switch.Port ID": 1, "Switch.Chassis ID": 1})
            for olt in olts:
                olt_id = olt["_id"]
                port = ''
                switch = ''
                if "Switch" in olt and "Port ID" in olt["Switch"]:
                    port = olt["Switch"]["Port ID"]
                if "Switch" in olt and "Chassis ID" in olt["Switch"]:
                    switch = olt["Switch"]["Chassis ID"]
                # Get OLTs Name if it exists
                olt_cfg_coll = self.database["OLT-CFG"]
                name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                    name = name_obj["OLT"]["Name"]
                else:
                    name = ""
                response[1].append(
                    {"_id": olt_id, "port": port, "switchID": switch, "cntl": controller_id, "name": name})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['port'], i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_for_controller)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs under PON Controller {controller_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all OLTs under the unknown switch for a given controller
    def get_all_olts_for_unknown_switch(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            olts = collection.find({"Switch.Chassis ID": ""}, {"_id": 1})
            for olt in olts:
                olt_id = olt["_id"]
                controller = olt["CNTL"]["MAC Address"]
                # Get OLTs Name if it exists
                olt_cfg_coll = self.database["OLT-CFG"]
                name_obj = olt_cfg_coll.find_one({"_id": olt_id}, {"OLT.Name": 1})
                if name_obj is not None and name_obj["OLT"] is not None and name_obj["OLT"]["Name"] is not None:
                    name = name_obj["OLT"]["Name"]
                else:
                    name = ""
                response[1].append({"_id": olt_id, "cntl": controller, "name": name})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olts_for_unknown_switch)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs in unknown Switches due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves a list of all Switch MAC addresses and Names
    def get_all_switches(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_macs = collection.distinct("Switch.Chassis ID", {})

            collection = self.database["SWI-CFG"]
            for switch in switch_macs:
                data = {"_id": switch, "name": ""}
                name = collection.find_one({"_id": switch}, {"SWI.Name": 1})
                if name is None:
                    data["name"] = ""
                else:
                    data["name"] = name["SWI"]["Name"]
                response[1].append(data)

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switches)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_all_switches_with_versions(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            switch_macs = collection.distinct("_id", {})

            for switch in switch_macs:
                data = {"_id": switch, "name": "", "version": "", "type": "Switch"}
                version = collection.find_one({"_id": switch}, {"CNTL.CFG Version": 1})
                data["version"] = version["CNTL"]["CFG Version"].split("-")[0]
                name = collection.find_one({"_id": switch}, {"SWI.Name": 1})
                if name is None:
                    data["name"] = ""
                else:
                    data["name"] = name["SWI"]["Name"]
                response[1].append(data)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switches_with_versions)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves a list of all Switch MAC addresses and Names for the specified Controller
    def get_all_switches_for_cntl(self, cntl_mac_address, user_email):
        response = [status.HTTP_200_OK, []]

        switch_macs = []

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_list = collection.find({"CNTL.MAC Address": cntl_mac_address}, {"Switch.Chassis ID": 1})
            for switch in switch_list:
                if switch["Switch"]["Chassis ID"] != "" and switch["Switch"]["Chassis ID"] not in switch_macs:
                    switch_macs.append(switch["Switch"]["Chassis ID"])

            collection = self.database["SWI-CFG"]
            for switch in switch_macs:
                data = {"_id": switch, "Name": ""}
                name = collection.find_one({"_id": switch}, {"SWI.Name": 1})
                if name is None:
                    data["Name"] = ""
                else:
                    data["Name"] = name["SWI"]["Name"]
                response[1].append(data)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switches_for_cntl)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active ONU MAC Addresses
    def get_all_onus(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection ONU_STATE
            collection = self.database["ONU-STATE"]
            onu_ids = collection.distinct("_id", {})
            collection = self.database["ONU-CFG"]
            index = 0
            while index < len(onu_ids):
                response[1].append({"_id": "", "Name": ""})
                response[1][index]["_id"] = onu_ids[index]
                document = collection.find_one({"_id": onu_ids[index]}, {"ONU.Name": 1})
                if document is not None and len(document["ONU"]) != 0:
                    response[1][index]["Name"] = document["ONU"]["Name"]
                index += 1
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onus)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE, ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all ONU alarm config identifiers
    def get_all_gpon_onu_serial_numbers(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            onu_cursor = collection.find({"ONU.PON Mode": "GPON"}, {"_id": 1})
            for doc in onu_cursor:
                response[1].append(doc['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_gpon_onu_serial_numbers)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all GPON ONU Serial Numbers due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves a list of ONUs with their versions, names and IDs
    def get_all_onus_with_versions(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection ONU_STATE
            collection = self.database["ONU-CFG"]
            onu_ids = collection.distinct("_id", {})
            index = 0
            while index < len(onu_ids):
                response[1].append({"_id": "", "name": ""})
                response[1][index]["_id"] = onu_ids[index]
                version = collection.find_one({"_id": onu_ids[index]}, {"CNTL.CFG Version": 1})
                response[1][index]["version"] = version["CNTL"]["CFG Version"].split("-")[0]
                document = collection.find_one({"_id": onu_ids[index]}, {"ONU.Name": 1})
                if document is not None and len(document["ONU"]) != 0:
                    response[1][index]["name"] = document["ONU"]["Name"]
                index += 1
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onus_with_versions)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves all unspecified ONUs
    def get_all_unspecified_onus(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all ONU CFG ids that are not found in an OLT STATE or CFG
            collection = self.database["ONU-CFG"]
            onus = collection.find({}, {"_id": 1, "ONU.Name": 1})
            collection = self.database["OLT-STATE"]
            olts_active = list(collection.find({}, {"ONUs": 1, "_id": 0}))
            collection = self.database["OLT-CFG"]
            olts_inactive = list(collection.find({}, {"ONUs": 1, "_id": 0}))
            for onu in onus:
                if onu["_id"] != "Default":
                    is_unspec = True
                    for olt_onu in olts_active:
                        if onu["_id"] in list(olt_onu["ONUs"].keys()):
                            is_unspec = False
                            break
                    for olt_onu in olts_inactive:
                        if onu["_id"] in list(olt_onu["ONUs"].keys()):
                            is_unspec = False
                            break
                    if is_unspec:
                        response[1].append({"_id": onu["_id"], "Name": onu["ONU"]["Name"]})
        except Exception as err:
            self.status_log("EXCEPTION (get_all_unspecified_onus)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get unspecified ONUs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG, OLT-STATE, OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all ONUs under the specified OLT
    def get_all_onus_for_olt(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            onus = collection.find({"_id": olt_id}, {"ONUs": 1})
            for onu in onus:
                response[1].append(onu)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onus_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs for OLT {olt_id} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve ONUs from ONU-STATE who's OLT matches the one passed it
    def get_all_onus_under_olt(self, olt_id, user_email):
        onu_state = ""
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            onus_obj = collection.find_one({"_id": olt_id}, {"ONUs": 1})
            if onus_obj is not None:
                onus = onus_obj["ONUs"]
                for onu in onus:
                    onu_coll = self.database["ONU-CFG"]
                    name = onu_coll.find_one({"_id": onu}, {"ONU.Name": 1})
                    if name is None:
                        name = ""
                    else:
                        name = name["ONU"]["Name"]
                    onu_states = collection.find_one({"_id": olt_id}, {"ONU States": 1})
                    states = onu_states["ONU States"]

                    found = False
                    for state in states:
                        if not found:
                            for value in states[state]:
                                if value == onu:
                                    onu_state = state
                                    found = True
                                    break
                                else:
                                    onu_state = ""

                    response[1].append({"_id": onu, "state": onu_state, "name": name})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onus_under_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs under OLT {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    # Retrieve an object formatted for ONU inventory report
    def get_onu_inventory(self, switch_id, user_email):
        response = [status.HTTP_200_OK, []]
        try:
            state_collection = self.database["ONU-STATE"]
            olt_collection = self.database["OLT-STATE"]

            switch_region_pop = self.get_switch_region_pop(user_email, switch_id)
            region = switch_region_pop[1]['Region']
            pop = switch_region_pop[1]['POP']

            #check if GPON and EPON mode are present for all ONUs in ONU state collection
            gpon = state_collection.find_one({'ONU.PON Mode': 'GPON'})
            epon = state_collection.find_one({'ONU.PON Mode': 'EPON'})

            #for OLTs with no switch info
            if switch_id == "":
                olts = list(olt_collection.aggregate([
                    {
                        '$match': {
                            'Switch.Chassis ID': {
                                '$exists': False
                            }
                        }
                    }, {
                        '$project': {
                            '_id': 1
                        }
                    }
                ]))
                olt_ids = [x['_id'] for x in olts]
            else:
                olts = list(olt_collection.find({'Switch.Chassis ID': switch_id}, {'_id':1}))
                olt_ids = [x['_id'] for x in olts]

            if epon and not gpon: #epon only
                onu_inventory = list(state_collection.aggregate([
                    {
                        '$match': {
                            'OLT.MAC Address': {'$in': olt_ids}
                        }
                    }, {
                        '$project': {
                            'CNTL.MAC Address': 1,
                            'OLT.MAC Address': 1,
                            'STATE-ONU': '$ONU',
                            'Time': 1
                        }
                    }, {
                        '$lookup': {
                            'from': 'ONU-CFG',
                            'localField': '_id',
                            'foreignField': '_id',
                            'as': 'CFG'
                        }
                    }, {
                        '$addFields': {
                            'Online_Duration': {
                                '$subtract': [{'$toDate': '$Time'}, {'$toDate': '$STATE-ONU.Online Time'}]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days_long': {
                                '$divide': ['$Online_Duration', 86400000]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days': {
                                '$trunc': ['$days_long']
                            },
                            'hours_long': {
                                '$multiply': [
                                    {
                                        '$subtract': ['$days_long', {'$trunc': ['$days_long']}]
                                    }, 24
                                ]
                            }
                        }
                    }, {
                        '$addFields': {
                            'hours': {
                                '$trunc': ['$hours_long']
                            },
                            'minutes': {
                                '$trunc': [
                                    {
                                        '$multiply': [
                                            {
                                                '$subtract': ['$hours_long', {'$trunc': ['$hours_long']}]
                                            }, 60
                                        ]
                                    }
                                ]
                            }
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            '[ONU-CFG][ONU][Address]': {'$arrayElemAt': ['$CFG.ONU.Address', 0]},
                            '[ONU-CFG][ONU][Location]': {'$arrayElemAt': ['$CFG.ONU.Location', 0]},
                            '[ONU-CFG][ONU][Name]': {'$arrayElemAt': ['$CFG.ONU.Name', 0]},
                            '[ONU-CFG][_id]': '$_id',
                            'Region': region,
                            'POP': pop,
                            '[ONU-CFG][ONU][Create Date]': {'$arrayElemAt': ['$CFG.ONU.Create Date', 0]},
                            '[ONU-STATE][ONU][Online Time]': '$STATE-ONU.Online Time',
                            'Online Duration': {
                                '$concat': [
                                    {
                                        '$toString': '$days'
                                    }, 'd, ', {
                                        '$toString': '$hours'
                                    }, 'h, ', {
                                        '$toString': '$minutes'
                                    }, 'm'
                                ]
                            },
                            '[ONU-STATE][CNTL][MAC Address]': {'$ifNull': ['$CNTL.MAC Address', '']},
                            '[ONU-STATE][OLT][MAC Address]': {'$ifNull': ['$OLT.MAC Address', '']},
                            '[ONU-STATE][ONU][Boot Version]': {'$ifNull': ['$STATE-ONU.Boot Version', '']},
                            '[ONU-STATE][ONU][Chip Model]': {'$ifNull': ['$STATE-ONU.Chip Model', '']},
                            '[ONU-STATE][ONU][Chip Version]': {'$ifNull': ['$STATE-ONU.Chip Version', '']},
                            '[ONU-STATE][ONU][DPoE Version]': {'$ifNull': ['$STATE-ONU.DPoE Version', '']},
                            '[ONU-STATE][ONU][FW Version]': {'$ifNull': ['$STATE-ONU.FW Version', '']},
                            '[ONU-STATE][ONU][JEDEC ID]': {'$ifNull': ['$STATE-ONU.JEDEC ID', '']},
                            '[ONU-STATE][ONU][Manufacturer]': {'$ifNull': ['$STATE-ONU.Manufacturer', '']},
                            '[ONU-STATE][ONU][Model]': {'$ifNull': ['$STATE-ONU.Model', '']},
                            '[ONU-STATE][ONU][Serial Number]': {'$ifNull': ['$STATE-ONU.Serial Number', '']},
                            '[ONU-STATE][ONU][Software Bundle]': {'$ifNull': ['$STATE-ONU.Software Bundle', '']},
                            '[ONU-STATE][ONU][Vendor]': {'$ifNull': ['$STATE-ONU.Vendor', '']}
                        }
                    }
                ]))
            elif gpon and not epon: #gpon only
                onu_inventory = list(state_collection.aggregate([
                    {
                        '$match': {
                            'OLT.MAC Address': {'$in': olt_ids}
                        }
                    }, {
                        '$project': {
                            'CNTL.MAC Address': 1,
                            'OLT.MAC Address': 1,
                            'STATE-ONU': '$ONU',
                            'Time': 1
                        }
                    }, {
                        '$lookup': {
                            'from': 'ONU-CFG',
                            'localField': '_id',
                            'foreignField': '_id',
                            'as': 'CFG'
                        }
                    }, {
                        '$addFields': {
                            'Online_Duration': {
                                '$subtract': [{'$toDate': '$Time'}, {'$toDate': '$STATE-ONU.Online Time'}]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days_long': {
                                '$divide': ['$Online_Duration', 86400000]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days': {
                                '$trunc': ['$days_long']
                            },
                            'hours_long': {
                                '$multiply': [
                                    {
                                        '$subtract': ['$days_long', {'$trunc': ['$days_long']}]
                                    }, 24
                                ]
                            }
                        }
                    }, {
                        '$addFields': {
                            'hours': {
                                '$trunc': ['$hours_long']
                            },
                            'minutes': {
                                '$trunc': [
                                    {
                                        '$multiply': [
                                            {
                                                '$subtract': ['$hours_long', {'$trunc': ['$hours_long']}]
                                            }, 60
                                        ]
                                    }
                                ]
                            }
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            '[ONU-CFG][_id]': '$_id',
                            '[ONU-CFG][ONU][Name]': {'$arrayElemAt': ['$CFG.ONU.Name', 0]},
                            '[ONU-CFG][ONU][Address]': {'$arrayElemAt': ['$CFG.ONU.Address', 0]},
                            '[ONU-CFG][ONU][Location]': {'$arrayElemAt': ['$CFG.ONU.Location', 0]},
                            'Region': region,
                            'POP': pop,
                            '[ONU-CFG][ONU][Create Date]': {'$arrayElemAt': ['$CFG.ONU.Create Date', 0]},
                            '[ONU-STATE][ONU][Online Time]': '$STATE-ONU.Online Time',
                            'Online Duration': {
                                '$concat': [
                                    {
                                        '$toString': '$days'
                                    }, 'd, ', {
                                        '$toString': '$hours'
                                    }, 'h, ', {
                                        '$toString': '$minutes'
                                    }, 'm'
                                ]
                            },
                            '[ONU-STATE][CNTL][MAC Address]': {'$ifNull': ['$CNTL.MAC Address', '']},
                            '[ONU-STATE][OLT][MAC Address]': {'$ifNull': ['$OLT.MAC Address', '']},
                            '[ONU-STATE][ONU][Equipment ID]': {'$ifNull': ['$STATE-ONU.Equipment ID', '']},
                            '[ONU-STATE][ONU][FW Version]': {'$ifNull': ['$STATE-ONU.FW Version', '']},
                            '[ONU-STATE][ONU][HW Version]': {'$ifNull': ['$STATE-ONU.HW Version', '']},
                            '[ONU-STATE][ONU][Host MAC Address]': {'$ifNull': ['$STATE-ONU.Host MAC Address', '']},
                            '[ONU-STATE][ONU][Manufacturer]': {'$ifNull': ['$STATE-ONU.Manufacturer', '']},
                            '[ONU-STATE][ONU][Model]': {'$ifNull': ['$STATE-ONU.Model', '']},
                            '[ONU-STATE][ONU][Registration ID]': {'$ifNull': ['$STATE-ONU.Registration ID', '']},
                            '[ONU-STATE][ONU][Serial Number]': {'$ifNull': ['$STATE-ONU.Serial Number', '']},
                            '[ONU-STATE][ONU][Vendor]': {'$ifNull': ['$STATE-ONU.Vendor', '']}
                        }
                    }
                ]))
            else: #both epon and gpon
                onu_inventory = list(state_collection.aggregate([
                    {
                        '$match': {
                            'OLT.MAC Address': {'$in': olt_ids}
                        }
                    }, {
                        '$project': {
                            'CNTL.MAC Address': 1,
                            'OLT.MAC Address': 1,
                            'STATE-ONU': '$ONU',
                            'Time': 1
                        }
                    }, {
                        '$lookup': {
                            'from': 'ONU-CFG',
                            'localField': '_id',
                            'foreignField': '_id',
                            'as': 'CFG'
                        }
                    }, {
                        '$addFields': {
                            'Online_Duration': {
                                '$subtract': [{'$toDate': '$Time'}, {'$toDate': '$STATE-ONU.Online Time'}]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days_long': {
                                '$divide': ['$Online_Duration', 86400000]
                            }
                        }
                    }, {
                        '$addFields': {
                            'days': {
                                '$trunc': ['$days_long']
                            },
                            'hours_long': {
                                '$multiply': [
                                    {
                                        '$subtract': ['$days_long', {'$trunc': ['$days_long']}]
                                    }, 24
                                ]
                            }
                        }
                    }, {
                        '$addFields': {
                            'hours': {
                                '$trunc': ['$hours_long']
                            },
                            'minutes': {
                                '$trunc': [
                                    {
                                        '$multiply': [
                                            {
                                                '$subtract': ['$hours_long', {'$trunc': ['$hours_long']}]
                                            }, 60
                                        ]
                                    }
                                ]
                            }
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            '[ONU-CFG][_id]': '$_id',
                            '[ONU-CFG][ONU][Name]': {'$arrayElemAt': ['$CFG.ONU.Name', 0]},
                            '[ONU-CFG][ONU][Address]': {'$arrayElemAt': ['$CFG.ONU.Address', 0]},
                            '[ONU-CFG][ONU][Location]': {'$arrayElemAt': ['$CFG.ONU.Location', 0]},
                            'Region': region,
                            'POP': pop,
                            '[ONU-CFG][ONU][Create Date]': {'$arrayElemAt': ['$CFG.ONU.Create Date', 0]},
                            '[ONU-STATE][ONU][Online Time]': '$STATE-ONU.Online Time',
                            'Online Duration': {
                                '$concat': [
                                    {
                                        '$toString': '$days'
                                    }, 'd, ', {
                                        '$toString': '$hours'
                                    }, 'h, ', {
                                        '$toString': '$minutes'
                                    }, 'm'
                                ]
                            },
                            '[ONU-STATE][CNTL][MAC Address]': {'$ifNull': ['$CNTL.MAC Address', '']},
                            '[ONU-STATE][OLT][MAC Address]': {'$ifNull': ['$OLT.MAC Address', '']},
                            '[ONU-STATE][ONU][Equipment ID]': {'$ifNull': ['$STATE-ONU.Equipment ID', '']},
                            '[ONU-STATE][ONU][FW Version]': {'$ifNull': ['$STATE-ONU.FW Version', '']},
                            '[ONU-STATE][ONU][HW Version]': {'$ifNull': ['$STATE-ONU.HW Version', '']},
                            '[ONU-STATE][ONU][Boot Version]': {'$ifNull': ['$STATE-ONU.Boot Version', '']},
                            '[ONU-STATE][ONU][Chip Model]': {'$ifNull': ['$STATE-ONU.Chip Model', '']},
                            '[ONU-STATE][ONU][Chip Version]': {'$ifNull': ['$STATE-ONU.Chip Version', '']},
                            '[ONU-STATE][ONU][DPoE Version]': {'$ifNull': ['$STATE-ONU.DPoE Version', '']},
                            '[ONU-STATE][ONU][Host MAC Address]': {'$ifNull': ['$STATE-ONU.Host MAC Address', '']},
                            '[ONU-STATE][ONU][JEDEC ID]': {'$ifNull': ['$STATE-ONU.JEDEC ID', '']},
                            '[ONU-STATE][ONU][Manufacturer]': {'$ifNull': ['$STATE-ONU.Manufacturer', '']},
                            '[ONU-STATE][ONU][Model]': {'$ifNull': ['$STATE-ONU.Model', '']},
                            '[ONU-STATE][ONU][Registration ID]': {'$ifNull': ['$STATE-ONU.Registration ID', '']},
                            '[ONU-STATE][ONU][Software Bundle]': {'$ifNull': ['$STATE-ONU.Software Bundle', '']},
                            '[ONU-STATE][ONU][Serial Number]': {'$ifNull': ['$STATE-ONU.Serial Number', '']},
                            '[ONU-STATE][ONU][Vendor]': {'$ifNull': ['$STATE-ONU.Vendor', '']}
                        }
                    }
                ]))
            response[1] = onu_inventory
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_inventory)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU inventory due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all ONU labels
    def get_all_onu_labels(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            distinct_labels = sorted(collection.distinct("ONU.Labels", {})[0:20], key=str.lower)
            response[1] = distinct_labels
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_labels)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU labels due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all OLT labels
    def get_all_olt_labels(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-CFG"]
            distinct_labels = sorted(collection.distinct("OLT.Labels", {})[0:20], key=str.lower)
            response[1] = distinct_labels
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_labels)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT labels due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all CNTL labels
    def get_all_cntl_labels(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            distinct_labels = sorted(collection.distinct("CNTL.Labels", {})[0:20], key=str.lower)
            response[1] = distinct_labels
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_labels)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get CNTL labels due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all SWI labels
    def get_all_switch_labels(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            distinct_labels = sorted(collection.distinct("SWI.Labels", {})[0:20], key=str.lower)
            response[1] = distinct_labels
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switch_labels)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get SWI labels due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get all Regions and associated POPs currently configured
    def get_all_switch_regions_pops(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            distinct_regions = sorted(collection.distinct("SWI.Region", {}), key=str.lower)
            for region in distinct_regions:
                if region != '':
                    distinct_pops = sorted(collection.distinct("SWI.POP", {"SWI.Region": region}), key=str.lower)
                    response[1][region] = distinct_pops
        except Exception as err:
            self.status_log("EXCEPTION (get_all_switch_regions_pops)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get SWI Regions and POPs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get the region and pop associated with a given switch MAC Address
    def get_switch_region_pop(self, user_email, mac_address):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            swi_cfg = collection.find_one({"_id": mac_address}, {"_id": 1, "SWI.Region": 1, "SWI.POP": 1})

            if swi_cfg is not None and 'Region' in swi_cfg['SWI'] and 'POP' in swi_cfg['SWI']:
                response[1]['Region'] = swi_cfg['SWI']['Region']
                response[1]['POP'] = swi_cfg['SWI']['POP']
            else:
                response[1]['Region'] = ''
                response[1]['POP'] = ''
        except Exception as err:
            self.status_log("EXCEPTION (get_switch_region_pop)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get SWI Region and POP due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve PON AUTO config
    def get_pon_auto_config(self, pon_auto_id):
        response = [status.HTTP_200_OK, [], None, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["AUTO-CFG"]
            response[1] = collection.find_one({"_id": pon_auto_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_config)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation config due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        return response

    # Retrieve PON AUTO config
    def update_pon_auto_config(self, pon_auto_id, new_config):
        response = [status.HTTP_200_OK, None, None, None]

        try:
            database_manager.get_database(self.database_id)
            if new_config and 'data' in new_config and '_id' in new_config['data']:
                collection = self.database["AUTO-CFG"]
                response[2] = new_config['data']
                response[3] = collection.find_one_and_replace(filter={"_id": pon_auto_id},
                                                              replacement=new_config['data'],
                                                              return_document=ReturnDocument.BEFORE, upsert=True)
            else:
                response = [status.HTTP_400_BAD_REQUEST,
                            "Request body must be of format '{{ data: <AUTO-CFG document> }}'", None, None]
        except Exception as err:
            self.status_log("EXCEPTION (update_pon_auto_config)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update PON Automation config due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        return response

    # Retrieve PON AUTO state
    def get_pon_auto_state(self, pon_auto_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["AUTO-STATE"]
            response[1] = collection.find_one({"_id": pon_auto_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # Retrieve PON AUTO state count from AUTO-STATE collection
    def get_pon_auto_state_count(self):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["AUTO-STATE"]
            response[1] = len(list(collection.find({}, {"_id": 1})))
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_state_count)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation state count due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # Retrive PON AUTO states and whether they're online or not
    def get_pon_auto_status(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            automation_states = list(self.database["AUTO-STATE"].find({}, {"Time":1}).sort("Time", pymongo.DESCENDING))
            for state in automation_states:
                online = False
                if "Time" in state:
                    online = self.automation_time_is_online(state["Time"])
                state["Online"] = online
                response[1].append(state)

        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_status)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation status due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)
        return response

    """
    Get tree data
    """

    # Retrieve all unique PON AUTO services from AUTO-STATE collection
    def get_pon_auto_services_for_tree(self):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["AUTO-STATE"]
            pon_auto_services = list(collection.aggregate([
                {
                    "$lookup": {
                        "from": "AUTO-CFG",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "CFG"
                    }
                },
                {
                    "$addFields": {
                        "Name": {
                            "$arrayElemAt": ["$CFG.AUTO.Name", 0]
                        }
                    }
                },
                {
                    "$project": {
                        "_id": 1,
                        "Name": 1,
                        "Time": 1,
                        "AUTO.CFG Version": 1
                    }
                }
            ]))

            for service in pon_auto_services:
                response[1].append({
                    "_id": service["_id"],
                    "name": service["Name"],
                    "offline": not self.automation_time_is_online(service["Time"])
                })

            response[1] = sorted(response[1], key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_auto_services_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Automation services list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # Retrieve all unique MGMT LANs from CNTL-STATEs
    def get_mgmt_lans_for_tree(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            lan_resp = collection.distinct("MGMT LAN.Name", {})

            index = 0
            while index < len(lan_resp):
                response[1].append({"name": ""})
                # Get State data
                response[1][index]["name"] = lan_resp[index]
                index += 1

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower()))
        except Exception as err:
            self.status_log("EXCEPTION (get_mgmt_lans_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of controllers and associated tree data
    def get_cntls_for_tree(self, user_email, mgmt_lan):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection CNTL_STATE
            collection = self.database.get_collection("CNTL-STATE")
            controllers = list(collection.aggregate([
                {
                    "$match": {
                        "MGMT LAN.Name": mgmt_lan
                    }
                },
                {
                    "$project": {
                        "_id": 1, "MGMT LAN.Name": 1, "CNTL.Version": 1, "CNTL.Pause": 1, "Time": 1,
                        "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                        "Alarm.4-WARNING": 1
                    }
                },
                {
                    "$lookup": {
                        "from": "CNTL-CFG",
                        "let": {"stateId": "$_id"},
                        "pipeline": [
                            {
                                "$match": {
                                    "$expr": {"$eq": ["$_id", "$$stateId"]}
                                }
                            },
                            {
                                "$project": {
                                    "CNTL.Name": 1
                                }
                            }
                        ],
                        "as": "CNTL-CFG"
                    }
                },
                {
                    "$lookup": {
                        "from": "CNTL-ALARM-HIST-STATE",
                        "let": {"stateId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$stateId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }
                        ], "as": "CNTL-ALARM-HIST-STATE"
                    }
                },
                {
                    '$lookup': {
                        'from': 'CNTL-AUTO-STATE',
                        "let": {"stateId": "$_id"},
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$stateId'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Enable': '$AUTO.Enable'
                                }
                            }
                        ],
                        'as': 'CNTL-AUTO'
                    }
                },
                {
                    '$lookup': {
                        'from': 'AUTO-STATE',
                        'as': 'AUTO-Time',
                        'pipeline': [
                            {
                                '$project': {
                                    'Time': 1
                                }
                            }
                        ]
                    }
                }
            ]))

            for controller in controllers:
                alarm_history_supported = False

                # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
                if controller and controller["CNTL"] and get_nested_value(controller,["CNTL","Version"]) is not None and \
                        (len(get_nested_value(controller,["CNTL","Version"])) > 2) and int(
                        controller["CNTL"]["Version"][1:2]) >= 3:
                    alarm_history_supported = True

                id = controller["_id"]
                name = get_nested_value(controller, ["CNTL-CFG", 0, "CNTL", "Name"], "")
                offline = self.controller_time_is_online(controller["Time"], controller["CNTL"]["Version"]) == False
                paused = controller["CNTL"]["Pause"]
                mgmt_lan = controller["MGMT LAN"]["Name"]

                auto_disabled = False
                if "CNTL-AUTO" in controller:
                    if len(controller["CNTL-AUTO"]) > 0 and "Enable" in controller["CNTL-AUTO"][0]:
                        auto_disabled = not controller["CNTL-AUTO"][0]["Enable"]

                auto_offline = True
                if "AUTO-Time" in controller and len(controller["AUTO-Time"]) > 0 and "Time" in controller["AUTO-Time"][0]:
                    auto_offline = not self.automation_time_is_online(controller["AUTO-Time"][0]["Time"])

                if alarm_history_supported:
                    emerg_alarms = []
                    alert_alarms = []
                    crit_alarms = []
                    error_alarms = []
                    warning_alarms = []

                    if 'CNTL-ALARM-HIST-STATE' in controller and len(controller['CNTL-ALARM-HIST-STATE']) > 0 and \
                            controller['CNTL-ALARM-HIST-STATE'][0]['Alarms']:
                        for alarm_history in controller['CNTL-ALARM-HIST-STATE'][0]['Alarms']:
                            alarm_text = alarm_history['Text']
                            if alarm_history['Severity'] == "0-EMERG":
                                emerg_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "1-ALERT":
                                alert_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "2-CRIT":
                                crit_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "3-ERROR":
                                error_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "4-WARNING":
                                warning_alarms.append(alarm_text)

                    errors = {
                        "EMERG": emerg_alarms,
                        "ALERT": alert_alarms,
                        "CRIT": crit_alarms,
                        "ERROR": error_alarms
                    }
                    warnings = warning_alarms
                else:
                    errors = {
                        "EMERG": controller["Alarm"]["0-EMERG"],
                        "ALERT": controller["Alarm"]["1-ALERT"],
                        "CRIT": controller["Alarm"]["2-CRIT"],
                        "ERROR": controller["Alarm"]["3-ERROR"]
                    }
                    warnings = controller["Alarm"]["4-WARNING"]

                response[1].append({"_id": id,
                                    "name": name,
                                    "offline": offline,
                                    "auto_offline": auto_offline,
                                    "cntl_auto_disabled": auto_disabled,
                                    "paused": paused,
                                    "mgmt_lan": mgmt_lan,
                                    "errors": errors,
                                    "warnings": warnings})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_cntls_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controllers list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve regions for tree
    def get_regions_for_tree(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_macs = collection.distinct("Switch.Chassis ID", {})
            unassigned_region_added = False

            collection = self.database["SWI-CFG"]

            switch_cfgs = collection.find({"_id": {"$in": switch_macs}}, {"_id": 1, "SWI.Region": 1, "SWI.POP": 1})

            for cfg in switch_cfgs:
                if (
                        ('Region' in cfg['SWI'] and 'POP' in cfg['SWI']) and
                        (cfg['SWI']['Region'].strip() != '' and cfg['SWI']['POP'].strip() != '')
                ):
                    if cfg['SWI']['Region'] not in [value for elem in response[1] for value in elem.values()]:
                        data = {"_id": cfg['SWI']['Region']}
                        response[1].append(data)
                # WARN: The space following Unassigned is intentional. This is to distinguish the special Unassigned heirarchy from a manually created Unassigned Region
                elif 'Unassigned ' not in [value for elem in response[1] for value in
                                           elem.values()]:  # Only add Unassigned node if not already added
                    data = {"_id": 'Unassigned '}
                    response[1].append(data)
                    unassigned_region_added = True

            if not unassigned_region_added:
                # Add unknown switch object if there's an olt without a switch
                collection = self.database["OLT-STATE"]
                olts_without_switch = collection.find_one({'Switch': {}})
                if olts_without_switch is not None:
                    data = {"_id": 'Unassigned '}
                    response[1].append(data)

        except Exception as err:
            self.status_log("EXCEPTION (get_regions_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Regions list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve regions for tree
    def get_pops_under_region_for_tree(self, user_email, region):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_macs = collection.distinct("Switch.Chassis ID", {})

            collection = self.database["SWI-CFG"]
            distinct_pops = sorted(collection.distinct("SWI.POP", {"_id": {"$in": switch_macs}, "SWI.Region": region}),
                                   key=str.lower)

            for pop in distinct_pops:
                if pop.strip() != '':
                    data = {"_id": pop}
                    response[1].append(data)

        except Exception as err:
            self.status_log("EXCEPTION (get_pops_under_region_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get POPs list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get All switches under a pop for the tree
    # POPs can have the same name under different Regions, so region is also required
    def get_switches_under_pop_for_tree(self, user_email, region, pop):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            switch_macs = collection.distinct("Switch.Chassis ID", {})
            switch_automation_states_list = list(self.database["SWI-AUTO-STATE"].find({"_id": {"$in": switch_macs}}, {"_id":1, "AUTO.Enable":1, "Time":1}))
            switch_automation_states = {}
            auto_offline = True
            if len(switch_automation_states_list) > 0:
                for switch_auto_state in switch_automation_states_list:
                    if switch_auto_state["_id"] != None:
                        switch_automation_states[switch_auto_state["_id"]] = switch_auto_state
                        if auto_offline:
                            auto_offline = not self.automation_time_is_online(switch_auto_state["Time"])

            olts_without_switch = None
            if region == 'Unassigned ' and pop == 'Unassigned ':
                olts_without_switch = collection.find_one({'Switch': {}})

            collection = self.database["SWI-CFG"]

            for switch in switch_macs:
                data = {"_id": switch, "name": "", "auto_offline": auto_offline, "switch_auto_disabled": True}
                switch_cfg_data = collection.find_one({"_id": switch}, {"SWI.Name": 1, "SWI.Region": 1, "SWI.POP": 1})

                if switch in switch_automation_states:
                    data["switch_auto_disabled"] = not get_nested_value(switch_automation_states, [switch, "AUTO", "Enable"], False)

                if switch_cfg_data is not None:
                    # WARN: The space following Unassigned is intentional. This is to distinguish the special Unassigned heirarchy from a manually created Unassigned Region
                    # If region and pop are 'Unassigned ', zero out region and pop fields to find all unassigned values in DB.
                    # API call does not work properly if you try to use blank values in URL path parameters
                    if region == 'Unassigned ' and pop == 'Unassigned ':
                        region = ''
                        pop = ''
                    # Assigned Switches
                    if region != '' and pop != '':
                        if 'Region' in switch_cfg_data["SWI"] and 'POP' in switch_cfg_data["SWI"] and \
                                switch_cfg_data["SWI"]["Region"] == region and switch_cfg_data["SWI"]["POP"] == pop:
                            if switch_cfg_data is None:
                                data["name"] = ""
                            else:
                                data["name"] = switch_cfg_data["SWI"]["Name"]
                            response[1].append(data)
                    else:  # Unassigned Switches
                        if 'Region' not in switch_cfg_data["SWI"] or 'POP' not in switch_cfg_data["SWI"] or \
                                switch_cfg_data["SWI"]["Region"].strip() == '' or switch_cfg_data["SWI"][
                            "POP"].strip() == '':
                            if switch_cfg_data is None:
                                data["name"] = ""
                            else:
                                data["name"] = switch_cfg_data["SWI"]["Name"]
                            response[1].append(data)

            if olts_without_switch != None:
                response[1].append({'_id': 'Unknown Switch', 'name':''})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))

        except Exception as err:
            self.status_log("EXCEPTION (get_switches_under_pop_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switches list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olts_under_switch_for_tree(self, user_email, switch_id):
        response = [status.HTTP_200_OK, []]
        isOffline = False
        alarm_history_supported = False
        system_status = ''

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            collection = self.database["OLT-STATE"]
            if switch_id == "Default":
                olts = collection.find({"Switch.Chassis ID": ""}, {"_id": 1})

            else:
                match_dict = {"Switch.Chassis ID": switch_id}
                if switch_id == 'Unknown Switch':
                    match_dict = {'Switch': {}}

                olts = list(collection.aggregate([
                    {
                        "$match": match_dict
                    },
                    {
                        "$project": {
                            "_id": 1, "CNTL.MAC Address": 1, "CNTL.Version": 1, "Switch.Port ID": 1,
                            "Switch.Chassis ID": 1,
                            "Protection.Peer": 1, "Protection.Status": 1, "ONUs": 1, "ONU States": 1,
                            "OLT.Reset Count": 1,
                            "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                            "Alarm.4-WARNING": 1,
                            "OLTs": 1
                        }
                    },
                    {
                        "$lookup": {
                            "from": "OLT-CFG",
                            "let": {"stateId": "$_id"},
                            "pipeline": [
                                {
                                    "$match": {
                                        "$expr": {"$eq": ["$_id", "$$stateId"]}
                                    }
                                },
                                {
                                    "$project": {
                                        "OLT.Reset Count": 1, "OLT.Name": 1
                                    }
                                }
                            ],
                            "as": "oltCfg"
                        }
                    },
                    {
                        "$lookup": {
                            "from": "OLT-ALARM-HIST-STATE",
                            "let": {"stateId": "$_id"},
                            "pipeline": [
                                {"$match": {"$expr": {"$eq": ["$_id", "$$stateId"]}}},
                                {
                                    "$project": {
                                        "Alarms": {
                                            "$filter": {
                                                "input": "$Alarms",
                                                "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                                  {"$eq": ["$$this.Ack State", False]}]}
                                            }
                                        }
                                    }
                                }
                            ], "as": "OLT-ALARM-HIST-STATE"
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'CNTL-AUTO-STATE',
                            "let": {"stateId": "$CNTL.MAC Address"},
                            'pipeline': [
                                {
                                    '$match': {
                                        '$expr': {
                                            '$eq': [
                                                '$_id', '$$stateId'
                                            ]
                                        }
                                    }
                                }, {
                                    '$project': {
                                        'Enable': '$AUTO.Enable'
                                    }
                                }
                            ],
                            'as': 'CNTL-AUTO'
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'AUTO-STATE',
                            'as': 'AUTO-Time',
                            'pipeline': [
                                {
                                    '$project': {
                                        'Time': 1
                                    }
                                }
                            ]
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'OLT-AUTO-STATE',
                            "let": {"stateId": "$_id"},
                            'pipeline': [
                                {
                                    '$match': {
                                        '$expr': {
                                            '$eq': [
                                                '$_id', '$$stateId'
                                            ]
                                        }
                                    }
                                }, {
                                    '$project': {
                                        'Enable': '$AUTO.Enable'
                                    }
                                }
                            ],
                            'as': 'OLT-AUTO'
                        }
                    }
                ]))

            # Getting pre-provisioned OLTs
            prepro_olts = list(self.database["SWI-CFG"].aggregate([
                {"$match": {"_id": switch_id}},
                {"$addFields": {"olts": {"$objectToArray": "$OLTs"}}},
                {"$project": {"olts": 1}},
            ]))

            if switch_id == "Unknown Switch":
                ifOffline = False
                pass

            # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
            if olts and len(olts) > 0 and olts[0]["CNTL"] and olts[0]["CNTL"]["Version"] and int(
                    olts[0]["CNTL"]["Version"][1:2]) >= 3:
                alarm_history_supported = True

            auto_offline = True
            for olt_data in olts:
                if auto_offline == True and "AUTO-Time" in olt_data and len(olt_data["AUTO-Time"]) > 0 and "Time" in \
                        olt_data["AUTO-Time"][0]:
                    auto_offline = not self.automation_time_is_online(olt_data["AUTO-Time"][0]["Time"])

            for olt in olts:
                reg_onus = 0
                system_status = ''
                # Get OLTs Name if it exists
                if olt['oltCfg'] and olt['oltCfg'][0]:
                    olt_cfg = olt['oltCfg'][0]
                    if olt_cfg["OLT"]["Name"] is not None:
                        name = olt_cfg["OLT"]["Name"]
                    if olt_cfg["OLT"]["Reset Count"] is not None:
                        reset_count_config = olt_cfg["OLT"]["Reset Count"]
                else:
                    name = ""
                    reset_count_config = 99999999

                # Get Offline status
                controller_id = olt["CNTL"]["MAC Address"]
                cntl_state = self.database["CNTL-STATE"].find_one({"_id": controller_id},
                                                                  {"Time": 1, "System Status": 1, "CNTL.Version": 1})
                isOffline = False
                if cntl_state is not None and "Time" in cntl_state:
                    if self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"]):
                        if olt is not None and olt["_id"] in cntl_state["System Status"]:
                            if cntl_state["System Status"][olt["_id"]]["OLT State"] == "Unspecified" or \
                                    cntl_state["System Status"][olt["_id"]]["OLT State"] == "Primary" or \
                                    cntl_state["System Status"][olt["_id"]]["OLT State"] == "Secondary":
                                isOffline = False
                                if olt is not None and "ONU States" in olt:
                                    if "Registered" in olt["ONU States"] and "Unspecified" in olt["ONU States"]:
                                        reg_onus = len(olt["ONU States"]["Registered"]) + len(
                                            olt["ONU States"]["Unspecified"])

                                # Determine OLT Protection Active/Standby Status
                                if 'Protection' in olt and 'Peer' in olt['Protection'] and len(
                                        olt['Protection']['Peer']) > 0:
                                    system_status = olt['Protection']['Status']

                            else:
                                isOffline = True
                        else:
                            isOffline = True
                    else:
                        isOffline = True
                else:
                    isOffline = True

                # Test if OLT is resetting if it is online
                if not isOffline:
                    isResetting = olt["OLT"]["Reset Count"] != reset_count_config
                    if isResetting:
                        system_status = "Pending Reset"

                if alarm_history_supported:
                    emerg_alarms = []
                    alert_alarms = []
                    crit_alarms = []
                    error_alarms = []
                    warning_alarms = []

                    if 'OLT-ALARM-HIST-STATE' in olt and len(olt['OLT-ALARM-HIST-STATE']) > 0 and \
                            olt['OLT-ALARM-HIST-STATE'][0]['Alarms']:
                        for alarm_history in olt['OLT-ALARM-HIST-STATE'][0]['Alarms']:
                            alarm_text = alarm_history['Text']
                            if alarm_history['Severity'] == "0-EMERG":
                                emerg_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "1-ALERT":
                                alert_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "2-CRIT":
                                crit_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "3-ERROR":
                                error_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "4-WARNING":
                                warning_alarms.append(alarm_text)

                    errors = {
                        "EMERG": emerg_alarms,
                        "ALERT": alert_alarms,
                        "CRIT": crit_alarms,
                        "ERROR": error_alarms
                    }
                    warnings = warning_alarms
                else:
                    errors = {
                        "EMERG": olt["Alarm"]["0-EMERG"],
                        "ALERT": olt["Alarm"]["1-ALERT"],
                        "CRIT": olt["Alarm"]["2-CRIT"],
                        "ERROR": olt["Alarm"]["3-ERROR"]
                    }
                    warnings = olt["Alarm"]["4-WARNING"]

                port = ''

                switch = ''

                if "Switch" in olt and "Port ID" in olt["Switch"]:
                    port = olt["Switch"]["Port ID"]
                if "Switch" in olt and "Chassis ID" in olt["Switch"]:
                    switch = olt["Switch"]["Chassis ID"]

                # Get OLT Automation Enable
                olt_auto_disabled = True
                cntl_auto_disabled = True

                if "CNTL-AUTO" in olt:
                    if len(olt["CNTL-AUTO"]) > 0 and "Enable" in olt["CNTL-AUTO"][0]:
                        cntl_auto_disabled = not olt["CNTL-AUTO"][0]["Enable"]

                if "OLT-AUTO" in olt:
                    if len(olt["OLT-AUTO"]) > 0 and "Enable" in olt["OLT-AUTO"][0]:
                        olt_auto_disabled = not olt["OLT-AUTO"][0]["Enable"]

                response[1].append({"_id": olt["_id"],
                                    "port": port,
                                    "cntl": olt["CNTL"]["MAC Address"],
                                    "name": name,
                                    "offline": isOffline,
                                    "auto_offline": auto_offline,
                                    "cntl_auto_disabled": cntl_auto_disabled,
                                    "olt_auto_disabled": olt_auto_disabled,
                                    "total_onus": len(olt["ONUs"]),
                                    "reg_onus": reg_onus,
                                    "switch": switch,
                                    "state": system_status,
                                    "errors": errors,
                                    "warnings": warnings})

            for doc in prepro_olts:
                unique = True
                for olt in doc["olts"]:
                    olt_id = olt["k"]
                    if olt_id is not "Default":
                        for item in response[1]:
                            if "_id" in item and olt_id == item["_id"]:
                                unique = False
                        if unique:
                            olt_cfg = self.database["OLT-CFG"].find_one({"_id": olt_id}, {"_id": 1, "OLT.Name": 1})
                            name = ''
                            errors = {
                                "EMERG": [],
                                "ALERT": [],
                                "CRIT": [],
                                "ERROR": []
                            }
                            warnings = []
                            if olt_cfg:
                                name = olt_cfg["OLT"]["Name"]
                            response[1].append({"_id": olt_id,
                                                "port": olt["v"]["Port ID"],
                                                "cntl": '',
                                                "name": name,
                                                "offline": True,
                                                "total_onus": 0,
                                                "reg_onus": 0,
                                                "switch": switch_id,
                                                "state": '',
                                                "errors": errors,
                                                "warnings": warnings})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['port'], i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_olts_under_switch_for_tree)", err)
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs for Switch {switch_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0])

        return response

    def get_olts_under_controller_for_tree(self, user_email, controller_id):
        response = [status.HTTP_200_OK, []]
        isOffline = False
        alarm_history_supported = False
        system_status = ''

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            olt_state_coll = self.database["OLT-STATE"]
            if controller_id == "Default":
                state_olts = olt_state_coll.find({"CNTL.MAC Address": ""}, {"_id": 1})
            else:
                state_olts = list(olt_state_coll.aggregate([
                    {
                        "$match": {
                            "CNTL.MAC Address": controller_id
                        }
                    },
                    {
                        "$project": {
                            "_id": 1, "CNTL.MAC Address": 1, "CNTL.Version": 1, "Switch.Port ID": 1,
                            "Switch.Chassis ID": 1,
                            "Protection.Peer": 1, "Protection.Status": 1, "ONUs": 1, "ONU States": 1,
                            "OLT.Reset Count": 1,
                            "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                            "Alarm.4-WARNING": 1
                        }
                    },
                    {
                        "$lookup": {
                            "from": "OLT-CFG",
                            "let": {"stateId": "$_id"},
                            "pipeline": [
                                {
                                    "$match": {
                                        "$expr": {"$eq": ["$_id", "$$stateId"]}
                                    }
                                },
                                {
                                    "$project": {
                                        "OLT.Reset Count": 1, "OLT.Name": 1
                                    }
                                }
                            ],
                            "as": "oltCfg"
                        }
                    },
                    {
                        "$lookup": {
                            "from": "OLT-ALARM-HIST-STATE",
                            "let": {"stateId": "$_id"},
                            "pipeline": [
                                {"$match": {"$expr": {"$eq": ["$_id", "$$stateId"]}}},
                                {
                                    "$project": {
                                        "Alarms": {
                                            "$filter": {
                                                "input": "$Alarms",
                                                "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                                  {"$eq": ["$$this.Ack State", False]}]}
                                            }
                                        }
                                    }
                                }
                            ], "as": "OLT-ALARM-HIST-STATE"
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'CNTL-AUTO-STATE',
                            "let": {"stateId": "$CNTL.MAC Address"},
                            'pipeline': [
                                {
                                    '$match': {
                                        '$expr': {
                                            '$eq': [
                                                '$_id', '$$stateId'
                                            ]
                                        }
                                    }
                                }, {
                                    '$project': {
                                        'Enable': '$AUTO.Enable'
                                    }
                                }
                            ],
                            'as': 'CNTL-AUTO'
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'AUTO-STATE',
                            'as': 'AUTO-Time',
                            'pipeline': [
                                {
                                    '$project': {
                                        'Time': 1
                                    }
                                }
                            ]
                        }
                    },
                    {
                        '$lookup': {
                            'from': 'OLT-AUTO-STATE',
                            "let": {"stateId": "$_id"},
                            'pipeline': [
                                {
                                    '$match': {
                                        '$expr': {
                                            '$eq': [
                                                '$_id', '$$stateId'
                                            ]
                                        }
                                    }
                                }, {
                                    '$project': {
                                        'Enable': '$AUTO.Enable'
                                    }
                                }
                            ],
                            'as': 'OLT-AUTO'
                        }
                    }
                ]))
            # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
            if state_olts and len(state_olts) > 0 and state_olts[0]["CNTL"] and state_olts[0]["CNTL"][
                "Version"] and int(state_olts[0]["CNTL"]["Version"][1:2]) >= 3:
                alarm_history_supported = True

            cntl_state = self.database["CNTL-STATE"].find_one({"_id": controller_id},
                                                              {"Time": 1, "System Status": 1, "OLTs": 1,
                                                               "CNTL.Version": 1})

            state_ids = []
            all_olts = []
            for olt in state_olts:
                state_ids.append(olt["_id"])
                all_olts.append(olt)

            inventoried_olts_without_state = list(self.database["CNTL-CFG"].aggregate([
                {"$match": {"_id": controller_id}},
                {"$project": {"OLTs.Primary": 1}},
                {"$lookup": {
                    "from": "OLT-CFG",
                    "let": {"primaryInventories": "$OLTs.Primary"},
                    "pipeline": [{"$match": {"$expr": {
                        "$and": [{"$in": ["$_id", "$$primaryInventories"]}, {"$not": {"$in": ["$_id", state_ids]}}]
                    }}},
                        {"$project": {"_id": 1, "ONUs": 1, "OLT.Name": 1, "OLT.Reset Count": 1}}
                    ], "as": "oltCfg"
                }
                }
            ]))[0]

            cntl_auto_disabled = True
            auto_offline = True

            for olt_data in all_olts:
                # Automation enable for parent devices.
                if cntl_auto_disabled == True and "CNTL-AUTO" in olt_data:
                    if len(olt_data["CNTL-AUTO"]) > 0 and len(olt_data["CNTL-AUTO"]) > 0 and "Enable" in olt_data["CNTL-AUTO"][0]:
                        cntl_auto_disabled = not olt_data["CNTL-AUTO"][0]["Enable"]
                if auto_offline == True and "AUTO-Time" in olt_data and len(olt_data["AUTO-Time"]) > 0 and "Time" in olt_data["AUTO-Time"][0]:
                    auto_offline = not self.automation_time_is_online(olt_data["AUTO-Time"][0]["Time"])

                # compile a list of olt consisting of stateolt and inventoried_olts_without_state olts
            for olt in inventoried_olts_without_state['oltCfg']:
                olt["CNTL"] = {}
                olt["CNTL"]["MAC Address"] = inventoried_olts_without_state["_id"]
                all_olts.append(olt)

            for olt in all_olts:
                isUnattached = False
                reg_onus = 0
                system_status = ''
                # Get OLTs Name if it exists
                if "oltCfg" in olt:
                    if len(olt["oltCfg"]) > 0 and "OLT" in olt["oltCfg"][0]:
                        #### IF YOU GET HERE, THEN YOU HAVE STATE && CFG
                        if "Name" in olt['oltCfg'][0]["OLT"]:
                            name = olt['oltCfg'][0]["OLT"]["Name"]
                        if "Reset Count" in olt['oltCfg'][0]["OLT"]:
                            reset_count_config = olt['oltCfg'][0]["OLT"]["Reset Count"]
                    else:
                        #### IF YOU GET HERE, THEN YOU HAVE STATE BUT NOT CFG
                        name = ""
                        reset_count_config = 99999999
                else:
                    #### IF YOU GET HERE, THEN YOU HAVE CFG BUT NO STATE
                    if "OLT" in olt:
                        if 'Name' in olt["OLT"]:
                            name = olt["OLT"]["Name"]
                        # These olts have no state and therefore a comparision is not made btw state and cfg
                        reset_count_config = -1

                        # Add fields that are missing in CFG as ''
                        olt['Switch'] = {}
                        olt["Switch"]["Port ID"] = ''
                        olt["Switch"]["Chassis ID"] = ''
                        olt["Alarm"] = {}
                        olt["Alarm"]["0-EMERG"] = ''
                        olt["Alarm"]["1-ALERT"] = ''
                        olt["Alarm"]["2-CRIT"] = ''
                        olt["Alarm"]["3-ERROR"] = ''
                        olt["Alarm"]["4-WARNING"] = ''

                # Getting Offline status
                isOffline = False
                if cntl_state is not None and "Time" in cntl_state:
                    if self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"]):
                        if olt is not None and olt["_id"] in cntl_state["System Status"]:
                            if cntl_state["System Status"][olt["_id"]]["OLT State"] == "Unspecified" or \
                                    cntl_state["System Status"][olt["_id"]]["OLT State"] == "Primary" or \
                                    cntl_state["System Status"][olt["_id"]]["OLT State"] == "Secondary":
                                isOffline = False
                                if olt is not None and "ONU States" in olt:
                                    if "Registered" in olt["ONU States"] and "Unspecified" in olt["ONU States"]:
                                        reg_onus = len(olt["ONU States"]["Registered"]) + len(
                                            olt["ONU States"]["Unspecified"])

                                # Determine OLT Protection Active/Standby Status
                                if 'Protection' in olt and 'Peer' in olt['Protection'] and len(
                                        olt['Protection']['Peer']) > 0:
                                    system_status = olt['Protection']['Status']
                            else:
                                isOffline = True
                        else:
                            # Check if OLT is inventoried, if not then its considered unattached
                            if olt is not None and olt["_id"] in cntl_state["OLTs"]["Primary"] or olt["_id"] in \
                                    cntl_state["OLTs"]["Primary Missing"] or olt["_id"] in cntl_state["OLTs"][
                                "Primary Free"] or \
                                    olt["_id"] in cntl_state["OLTs"]["Secondary"] or olt["_id"] in \
                                    cntl_state["OLTs"]["Secondary Free"] or olt["_id"] in cntl_state["OLTs"][
                                "Secondary Missing"] or olt["_id"] in cntl_state["OLTs"]["Secondary Others"]:
                                isOffline = True
                            else:
                                isUnattached = True
                    else:
                        isOffline = True
                else:
                    isOffline = True

                #Get OLT Automation Enable
                olt_auto_disabled = True
                if "OLT-AUTO" in olt and len(olt["OLT-AUTO"]) > 0:
                    if "Enable" in olt["OLT-AUTO"][0]:
                        olt_auto_disabled = not olt["OLT-AUTO"][0]["Enable"]

                if not isUnattached:
                    # Test if OLT is resetting if it is online
                    if not isOffline:
                        isResetting = olt["OLT"]["Reset Count"] != reset_count_config
                        if reset_count_config == -1:
                            isResetting = False
                        if isResetting:
                            system_status = "Pending Reset"

                    if alarm_history_supported:
                        emerg_alarms = []
                        alert_alarms = []
                        crit_alarms = []
                        error_alarms = []
                        warning_alarms = []

                        if 'OLT-ALARM-HIST-STATE' in olt and len(olt['OLT-ALARM-HIST-STATE']) > 0 and \
                                olt['OLT-ALARM-HIST-STATE'][0]['Alarms']:
                            for alarm_history in olt['OLT-ALARM-HIST-STATE'][0]['Alarms']:
                                alarm_text = alarm_history['Text']
                                if alarm_history['Severity'] == "0-EMERG":
                                    emerg_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "1-ALERT":
                                    alert_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "2-CRIT":
                                    crit_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "3-ERROR":
                                    error_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "4-WARNING":
                                    warning_alarms.append(alarm_text)

                        errors = {
                            "EMERG": emerg_alarms,
                            "ALERT": alert_alarms,
                            "CRIT": crit_alarms,
                            "ERROR": error_alarms
                        }
                        warnings = warning_alarms
                    else:
                        errors = {
                            "EMERG": olt["Alarm"]["0-EMERG"],
                            "ALERT": olt["Alarm"]["1-ALERT"],
                            "CRIT": olt["Alarm"]["2-CRIT"],
                            "ERROR": olt["Alarm"]["3-ERROR"]
                        }
                        warnings = olt["Alarm"]["4-WARNING"]

                    switch_port = ""
                    switch_chassis = ""
                    if "Switch" in olt:
                        if "Port ID" in olt["Switch"]:
                            switch_port = olt["Switch"]["Port ID"]
                        if "Chassis ID" in olt["Switch"]:
                            switch_chassis = olt["Switch"]["Chassis ID"]

                    response[1].append({"_id": olt["_id"],
                                        "port": switch_port,
                                        "cntl": olt["CNTL"]["MAC Address"],
                                        "name": name,
                                        "offline": isOffline,
                                        "auto_offline": auto_offline,
                                        "cntl_auto_disabled": cntl_auto_disabled,
                                        "olt_auto_disabled": olt_auto_disabled,
                                        "total_onus": len(olt["ONUs"]),
                                        "reg_onus": reg_onus,
                                        "switch": switch_chassis,
                                        "state": system_status,
                                        "errors": errors,
                                        "warnings": warnings})
            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_olts_under_controller_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs for Controller {controller_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, OLT-CFG",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onus_under_olt_for_tree(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]
        alarm_history_supported = False

        try:
            database_manager.get_database(self.database_id)
            # Get System Status from the controller of the given OLT
            system_status_docs = list(self.database["CNTL-STATE"].find(
                {'$or': [{f"System Status.{olt_id}.OLT State": "Primary"},
                         {f"System Status.{olt_id}.OLT State": "Secondary"},
                         {f"System Status.{olt_id}.OLT State": "Unspecified"}]},
                {"System Status": 1, "Time": 1, "_id": 1, "CNTL.Version": 1}
            ).sort("Time", pymongo.DESCENDING).limit(1))

            system_status_doc = None
            if len(system_status_docs) > 0:
                system_status_doc = system_status_docs[0]
                match_document = {
                    "_id": {"$in": list(system_status_doc["System Status"][olt_id]["ONUs"].keys())}
                }
            else:
                match_document = {
                    "OLT.MAC Address": olt_id
                }

            onu_cfg_coll = self.database["ONU-CFG"]
            onus = list(onu_cfg_coll.aggregate([
                {
                    "$match": match_document
                },
                {
                    "$project": {
                        "ONU.Name": 1, "ONU.Reset Count": 1
                    }
                },
                {
                    "$lookup": {
                        "from": "OLT-STATE",
                        "let": {"oltId": olt_id},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$oltId"]}}},
                            {"$project": {"_id": 1, "ONU States": 1}}
                        ],
                        "as": "OLT-STATE"
                    }
                },
                {
                    "$lookup": {
                        "from": "ONU-STATE",
                        "let": {"onuId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$onuId"]}}},
                            {"$project": {"_id": 1, "CNTL.MAC Address": 1, "CNTL.Version": 1, "OLT.MAC Address": 1,
                                          "ONU.Reset Count": 1,
                                          "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                                          "Alarm.4-WARNING": 1}}
                        ],
                        "as": "ONU-STATE"
                    }
                },
                {
                    "$lookup": {
                        "from": "ONU-ALARM-HIST-STATE",
                        "let": {"onuId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$onuId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }
                        ], "as": "ONU-ALARM-HIST-STATE"
                    }
                },
                {
                    '$lookup': {
                        'from': 'CNTL-AUTO-STATE',
                        "let": {"stateId": {'$arrayElemAt': ["$ONU-STATE", 0]}},
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$stateId.CNTL.MAC Address'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Enable': '$AUTO.Enable'
                                }
                            }
                        ],
                        'as': 'CNTL-AUTO'
                    }
                },
                {
                    '$lookup': {
                        'from': 'AUTO-STATE',
                        'as': 'AUTO-Time',
                        'pipeline': [
                            {
                                '$project': {
                                    'Time': 1
                                }
                            }
                        ]
                    }
                },
                {
                    '$lookup': {
                        'from': 'OLT-AUTO-STATE',
                        "let": {"stateId": {'$arrayElemAt': ["$ONU-STATE", 0]}},
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$stateId.OLT.MAC Address'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Enable': '$AUTO.Enable'
                                }
                            }
                        ],
                        'as': 'OLT-AUTO'
                    }
                },
                {
                    '$lookup': {
                        'from': 'ONU-AUTO-STATE',
                        "let": {"stateId": "$_id"},
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$stateId'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Enable': '$AUTO.Enable'
                                }
                            }
                        ],
                        'as': 'ONU-AUTO'
                    }
                }
            ]))

            # Special case where OLT went offline and ONU didn't move to another OLT.
            special = list(self.database["OLT-STATE"].aggregate([
                {"$match": {"_id": olt_id}},
                {"$project": {"result": {"$objectToArray": "$ONUs"}, "ONU States": 1, "ONUs": 1}},
                # Get keys/values of the ONUs dictionary. Also get reset count
                {"$project": {"keys": "$result.k", "ONU States": 1, "ONUs": 1}},
                # Separate out the keys (k) which are the ONU ids
                {"$unwind": "$keys"},  # Place in a list to get their states next
                {
                    "$lookup": {
                        "from": "ONU-STATE",
                        "let": {"onuId": "$keys"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$onuId"]}}},
                            {"$project": {"_id": 1, "CNTL.MAC Address": 1, "CNTL.Version": 1, "OLT.MAC Address": 1,
                                          "ONU.Reset Count": 1,
                                          "Alarm.0-EMERG": 1, "Alarm.1-ALERT": 1, "Alarm.2-CRIT": 1, "Alarm.3-ERROR": 1,
                                          "Alarm.4-WARNING": 1}}
                        ], "as": "ONU-STATE"
                    }
                },
                {"$unwind": {"path": "$ONU-STATE", "preserveNullAndEmptyArrays": True}},
                # preserveNullAndEmptyArrays keeps the pre-provisioned ONUs with no state
                {
                    "$lookup": {
                        "from": "ONU-ALARM-HIST-STATE",
                        "let": {"onuId": "$keys"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$onuId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }
                        ], "as": "ONU-ALARM-HIST-STATE"
                    }
                },
                {"$unwind": {"path": "$ONU-ALARM-HIST-STATE", "preserveNullAndEmptyArrays": True}},
                {
                    "$lookup": {
                        "from": "ONU-CFG",
                        "let": {"onuId": "$keys"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$onuId"]}}},
                            {
                                "$project": {
                                    "ONU.Name": 1, "ONU.Reset Count": 1
                                }
                            }
                        ], "as": "ONU-CFG"
                    }
                },
                {"$unwind": {"path": "$ONU-CFG", "preserveNullAndEmptyArrays": True}},
            ]))

            # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
            if onus and len(onus) > 0 and get_nested_value(onus[0],["ONU-STATE", 0, "CNTL", "Version"]) is not None and len(get_nested_value(onus[0],["ONU-STATE", 0, "CNTL", "Version"])) > 2 and int(onus[0]["ONU-STATE"][0]["CNTL"]["Version"][1:2]) >= 3:
                alarm_history_supported = True

            cntl_auto_disabled = True
            olt_auto_disabled = True
            auto_offline = True

            for onu_data in onus:
                # Automation enable for parent devices.
                if cntl_auto_disabled == True and "CNTL-AUTO" in onu_data:
                    if len(onu_data["CNTL-AUTO"]) > 0 and "Enable" in onu_data["CNTL-AUTO"][0]:
                        cntl_auto_disabled = not onu_data["CNTL-AUTO"][0]["Enable"]
                if olt_auto_disabled == True and "OLT-AUTO" in onu_data:
                    if len(onu_data["OLT-AUTO"]) > 0 and "Enable" in onu_data["OLT-AUTO"][0]:
                        olt_auto_disabled = not onu_data["OLT-AUTO"][0]["Enable"]
                if auto_offline == True and "AUTO-Time" in onu_data and len(onu_data["AUTO-Time"]) > 0 and "Time" in \
                        onu_data["AUTO-Time"][0]:
                    auto_offline = not self.automation_time_is_online(onu_data["AUTO-Time"][0]["Time"])

            for onu_data in onus:
                onu_id = onu_data['_id']
                cntl_id = get_nested_value(onu_data, ["ONU-STATE", 0, "CNTL", "MAC Address"], "")
                name = onu_data["ONU"]["Name"]
                system_status = "Offline"
                is_offline = True
                is_resetting = False
                config_reset_count = onu_data["ONU"]["Reset Count"]
                state_reset_count = get_nested_value(onu_data, ["ONU-STATE", 0, "ONU", "Reset Count"], -1)

                # Check that CNTL is online.
                if system_status_doc is not None and "Time" in system_status_doc and "CNTL" in system_status_doc and "Version" in \
                        system_status_doc["CNTL"] and self.controller_time_is_online(system_status_doc["Time"],
                                                                                     system_status_doc["CNTL"][
                                                                                         "Version"]):
                    if olt_id in system_status_doc["System Status"]:
                        if onu_id in system_status_doc["System Status"][olt_id]["ONUs"]:
                            system_status = system_status_doc["System Status"][olt_id]["ONUs"][onu_id]

                            # If ONU Status is Deregistered in CNTL-STATE, we must verify this is correct by checking OLT-STATE
                            if system_status == "Deregistered":
                                olt_state = get_nested_value(onu_data, ["OLT-STATE", 0])
                                if olt_state:
                                    # Setting ONU status based on OLT-State ONU States
                                    if onu_data in olt_state["ONU States"]["Disabled"]:
                                        system_status = "Disabled"
                                    elif onu_data in olt_state["ONU States"]["Disallowed Admin"]:
                                        system_status = "Disallowed Admin"
                                    elif onu_data in olt_state["ONU States"]["Disallowed Error"]:
                                        system_status = "Disallowed Error"
                                    elif onu_data in olt_state["ONU States"]["Dying Gasp"]:
                                        system_status = "Dying Gasp"
                                    elif onu_data in olt_state["ONU States"]["Unprovisioned"]:
                                        system_status = "Unprovisioned"

                            if system_status_doc["System Status"][olt_id]["ONUs"][onu_id] == "Unspecified" or \
                                    system_status_doc["System Status"][olt_id]["ONUs"][onu_id] == "Registered" or \
                                    system_status_doc["System Status"][olt_id]["ONUs"][onu_id] == "FW Upgrade":
                                is_offline = False

                # Check if the ONU is resetting if online
                if not is_offline:
                    if config_reset_count >= 0:
                        is_resetting = state_reset_count != config_reset_count

                    if is_resetting:
                        system_status = "Pending Reset"

                # If the ONU has no ONU-STATE data, the device is Preprovisioned
                onu_state = get_nested_value(onu_data, ["ONU-STATE", 0])
                if onu_state is None or 0:
                        system_status = "Preprovisioned"

                # Get OLT Automation Enable
                onu_auto_disabled = True
                if "ONU-AUTO" in onu_data:
                    if len(onu_data["ONU-AUTO"]) > 0 and "Enable" in onu_data["ONU-AUTO"][0]:
                        onu_auto_disabled = not onu_data["ONU-AUTO"][0]["Enable"]

                if alarm_history_supported:
                    emerg_alarms = []
                    alert_alarms = []
                    crit_alarms = []
                    error_alarms = []
                    warning_alarms = []

                    if 'ONU-ALARM-HIST-STATE' in onu_data and len(onu_data['ONU-ALARM-HIST-STATE']) > 0 and \
                            onu_data['ONU-ALARM-HIST-STATE'][0]['Alarms']:
                        for alarm_history in onu_data['ONU-ALARM-HIST-STATE'][0]['Alarms']:
                            alarm_text = alarm_history['Text']
                            if alarm_history['Severity'] == "0-EMERG":
                                emerg_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "1-ALERT":
                                alert_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "2-CRIT":
                                crit_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "3-ERROR":
                                error_alarms.append(alarm_text)
                            elif alarm_history['Severity'] == "4-WARNING":
                                warning_alarms.append(alarm_text)

                    errors = {
                        "EMERG": emerg_alarms,
                        "ALERT": alert_alarms,
                        "CRIT": crit_alarms,
                        "ERROR": error_alarms
                    }
                    warnings = warning_alarms
                else:
                    errors = {
                        "EMERG": get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "0-EMERG"], []),
                        "ALERT": get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "1-ALERT"], []),
                        "CRIT": get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "2-CRIT"], []),
                        "ERROR": get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "3-ERROR"], [])
                    }
                    warnings = get_nested_value(onu_data, ["ONU-STATE", 0, "Alarm", "4-WARNING"], [])

                response[1].append({"_id": onu_id,
                                    "cntl": cntl_id,
                                    "name": name,
                                    "offline": is_offline,
                                    "auto_offline": auto_offline,
                                    "cntl_auto_disabled": cntl_auto_disabled,
                                    "olt_auto_disabled": olt_auto_disabled,
                                    "onu_auto_disabled": onu_auto_disabled,
                                    "state": system_status,
                                    "errors": errors,
                                    "warnings": warnings})

            # If using PON Controller major release version of 3 or greater than ALARM-HIST-STATE is supported
            if special and len(special) > 0 and get_nested_value(special[0],
                                                                 ["ONU-STATE", "CNTL", "Version"]) is not None and \
                    int(special[0]["ONU-STATE"]["CNTL"]["Version"][1:2]) >= 3:
                alarm_history_supported = True

            for onu_data in special:
                onu_id = onu_data['keys']
                cntl_id = get_nested_value(onu_data, ["ONU-STATE", "CNTL", "MAC Address"], "")
                name = get_nested_value(onu_data, ["ONU-CFG", "ONU", "Name"], "")
                system_status = "Offline"
                is_offline = True
                unique = True

                for item in response[1]:
                    if "_id" in item and onu_id == item["_id"]:
                        unique = False
                if unique:
                    alarm_history_supported = True

                    if alarm_history_supported:
                        emerg_alarms = []
                        alert_alarms = []
                        crit_alarms = []
                        error_alarms = []
                        warning_alarms = []

                        if 'ONU-ALARM-HIST-STATE' in onu_data and len(onu_data['ONU-ALARM-HIST-STATE']) > 0 and \
                                onu_data['ONU-ALARM-HIST-STATE']['Alarms']:
                            for alarm_history in onu_data['ONU-ALARM-HIST-STATE']['Alarms']:
                                alarm_text = alarm_history['Text']
                                if alarm_history['Severity'] == "0-EMERG":
                                    emerg_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "1-ALERT":
                                    alert_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "2-CRIT":
                                    crit_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "3-ERROR":
                                    error_alarms.append(alarm_text)
                                elif alarm_history['Severity'] == "4-WARNING":
                                    warning_alarms.append(alarm_text)

                        errors = {
                            "EMERG": emerg_alarms,
                            "ALERT": alert_alarms,
                            "CRIT": crit_alarms,
                            "ERROR": error_alarms
                        }
                        warnings = warning_alarms
                    else:
                        errors = {
                            "EMERG": get_nested_value(onu_data, ["ONU-STATE", "Alarm", "0-EMERG"], []),
                            "ALERT": get_nested_value(onu_data, ["ONU-STATE", "Alarm", "1-ALERT"], []),
                            "CRIT": get_nested_value(onu_data, ["ONU-STATE", "Alarm", "2-CRIT"], []),
                            "ERROR": get_nested_value(onu_data, ["ONU-STATE", "Alarm", "3-ERROR"], [])
                        }
                        warnings = get_nested_value(onu_data, ["ONU-STATE", "Alarm", "4-WARNING"], [])

                    response[1].append({"_id": onu_id,
                                        "cntl": cntl_id,
                                        "name": name,
                                        "offline": is_offline,
                                        "state": system_status,
                                        "errors": errors,
                                        "warnings": warnings})

            val_array = response[1]
            response[1] = sorted(val_array, key=lambda i: (i['name'].lower(), i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_onus_under_olt_for_tree)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs for OLT {olt_id} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-STATE, ONU-CFG",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_configs_for_olt(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            olt_state_coll = self.database["OLT-STATE"]
            olt = olt_state_coll.find_one({"_id": olt_id}, {"ONUs": 1})

            if olt is not None and "ONUs" in olt:
                onu_list = list(olt["ONUs"])
                if onu_list and len(onu_list) > 0:
                    onu_cfg_coll = self.database["ONU-CFG"]
                    response[1] = onu_cfg_coll.find({"_id": {"$in": onu_list}})
                    response[1] = sorted(response[1], key=lambda i: (i['ONU']['Name'].lower(), i['_id']))

        except Exception as err:
            self.status_log("EXCEPTION (get_onu_configs_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs for OLT {olt_id} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-CFG",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_states_for_olt(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get all distinct document ids from collection OLT_STATE
            olt_state_coll = self.database["OLT-STATE"]
            olt = olt_state_coll.find_one({"_id": olt_id}, {"ONUs": 1})
            onu_state_coll = self.database["ONU-STATE"]
            if olt is not None and "ONUs" in olt:
                for onu in olt["ONUs"]:
                    onu_state = onu_state_coll.find_one({"_id": onu})
                    if onu_state is not None:
                        response[1].append(onu_state)

            response[1] = sorted(response[1], key=lambda i: (i['_id']))
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_configs_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs for OLT {olt_id} list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-STATE",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def search_tree_device(self, user_email, device_type, filter_str):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            if device_type == "Controller":
                collection = self.database.get_collection("CNTL-CFG")
                controllers = collection.find({"$or": [
                    {"_id": {"$regex": filter_str, "$options": 'i'}},
                    {"CNTL.Name": {"$regex": filter_str, "$options": 'i'}}
                ]},
                    {"_id": 1, "CNTL.Name": 1}
                )

                for controller in controllers:
                    response[1].append({"_id": controller["_id"], "name": controller["CNTL"]["Name"]})
            elif device_type == 'Switch':
                collection = self.database.get_collection("SWI-CFG")
                switches = collection.find({"$or": [
                    {"_id": {"$regex": filter_str, "$options": 'i'}},
                    {"SWI.Name": {"$regex": filter_str, "$options": 'i'}}
                ]},
                    {"_id": 1, "SWI.Name": 1}
                )

                for switch in switches:
                    response[1].append({"_id": switch["_id"], "name": switch["SWI"]["Name"]})
            elif device_type == 'OLT':
                collection = self.database.get_collection("OLT-CFG")
                olts = collection.find({"$or": [
                    {"_id": {"$regex": filter_str, "$options": 'i'}},
                    {"OLT.Name": {"$regex": filter_str, "$options": 'i'}}
                ]},
                    {"_id": 1, "OLT.Name": 1}
                )

                for olt in olts:
                    response[1].append({"_id": olt["_id"], "name": olt["OLT"]["Name"]})
            elif device_type == 'Port':
                collection = self.database.get_collection("OLT-STATE")
                olts = collection.find({"Switch.Port ID": {"$regex": filter_str, "$options": 'i'}},
                                       {"_id": 1, "Switch.Port ID": 1}
                                       )

                for olt in olts:
                    response[1].append({"_id": olt["_id"], "name": olt["Switch"]["Port ID"]})
            elif device_type == 'ONU':
                collection = self.database.get_collection("ONU-CFG")
                onus = collection.find({"$or": [
                    {"_id": {"$regex": filter_str, "$options": 'i'}},
                    {"ONU.Name": {"$regex": filter_str, "$options": 'i'}}
                ]},
                    {"_id": 1, "ONU.Name": 1}
                )

                for onu in onus:
                    response[1].append({"_id": onu["_id"], "name": onu["ONU"]["Name"]})
        except Exception as err:
            self.status_log("EXCEPTION (search_tree_device)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get filtered tree device list due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-STATE, SWI-CFG, CNTL-STATE",
                       "returned": response[1],
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all states
    """

    # Retrieve an array of all active CNTL states
    def get_all_cntl_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            cntls_cursor = collection.find()
            for state in cntls_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active OLT states
    def get_all_olt_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            olts_cursor = collection.find()
            for state in olts_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all active ONU states
    def get_all_onu_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            onus_cursor = collection.find()
            for state in onus_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all alarms
    """

    # Retrieve and array of all controller alarms
    def get_all_cntl_alarms(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            response[1] = list(collection.aggregate([
                {
                    "$lookup": {
                        "from": "CNTL-ALARM-HIST-STATE",
                        "let": {"cntlId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$cntlId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }, {
                                '$unwind': {
                                    'path': '$Alarms',
                                    'preserveNullAndEmptyArrays': False
                                }
                            }, {
                                '$group': {
                                    '_id': '_id',
                                    'Alarms': {
                                        '$push': {
                                            'Text': '$Alarms.Text',
                                            'Severity': '$Alarms.Severity',
                                            'Source': '$Alarms.Source',
                                            'Object Type': '$Alarms.Object Type',
                                            'Alarm Type': '$Alarms.Alarm Type'
                                        }
                                    }
                                }
                            }
                        ],
                        "as": "alarmHistory"
                    }
                },
                {
                    "$addFields": {
                        "majorVersion": {
                            '$cond': {
                                'if': {
                                    '$gte': [
                                        {
                                            '$strLenCP': '$CNTL.Version'
                                        }, 2
                                    ]
                                },
                                'then': {
                                    '$toInt': {
                                        '$substr': [
                                            '$CNTL.Version', 1, 1
                                        ]
                                    }
                                },
                                'else': 1
                            }
                        },
                        "CNTL-ALARM-HIST-STATE": {
                            "$arrayElemAt": ["$alarmHistory.Alarms", 0]
                        }
                    }
                },
                {
                    "$project": {
                        "Time": 1,
                        "majorVersion": 1,
                        "Alarm": 1,
                        "CNTL-ALARM-HIST-STATE": 1
                    }
                },
                {
                    "$project": {
                        "CNTL.MAC Address": 1,
                        "OLT.MAC Address": 1,
                        "Time": 1,
                        "Alarm": {
                            "$cond": {
                                "if": {"$gte": ["$majorVersion", 3]},
                                "then": "$$REMOVE",
                                "else": "$Alarm"
                            }
                        },
                        "CNTL-ALARM-HIST-STATE": {
                            "$cond": {
                                "if": {"$gte": ["$majorVersion", 3]},
                                "then": "$CNTL-ALARM-HIST-STATE",
                                "else": "$$REMOVE"
                            }
                        }
                    }
                }
            ]))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller alarms due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve and array of all OLT alarms
    def get_all_olt_alarms(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            response[1] = list(collection.aggregate([
                {
                    "$lookup": {
                        "from": "OLT-ALARM-HIST-STATE",
                        "let": {"oltId": "$_id"},
                        "pipeline": [
                            {"$match": {"$expr": {"$eq": ["$_id", "$$oltId"]}}},
                            {
                                "$project": {
                                    "Alarms": {
                                        "$filter": {
                                            "input": "$Alarms",
                                            "cond": {"$and": [{"$eq": ["$$this.Active State", True]},
                                                              {"$eq": ["$$this.Ack State", False]}]}
                                        }
                                    }
                                }
                            }, {
                                '$unwind': {
                                    'path': '$Alarms',
                                    'preserveNullAndEmptyArrays': False
                                }
                            }, {
                                '$group': {
                                    '_id': '_id',
                                    'Alarms': {
                                        '$push': {
                                            'Text': '$Alarms.Text',
                                            'Severity': '$Alarms.Severity',
                                            'Source': '$Alarms.Source',
                                            'Object Type': '$Alarms.Object Type',
                                            'Alarm Type': '$Alarms.Alarm Type'
                                        }
                                    }
                                }
                            }
                        ],
                        "as": "alarmHistory"
                    }
                },
                {
                    "$addFields": {
                        "majorVersion": {
                            '$cond': {
                                'if': {
                                    '$gte': [
                                        {
                                            '$strLenCP': '$CNTL.Version'
                                        }, 2
                                    ]
                                },
                                'then': {
                                    '$toInt': {
                                        '$substr': [
                                            '$CNTL.Version', 1, 1
                                        ]
                                    }
                                },
                                'else': 1
                            }
                        },
                        "OLT-ALARM-HIST-STATE": {
                            "$arrayElemAt": ["$alarmHistory.Alarms", 0]
                        }
                    }
                },
                {
                    "$project": {
                        "CNTL.MAC Address": 1,
                        "OLT.MAC Address": 1,
                        "Time": 1,
                        "majorVersion": 1,
                        "Alarm": 1,
                        "OLT-ALARM-HIST-STATE": 1
                    }
                },
                {
                    "$project": {
                        "CNTL.MAC Address": 1,
                        "Time": 1,
                        "Alarm": {
                            "$cond": {
                                "if": {"$gte": ["$majorVersion", 3]},
                                "then": "$$REMOVE",
                                "else": "$Alarm"
                            }
                        },
                        "OLT-ALARM-HIST-STATE": {
                            "$cond": {
                                "if": {"$gte": ["$majorVersion", 3]},
                                "then": "$OLT-ALARM-HIST-STATE",
                                "else": "$$REMOVE"
                            }
                        }
                    }
                }
            ]))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT alarms due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve and array of all ONU alarms
    def get_all_onu_alarms(self, user_email, attribute):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database.get_collection("ONU-STATE")
            response[1] = list(collection.aggregate([
                {
                    '$lookup': {
                        'from': 'ONU-ALARM-HIST-STATE',
                        'let': {
                            'onuId': '$_id'
                        },
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$onuId'
                                        ]
                                    }
                                }
                            }, {
                                '$project': {
                                    'Alarms': {
                                        '$filter': {
                                            'input': '$Alarms',
                                            'cond': {
                                                '$and': [
                                                    {
                                                        '$eq': [
                                                            '$$this.Active State', True
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$$this.Ack State', False
                                                        ]
                                                    }
                                                ]
                                            }
                                        }
                                    }
                                }
                            }, {
                                '$unwind': {
                                    'path': '$Alarms',
                                    'preserveNullAndEmptyArrays': False
                                }
                            }, {
                                '$group': {
                                    '_id': '_id',
                                    'Alarms': {
                                        '$push': {
                                            'Text': '$Alarms.Text',
                                            'Severity': '$Alarms.Severity',
                                            'Source': '$Alarms.Source',
                                            'Object Type': '$Alarms.Object Type',
                                            'Alarm Type': '$Alarms.Alarm Type'
                                        }
                                    }
                                }
                            }
                        ],
                        'as': 'alarmHistory'
                    }
                }, {
                    '$addFields': {
                        'majorVersion': {
                            '$cond': {
                                'if': {
                                    '$gte': [
                                        {
                                            '$strLenCP': '$CNTL.Version'
                                        }, 2
                                    ]
                                },
                                'then': {
                                    '$toInt': {
                                        '$substr': [
                                            '$CNTL.Version', 1, 1
                                        ]
                                    }
                                },
                                'else': 1
                            }
                        },
                        'ONU-ALARM-HIST-STATE': {
                            '$arrayElemAt': [
                                '$alarmHistory.Alarms', 0
                            ]
                        }
                    }
                }, {
                    '$project': {
                        'CNTL.MAC Address': 1,
                        'OLT.MAC Address': 1,
                        'Time': 1,
                        'majorVersion': 1,
                        'Alarm': 1,
                        'ONU-ALARM-HIST-STATE': 1
                    }
                }, {
                    '$project': {
                        'CNTL.MAC Address': 1,
                        'OLT.MAC Address': 1,
                        'Time': 1,
                        'Alarm': {
                            '$cond': {
                                'if': {
                                    '$gte': [
                                        '$majorVersion', 3
                                    ]
                                },
                                'then': '$$REMOVE',
                                'else': '$Alarm'
                            }
                        },
                        'ONU-ALARM-HIST-STATE': {
                            '$cond': {
                                'if': {
                                    '$gte': [
                                        '$majorVersion', 3
                                    ]
                                },
                                'then': '$ONU-ALARM-HIST-STATE',
                                'else': '$$REMOVE'
                            }
                        }
                    }
                }
            ]))
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_alarms)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU alarms due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all ONU states for a specified OLT
    """

    def get_all_onu_states_for_olt(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            query = {"OLT.MAC Address": olt_id}
            onu_states_cursor = collection.find(query)
            for state in onu_states_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_states_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU states for OLT {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Allowing ONU Registration
    """

    # Checking if ONU is disallowed on any OLT AND if there is a different Reg Allow ONU pending on the given OLT
    def get_onu_registration_allow(self, onu_id, user_email):
        response = [status.HTTP_200_OK, {"is_disallowed": False, "can_allow": True}]

        try:
            database_manager.get_database(self.database_id)
            # Retrieve all OLT states where the given ONU is disallowed
            olt_state_collection = self.database["OLT-STATE"]
            olt_states_cursor = olt_state_collection.find({"ONU States.Disallowed Error": onu_id},
                                                          {"_id": 1, "OLT.Reg Allow ONU": 1, "OLT.Reg Allow Count": 1})
            # If the ONU is disallowed in 1 or more OLTs, then continue
            if olt_states_cursor.count() > 0:
                response[1]['is_disallowed'] = True
                olt_config_collection = self.database["OLT-CFG"]

                # Check if the OLT the ONU is disallowed on is currently in the process of allowing an ONU already
                for olt in olt_states_cursor:
                    olt_cfg = olt_config_collection.find_one({"_id": olt["_id"]}, {"_id": 1, "OLT.Reg Allow ONU": 1,
                                                                                   "OLT.Reg Allow Count": 1})
                    if olt_cfg is not None:
                        if not (olt_cfg["OLT"]["Reg Allow ONU"] == olt["OLT"]["Reg Allow ONU"] and olt_cfg["OLT"][
                            "Reg Allow Count"] == olt["OLT"]["Reg Allow Count"]):
                            response[1]['can_allow'] = False

        except Exception as err:
            self.status_log("EXCEPTION (get_onu_registration_allow)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLTs {onu_id} is disallowed on due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Checking if ONU is disallowed on any OLT AND if there is a different Reg Allow ONU pending on the given OLT
    def put_onu_registration_allow(self, onu_id, user_email):
        response = [status.HTTP_200_OK, [], None, {}]

        try:
            database_manager.get_database(self.database_id)
            # Retrieve all OLT states where the given ONU is disallowed
            olt_state_collection = self.database["OLT-STATE"]
            olt_states_cursor = olt_state_collection.find({"ONU States.Disallowed Error": onu_id},
                                                          {"_id": 1, "OLT.Reg Allow ONU": 1, "OLT.Reg Allow Count": 1})
            # If the ONU is disallowed in 1 or more OLTs, then continue
            if olt_states_cursor.count() > 0:
                olt_config_collection = self.database["OLT-CFG"]

                new_doc = {}
                # Update OLT config reg allow ONU with onu mac and increment Reg Allow count, if there is a config file to update
                for olt in olt_states_cursor:
                    update_document = {"$set": {"OLT.Reg Allow ONU": onu_id},
                                       "$inc": {"OLT.Reg Allow Count": 1, "OLT.CFG Change Count": 1}}
                    olt_config_collection.update_one({"_id": olt["_id"]}, update_document)
                    new_doc[olt["_id"]] = update_document
                response[2] = new_doc
        except Exception as err:
            self.status_log("EXCEPTION (put_onu_registration_allow)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not clear OLTs {onu_id} is disallowed on due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-CONFIG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get state of the specified device
    """

    # Retrieve the state of the specified CNTL
    def get_cntl_state(self, controller_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            response[1] = collection.find_one({"_id": controller_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the auth state of the specified CNTL
    def get_cntl_auth_state(self, controller_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            if "CNTL-AUTH-STATE" in self.database.collection_names():
                collection = self.database["CNTL-AUTH-STATE"]
                response[1] = collection.find_one({"_id": controller_id})
            else:
                response[1] = None
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_auth_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} authentication state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-AUTH-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the state of the specified Switch
    def get_switch_state(self, switch_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            cursor = collection.find({"Switch.Chassis ID": switch_id})
            if cursor.count() > 0:
                response[1] = cursor.sort("Time", pymongo.DESCENDING)[0]
        except Exception as err:
            self.status_log("EXCEPTION (get_switch_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switch {switch_id} state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the state of the specified OLT
    def get_olt_state(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            response[1] = collection.find_one({"_id": olt_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the state of the specified ONU
    def get_onu_state(self, onu_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            response[1] = collection.find_one({"_id": onu_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all configs
    """

    # Retrieve all controller configs
    def get_all_cntl_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            query = {"_id": {'$regex': '^((?!Default).)*$'}}
            cntl_cfgs_cursor = collection.find(query)
            for cfg in cntl_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all olt configs
    def get_all_olt_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            query = {"_id": {'$regex': '^((?!Default).)*$'}}
            collection = self.database["OLT-CFG"]
            olt_cfgs_cursor = collection.find(query)
            for cfg in olt_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all onu configs
    def get_all_onu_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            query = {"_id": {'$regex': '^((?!Default).)*$'}}
            collection = self.database["ONU-CFG"]
            onu_cfgs_cursor = collection.find(query)
            for cfg in onu_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get configuration settings for the specified device
    """

    # Retrieve the config of the specified CNTL
    def get_cntl_cfg(self, controller_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            response[1] = collection.find_one({"_id": controller_id})

            # Return 404 if configuration for specified Controller is not found
            if response[1] is None:
                response = [status.HTTP_404_NOT_FOUND,
                            f"Configuration document for PON Controller {controller_id} was not found.::No documents in CNTL-CFG collection with _id of {controller_id} exist."]
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the config of the specified Switch
    def get_switch_cfg(self, switch_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            response[1] = collection.find_one({"_id": switch_id})

            # Return 404 if configuration for specified Switch is not found
            if response[1] is None:
                response = [status.HTTP_404_NOT_FOUND,
                            f"Configuration document for Switch {switch_id} was not found.::No documents in SWI-CFG collection with _id of {switch_id} exist."]
        except Exception as err:
            self.status_log("EXCEPTION (get_switch_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Switch {switch_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the config of the specified OLT
    def get_olt_cfg(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-CFG"]
            response[1] = collection.find_one({"_id": olt_id})

            # Return 404 if configuration for specified OLT is not found
            if response[1] is None:
                response = [status.HTTP_404_NOT_FOUND,
                            f"Configuration document for OLT {olt_id} was not found.::No documents in OLT-CFG collection with _id of {olt_id} exist."]
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the config of the specified ONU
    def get_onu_cfg(self, onu_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get ONU with '_id'=mac_address from collection response[1]
            collection = self.database["ONU-CFG"]
            query = {"_id": onu_id}
            response[1] = collection.find_one(query)

            # Return 404 if configuration for specified ONU is not found
            if response[1] is None:
                response = [status.HTTP_404_NOT_FOUND,
                            f"Configuration document for ONU {onu_id} was not found.::No documents in ONU-CFG collection with _id of {onu_id} exist."]
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get output logs for the specified device
    """

    # Retrieve the logs of the specified CNTL since the specified time
    def get_cntl_log(self, controller_id, since_utc_time, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection CNTL-LOG
            collection_name = "SYSLOG-CNTL-{}".format(controller_id.replace(":", ""))
            collection = self.database[collection_name]
            cntl_log_cursor = collection.find({"time": {"$gt": since_utc_time}, "device ID": controller_id},
                                              {"_id": 0, "device ID": 0}).limit(10000)
            for log in cntl_log_cursor:
                response[1].append(log)
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_log)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-CNTL-{}".format(controller_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the logs of the specified OLT since the specified time
    def get_olt_log(self, olt_id, since_utc_time, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection OLT-LOG
            collection_name = "SYSLOG-OLT-{}".format(olt_id.replace(":", ""))
            collection = self.database[collection_name]
            olt_log_cursor = collection.find({"time": {"$gt": since_utc_time}, "device ID": olt_id},
                                             {"_id": 0, "device ID": 0}).limit(10000)
            for log in olt_log_cursor:
                response[1].append(log)
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_log)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-OLT-{}".format(olt_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the logs of the specified ONU since the specified time
    def get_onu_log(self, onu_id, since_utc_time, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection ONU-LOG
            collection_name = "SYSLOG-ONU-{}".format(onu_id.replace(":", ""))
            collection = self.database[collection_name]
            onu_log_cursor = collection.find({"time": {"$gt": since_utc_time}, "device ID": onu_id},
                                             {"_id": 0, "device ID": 0}).limit(10000)
            for log in onu_log_cursor:
                response[1].append(log)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_log)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-ONU-{}".format(onu_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get output stats for the specified device
    """

    # Retrieve the stats of the specified CNTL
    def get_cntl_stats(self, controller_id, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection CNTL_<mac_address>_STATS
            collection_name = "STATS-CNTL-{}".format(controller_id.replace(":", ""))
            collection = self.database[collection_name]
            if to_utc_time:
                cntl_stats_cursor = collection.find({"Time": {"$gte": since_utc_time, "$lte": to_utc_time}}).limit(
                    10000)
            else:
                cntl_stats_cursor = collection.find({"Time": {"$gt": since_utc_time}}).limit(10000)
            for stats in cntl_stats_cursor:
                stats["mac_address"] = controller_id
                response[1].append(stats)
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {controller_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-CNTL-{}".format(controller_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the stats of the specified OLT
    def get_olt_stats(self, olt_id, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get logs from collection OLT_<mac_address>_STATS
            collection_name = "STATS-OLT-{}".format(olt_id.replace(":", ""))
            collection = self.database[collection_name]
            if to_utc_time:
                olt_stats_cursor = collection.find({"Time": {"$gte": since_utc_time, "$lte": to_utc_time}}).limit(10000)
            else:
                olt_stats_cursor = collection.find({"Time": {"$gt": since_utc_time}}).limit(10000)
            for stats in olt_stats_cursor:
                stats["mac_address"] = olt_id
                response[1].append(stats)
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {olt_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-OLT-{}".format(olt_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve the stats of the specified ONU
    def get_onu_stats(self, onu_id, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            # Get stats from collection ONU_<mac_address>_STATS
            collection_name = "STATS-ONU-{}".format(onu_id.replace(":", ""))
            collection = self.database[collection_name]
            if to_utc_time:
                onu_stats_cursor = collection.find({"Time": {"$gte": since_utc_time, "$lte": to_utc_time}}).limit(10000)
            else:
                onu_stats_cursor = collection.find({"Time": {"$gt": since_utc_time}}).limit(10000)
            for stats in onu_stats_cursor:
                stats["mac_address"] = onu_id
                response[1].append(stats)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU-{}".format(onu_id.replace(":", "")), "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_many_onu_stats(self, onus, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)

            # Angular httpclient passes arrays back as strings so this puts them back into a list
            onu_ids = onus.split(",")

            collection = self.database["STATS-ONU"]
            if to_utc_time:
                response[1] = list(collection.find({
                    "$and": [
                        {"device ID": {"$in": onu_ids}},
                        {"valid": True},
                        {"Time": {"$gte": since_utc_time, "$lte": to_utc_time}},
                    ]
                }).limit(10000))
            else:
                response[1] = list(collection.find({
                    "$and": [
                        {"device ID": {"$in": onu_ids}},
                        {"valid": True},
                        {"Time": {"$gte": since_utc_time}},
                    ]
                }).limit(10000))

        except Exception as err:
            self.status_log("EXCEPTION (get_many_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_many_distinct_onu_stats(self, onus, stats, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)

            # Angular httpclient passes arrays back as strings so this puts them back into a list
            onu_ids = onus.split(",")
            selected_stats = stats.split(",")
            stat_dict = {
                "Time": 1,
                "device ID": 1
            }
            for stat in selected_stats:
                stat_dict.update({stat : 1})

            collection = self.database["STATS-ONU"]
            if to_utc_time:
                response[1] = list(collection.aggregate([
                    {
                        "$match": {
                            "$and": [
                                {"device ID": {"$in": onu_ids}},
                                {"valid": True},
                                {"Time": {"$gte": since_utc_time, "$lte": to_utc_time}},
                            ]
                        }
                    },
                    {"$project": stat_dict}
                ]))
            else:
                response[1] = list(collection.aggregate([
                    {
                        "$match": {
                            "$and": [
                                {"device ID": {"$in": onu_ids}},
                                {"valid": True},
                                {"Time": {"$gte": since_utc_time}},
                            ]
                        }
                    },
                    {"$project": stat_dict}
                ]))

        except Exception as err:
            self.status_log("EXCEPTION (get_many_distinct_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onus} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    def get_onu_stats_totals(self, onus, user_email, since_utc_time, to_utc_time=None):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            statDocs = []
            allGroups = {}
            ommitSum = ['asic', 'laser', 'xcvr', 'rx optical level idle', 'tx optical level', 'equalization delay',
                        'fiber distance', 'one way delay'];

            # Angular httpclient passes arrays back as strings so this puts them back into a list
            onu_ids = onus.split(",")

            collection = self.database["STATS-ONU"]
            if to_utc_time:
                statDocs = list(collection.find({
                    "$and": [
                        {"device ID": {"$in": onu_ids}},
                        {"valid": True},
                        {"Time": {"$gte": since_utc_time, "$lte": to_utc_time}},
                    ]
                }).limit(10000))
            else:
                statDocs = list(collection.find({
                    "$and": [
                        {"device ID": {"$in": onu_ids}},
                        {"valid": True},
                        {"Time": {"$gte": since_utc_time}},
                    ]
                }).limit(10000))

            # Calculate the totals for each stat
            for entry in statDocs:
                groups = entry.keys()
                allGroups['Time'] = entry['Time']
                for group in groups:
                    if group != 'Automation' and type(entry[group]) is dict and len(entry[group].keys()) > 0:
                        if group not in allGroups:
                            allGroups[group] = {}
                        for k, v in entry[group].items():
                            if k != "control_block" and k != "Learned Addresses":
                                if k in allGroups[group] and k.lower() not in ommitSum:
                                    value = allGroups[group][k]
                                    value += v
                                    allGroups[group][k] = value
                                else:
                                    allGroups[group].update({k: v})

            response[1] = allGroups

        except Exception as err:
            self.status_log("EXCEPTION (get_onu_stats_totals)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all output stats
    """

    # Retrieve all olt stats
    def get_all_olt_stats(self, since_utc_time, user_email):
        response = [status.HTTP_200_OK, []]
        olt_macs = self.get_all_olts(user_email)[1]

        try:
            database_manager.get_database(self.database_id)
            for mac in olt_macs:
                olt_id = mac["_id"]
                response[1][olt_id] = []
                collection_name = "STATS-OLT-{}".format(olt_id.replace(":", ""))
                collection = self.database[collection_name]
                olt_stats_cursor = collection.find({"Time": {"$gt": since_utc_time}}).limit(10000)
                for document in olt_stats_cursor:
                    response[1][olt_id].append(document)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-OLT-<ALL>", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all onu stats
    def get_all_onu_stats(self, since_utc_time, user_email):
        response = [status.HTTP_200_OK, []]
        onu_macs = self.get_all_onus(user_email)[1]

        try:
            database_manager.get_database(self.database_id)
            for mac in onu_macs:
                response[1][mac] = []
                collection_name = "STATS-ONU-{}".format(mac.replace(":", ""))
                collection = self.database[collection_name]
                onu_stats_cursor = collection.find({"Time": {"$gt": since_utc_time}}).limit(10000)
                for document in onu_stats_cursor:
                    response[1][mac].append(document)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_stats)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU statistics due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "STATS-ONU-<ALL>", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get alarms Configurations
    """

    def get_cntl_alarms_cfg(self, alarm_cfg_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-ALARM-CFG"]
            response[1] = collection.find_one({"_id": alarm_cfg_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller {alarm_cfg_id} alarm configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olt_alarms_cfg(self, alarm_cfg_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-ALARM-CFG"]
            response[1] = collection.find_one({"_id": alarm_cfg_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT {alarm_cfg_id} alarm configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_alarms_cfg(self, alarm_cfg_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-ALARM-CFG"]
            response[1] = collection.find_one({"_id": alarm_cfg_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {alarm_cfg_id} alarm configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get all alarm configs
    """

    # Retrieve all controller alarm configs
    def get_all_cntl_alarms_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-ALARM-CFG"]
            cntl_alarms_cfgs_cursor = collection.find()
            for cfg in cntl_alarms_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all controller alarm config identifiers
    def get_all_cntl_alarms_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-ALARM-CFG"]
            cntl_alarms_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in cntl_alarms_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all olt alarm configs
    def get_all_olt_alarms_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-ALARM-CFG"]
            olt_alarms_cfgs_cursor = collection.find()
            for cfg in olt_alarms_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all OLT alarm config identifiers
    def get_all_olt_alarms_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-ALARM-CFG"]
            olt_alarms_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in olt_alarms_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all onu alarm configs
    def get_all_onu_alarms_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-ALARM-CFG"]
            onu_alarms_cfgs_cursor = collection.find()
            for cfg in onu_alarms_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all ONU alarm config identifiers
    def get_all_onu_alarms_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-ALARM-CFG"]
            onu_alarms_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in onu_alarms_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_alarms_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU alarm configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-ALARM-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Put device configuration settings
    """

    # Put the new configuration for the specified CNTL
    def put_cntl_cfg(self, controller_id, config, user_email):
        response = [status.HTTP_200_OK, f"PON Controller {controller_id} configuration was updated."]

        try:
            database_manager.get_database(self.database_id)
            # Replace the current CFG if present
            collection = self.database["CNTL-CFG"]
            old = collection.find_one({"_id": controller_id})
            config['data']['cntlConfig']['CNTL']['CFG Change Count'] += 1
            inserted = collection.replace_one({"_id": controller_id}, config['data']['cntlConfig'], upsert=True)

            if inserted.upserted_id == controller_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update PON Controller {controller_id} configuration due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_cntl_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update PON Controller {controller_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "CNTL-CFG",
                       "old": old, "new": config['data']['cntlConfig'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Update all Management LAN Names
    def update_mgmt_lan_names(self, cntl_id, new_name, user_email):
        old = ""
        response = [status.HTTP_200_OK, f"PON Controller Management LAN names were updated."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-STATE"]
            doc = collection.find_one({"_id": cntl_id})
            # doc will be None if this is a pre-provisioned device
            if doc is not None:
                old = doc["MGMT LAN"]["Name"]
                devices = doc["MGMT LAN Devices"]
                collection = self.database["CNTL-CFG"]
                for device in devices:
                    if devices[device]["Device Type"] == "Tibit ME":
                        collection.update_one({'_id': device}, {"$set": {"MGMT LAN.Name": new_name},
                                                                "$inc": {"CNTL.CFG Change Count": 1}}, upsert=True)
            else:
                response = [status.HTTP_200_OK,
                            f"No STATE document for this device. Management LAN names cannot be updated."]
        except Exception as err:
            self.status_log("EXCEPTION (update_mgmt_lan_names)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update PON Controller Management LAN names due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "CNTL-STATE, CNTL-CFG",
                       "old": old, "new": new_name, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    # Put the new configuration for the specified Switch
    def put_switch_cfg(self, switch_id, config, user_email):
        response = [status.HTTP_200_OK, f"Switch {switch_id} configuration was updated."]

        try:
            database_manager.get_database(self.database_id)
            # Replace the current CFG if present
            collection = self.database["SWI-CFG"]
            old = collection.find_one({"_id": switch_id})
            config['data']['switchConfig']['SWI']['CFG Change Count'] += 1
            inserted = collection.replace_one({"_id": switch_id}, config['data']['switchConfig'], upsert=True)

            if inserted.upserted_id == switch_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update Switch {switch_id} configuration due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_switch_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update Switch {switch_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "SWI-CFG",
                       "old": old, "new": config['data']['switchConfig'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Put the new configuration for the specified OLT
    def put_olt_cfg(self, olt_id, config, user_email):
        response = [status.HTTP_200_OK, f"OLT {olt_id} configuration was updated."]

        try:
            database_manager.get_database(self.database_id)
            # Replace the current CFG if present
            collection = self.database["OLT-CFG"]
            old = collection.find_one({"_id": olt_id})
            config['data']['oltConfig']['OLT']['CFG Change Count'] += 1
            inserted = collection.replace_one({"_id": olt_id}, config['data']['oltConfig'], upsert=True)

            if inserted.upserted_id == olt_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update OLT {olt_id} configuration due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_olt_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT {olt_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "OLT-CFG",
                       "old": old, "new": config['data']['oltConfig'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Put the new configuration for the specified ONU
    def put_onu_cfg(self, onu_id, config, user_email):
        response = [status.HTTP_200_OK, f"ONU {onu_id} configuration was updated."]

        try:
            database_manager.get_database(self.database_id)
            # Replace the current CFG if present
            collection = self.database["ONU-CFG"]
            old = collection.find_one({"_id": onu_id})
            config['data']['onuConfig']['ONU']['CFG Change Count'] += 1
            inserted = collection.replace_one({"_id": onu_id}, config['data']['onuConfig'], upsert=True)

            if inserted.upserted_id == onu_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update ONU {onu_id} configuration due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_onu_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update ONU {onu_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "ONU-CFG",
                       "old": old, "new": config['data']['onuConfig'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    """
    Delete Device Configuration Settings
    """

    # Delete CNTL configurations
    def delete_cntl_cfg(self, controller_id, user_email):
        response = [status.HTTP_200_OK, f"PON Controller {controller_id} configuration was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            collection.delete_one({"_id": controller_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_cntl_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete PON Controller {controller_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete CNTL configurations
    def delete_switch_cfg(self, switch_id, user_email):
        response = [status.HTTP_200_OK, f"Switch {switch_id} configuration was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            collection.delete_one({"_id": switch_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_switch_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete Switch {switch_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete OLT configurations
    def delete_olt_cfg(self, olt_id, user_email):
        response = [status.HTTP_200_OK, f"OLT {olt_id} configuration was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-CFG"]
            collection.delete_one({"_id": olt_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete OLT {olt_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete ONU configurations
    def delete_onu_cfg(self, onu_id, user_email):
        response = [status.HTTP_200_OK, f"ONU {onu_id} configuration was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            collection.delete_one({"_id": onu_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete ONU {onu_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Put device alarms configuration settings
    """

    # Put the new configuration for the specified CNTL
    def put_cntl_alarms_cfg(self, alarm_cfg_id, config, user_email):
        response = [status.HTTP_200_OK, f"PON Controller alarm configuration {alarm_cfg_id} was updated."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-ALARM-CFG"]
            old = collection.find_one({"_id": alarm_cfg_id})

            if '_id' in config['data']['alarms_cfg']:
                # If alarm_cfg_id and config[_id] don't match, then the user has updated the config to use a new id, alarm_cfg_id. In this case, we must remove the old config with the id of; config['data']['alarms_cfg']['_id']
                if alarm_cfg_id is not config['data']['alarms_cfg']['_id']:
                    collection.delete_one(({"_id": config['data']['alarms_cfg']['_id']}))
                del config['data']['alarms_cfg']['_id']
            inserted = collection.replace_one({"_id": alarm_cfg_id}, config["data"]["alarms_cfg"], upsert=True)

            if inserted.upserted_id == alarm_cfg_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update PON Controller alarm configuration {alarm_cfg_id} due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_cntl_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update PON Controller alarm configuration {alarm_cfg_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "CNTL-ALARM-CFG",
                       "old": old, "new": config['data']['alarms_cfg'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Put the new configuration for the specified OLT
    def put_olt_alarms_cfg(self, alarm_cfg_id, config, user_email):
        response = [status.HTTP_200_OK, f"OLT alarm configuration {alarm_cfg_id} was updated."]

        try:
            database_manager.get_database(self.database_id)
            # Replace the current CFG if present
            collection = self.database["OLT-ALARM-CFG"]
            old = collection.find_one({"_id": alarm_cfg_id})

            if '_id' in config['data']['alarms_cfg']:
                # If alarm_cfg_id and config[_id] don't match, then the user has updated the config to use a new id, alarm_cfg_id.
                # In this case, we must remove the old config with the id of; config['data']['alarms_cfg']['_id']
                if alarm_cfg_id is not config['data']['alarms_cfg']['_id']:
                    collection.delete_one(({"_id": config['data']['alarms_cfg']['_id']}))
                del config['data']['alarms_cfg']['_id']
            inserted = collection.replace_one({"_id": alarm_cfg_id}, config["data"]["alarms_cfg"], upsert=True)

            if inserted.upserted_id == alarm_cfg_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update OLT alarm configuration {alarm_cfg_id} due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_olt_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT alarm configuration {alarm_cfg_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "OLT-ALARM-CFG",
                       "old": old, "new": config['data']['alarms_cfg'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Put the new configuration for the specified ONU
    def put_onu_alarms_cfg(self, alarm_cfg_id, config, user_email):
        response = [status.HTTP_200_OK, f"ONU alarm configuration {alarm_cfg_id} was updated."]

        try:
            database_manager.get_database(self.database_id)
            # Replace the current CFG if present
            collection = self.database["ONU-ALARM-CFG"]
            old = collection.find_one({"_id": alarm_cfg_id})

            if '_id' in config['data']['alarms_cfg']:
                # If alarm_cfg_id and config[_id] don't match, then the user has updated the config to use a new id, alarm_cfg_id.
                # In this case, we must remove the old config with the id of; config['data']['alarms_cfg']['_id']
                if alarm_cfg_id is not config['data']['alarms_cfg']['_id']:
                    collection.delete_one(({"_id": config['data']['alarms_cfg']['_id']}))
                del config['data']['alarms_cfg']['_id']
            inserted = collection.replace_one({"_id": alarm_cfg_id}, config["data"]["alarms_cfg"], upsert=True)

            if inserted.upserted_id == alarm_cfg_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update ONU alarm configuration {alarm_cfg_id} due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_onu_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update ONU alarm configuration {alarm_cfg_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "ONU-ALARM-CFG",
                       "old": old, "new": config['data']['alarms_cfg'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    """
    Delete Device Alarms configuration settings
    """

    # Delete cntl alarm configurations
    def delete_cntl_alarms_cfg(self, alarm_cfg_id, user_email):
        response = [status.HTTP_200_OK, f"PON Controller alarm configuration {alarm_cfg_id} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-ALARM-CFG"]
            collection.delete_one({"_id": alarm_cfg_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_cntl_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete PON Controller alarm configuration {alarm_cfg_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-ALARM-CFG",
                       "deleted": response[0] == 200, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete olt alarm configurations
    def delete_olt_alarms_cfg(self, alarm_cfg_id, user_email):
        response = [status.HTTP_200_OK, f"OLT alarm configuration {alarm_cfg_id} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-ALARM-CFG"]
            collection.delete_one({"_id": alarm_cfg_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete OLT alarm configuration {alarm_cfg_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-ALARM-CFG",
                       "deleted": response[0] == 200, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete onu alarm configurations
    def delete_onu_alarms_cfg(self, alarm_cfg_id, user_email):
        response = [status.HTTP_200_OK, f"ONU alarm configuration {alarm_cfg_id} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-ALARM-CFG"]
            collection.delete_one({"_id": alarm_cfg_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete ONU alarm configuration {alarm_cfg_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-ALARM-CFG",
                       "deleted": response[0] == 200, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Sla configs
    """

    # Retrieve all sla config identifiers
    def get_all_sla_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SLA-CFG"]
            sla_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in sla_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_sla_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all sla configuration ids due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SLA-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # get All sla configurations
    def get_all_sla_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SLA-CFG"]
            sla_cfgs_cursor = collection.find()
            for cfg in sla_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_sla_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all SLA configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SLA-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_sla_cfg(self, sla_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SLA-CFG"]
            response[1] = collection.find_one({"_id": sla_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_alarms_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get SLA {sla_id} configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SLA-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Put the new sla configuration
    def put_sla_cfg(self, sla_id, config, user_email):
        response = [status.HTTP_200_OK, f"SLA configuration {sla_id} was updated."]

        try:
            database_manager.get_database(self.database_id)
            # replace current CFG if present
            collection = self.database["SLA-CFG"]
            old = collection.find_one({"_id": sla_id})
            if '_id' in config['data']['slaConfig']:
                # If sla_id and config[_id] don't match, then the user has updated the config to use a new id, alarm_cfg_id. In this case, we must remove the old config with the id of; config['data']['alarms_cfg']['_id']
                if sla_id is not config['data']['slaConfig']['_id']:
                    collection.delete_one(({"_id": config['data']['slaConfig']['_id']}))
                del config['data']['slaConfig']['_id']
            inserted = collection.replace_one({"_id": sla_id}, config['data']['slaConfig'], upsert=True)

            if inserted.upserted_id == sla_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update SLA configuration {sla_id} due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_sla_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update SLA configuration {sla_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "SLA-CFG",
                       "old": old, "new": config['data']['slaConfig'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Delete sla configurations
    def delete_sla_cfg(self, sla_id, user_email):
        response = [status.HTTP_200_OK, f"SLA configuration {sla_id} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SLA-CFG"]
            collection.delete_one({"_id": sla_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_sla_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete SLA configuration {sla_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SLA-CFG",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Replace sla in onu config with default
    def replace_sla(self, sla_id, user_email):
        response = [status.HTTP_200_OK, f"All ONU configurations using SLA {sla_id} were updated to SLA Min.", None,
                    None]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            collection.update_many({'OLT-Service 0.SLA-CFG': sla_id}, {'$set': {'OLT-Service 0.SLA-CFG': 'Min'}})
            collection.update_many({'OLT-Service 1.SLA-CFG': sla_id}, {'$set': {'OLT-Service 1.SLA-CFG': 'Min'}})
            collection.update_many({'OLT-Service 2.SLA-CFG': sla_id}, {'$set': {'OLT-Service 2.SLA-CFG': 'Min'}})
            collection.update_many({'OLT-Service 3.SLA-CFG': sla_id}, {'$set': {'OLT-Service 3.SLA-CFG': 'Min'}})
            collection.update_many({'OLT-Service 4.SLA-CFG': sla_id}, {'$set': {'OLT-Service 4.SLA-CFG': 'Min'}})
            collection.update_many({'OLT-Service 5.SLA-CFG': sla_id}, {'$set': {'OLT-Service 5.SLA-CFG': 'Min'}})
            collection.update_many({'OLT-Service 6.SLA-CFG': sla_id}, {'$set': {'OLT-Service 6.SLA-CFG': 'Min'}})
            collection.update_many({'OLT-Service 7.SLA-CFG': sla_id}, {'$set': {'OLT-Service 7.SLA-CFG': 'Min'}})
            response[2] = 'Min'
            response[3] = sla_id
        except Exception as err:
            self.status_log("EXCEPTION (replace_sla)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update ONU configurations using SLA {sla_id} due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "ONU-CFG",
                       "old": "Many", "new": "Many", "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    """
    Get Service configs
    """

    # get All service configurations
    def get_all_service_cfgs(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SRV-CFG"]
            service_cfgs_cursor = collection.find().sort("_id", 1)
            for cfg in service_cfgs_cursor:
                response[1].append(cfg)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_service_cfgs)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all Service configurations due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SRV-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all service config identifiers
    def get_all_service_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SRV-CFG"]
            srv_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in srv_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_service_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all service configuration ids due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SRV-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all service config identifiers
    def get_all_downstream_map_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["DS-MAP-CFG"]
            ds_map_cfgs_cursor = collection.find({}, {"_id": 1})
            for cfg in ds_map_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_downstream_map_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all Downstream map configuration ids due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "DS-MAP-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all OMCI service config identifiers
    def get_all_omci_service_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SRV-CFG"]
            srv_cfgs_cursor = collection.find({"OMCI": {'$exists': True}}, {"_id": 1})
            for cfg in srv_cfgs_cursor:
                response[1].append(cfg['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_service_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all omci service configuration ids due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SRV-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Put the new service configuration
    def put_service_cfg(self, srv_id, config, user_email):
        response = [status.HTTP_200_OK, f"Service configuration {srv_id} was updated."]

        try:
            database_manager.get_database(self.database_id)
            # replace current CFG if present
            collection = self.database["SRV-CFG"]
            old = collection.find_one({"_id": srv_id})
            if '_id' in config['data']['serviceConfig']:
                # If srv_id and config[_id] don't match, then the user has updated the config to use a new id, alarm_cfg_id. In this case, we must remove the old config with the id of; config['data']['alarms_cfg']['_id']
                if srv_id is not config['data']['serviceConfig']['_id']:
                    collection.delete_one(({"_id": config['data']['serviceConfig']['_id']}))
                del config['data']['serviceConfig']['_id']
            inserted = collection.replace_one({"_id": srv_id}, config['data']['serviceConfig'], upsert=True)

            if inserted.upserted_id == srv_id:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update Service configuration {srv_id} due to No documents were modified.::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (put_service_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update Service configuration {srv_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "SRV-CFG",
                       "old": old, "new": config['data']['serviceConfig'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Delete service configurations
    def delete_service_cfg(self, srv_id, user_email):
        response = [status.HTTP_200_OK, f"Service configuration {srv_id} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SRV-CFG"]
            collection.delete_one({"_id": srv_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_service_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete Service configuration {srv_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SRV-CFG",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Get a service configuration based on ID
    def get_service_cfg(self, srv_id, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SRV-CFG"]
            response[1] = collection.find_one({"_id": srv_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_service_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get Service configuration {srv_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SRV-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Replace service config in onu config with default
    def replace_srv(self, srv_id, user_email):
        response = [status.HTTP_200_OK,
                    f"All ONU configurations using Service configuration {srv_id} were updated to SLA Min.", None, None]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            collection.update_many(
                {'ONU.SRV-CFG': srv_id},
                {
                    '$set': {'ONU.SRV-CFG': 'DISABLED'},
                    '$inc': {'ONU.CFG Change Count': 1}
                }
            )
            response[2] = "DISABLED"
            response[3] = srv_id
        except Exception as err:
            self.status_log("EXCEPTION (replace_srv)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update ONU configurations using Service configuration {srv_id} due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "ONU-CFG",
                       "Returned": response[1], "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    """
    Uploading Binary Files
    """

    # Put an image into the database
    def upload_image(self, image_dict, user_email):
        response = [status.HTTP_200_OK, f"Picture {image_dict['data']['id']} was uploaded."]

        try:
            database_manager.get_database(self.database_id)
            file_id = os.path.splitext(image_dict['data']['id'])[0]
            file_data = json.dumps(image_dict['data']['val'])
            image_bytes = io.BytesIO()
            with Image.open(io.BytesIO(base64.b64decode(file_data.split(",")[1]))) as image:
                if image.size[1] > 200:
                    file_data = resizeimage.resize_height(image, 200)
                    file_data.save(image_bytes, format='PNG')
                    image_bytes = image_bytes.getvalue()
                else:
                    image_bytes = base64.b64decode(file_data.split(",")[1])

            fs = gridfs.GridFSBucket(db=self.database, bucket_name="PICTURES")
            old = self.database["PICTURES.files"].find_one({"_id": file_id})
            if old is None:
                response[0] = status.HTTP_201_CREATED
                up = fs.open_upload_stream_with_id(file_id=file_id, filename=image_dict['data']['filename'],
                                                   metadata=image_dict['data']['metadata'])
                up.write(image_bytes)
                up.close()
            else:
                fs.delete(file_id=file_id)

                # Remove from chunks collection if necessary
                if self.database['PICTURES.chunks'].find({'files_id': file_id}).count() > 0:
                    self.database['PICTURES.chunks'].delete_many({'files_id': file_id})

                up = fs.open_upload_stream_with_id(file_id=file_id, filename=image_dict['data']['filename'],
                                                   metadata=image_dict['data']['metadata'])
                up.write(image_bytes)
                up.close()

            # Add to defaults
            if image_dict['data']['metadata']["Device Type"] == "CNTL":
                relative_path = "pictures/controllers/"
            elif image_dict['data']['metadata']["Device Type"] == "SWI":
                relative_path = "pictures/switches/"
            elif image_dict['data']['metadata']["Device Type"] == "OLT":
                relative_path = "pictures/olts/"
            elif image_dict['data']['metadata']["Device Type"] == "ONU":
                relative_path = "pictures/onus/"
            else:
                relative_path = "pictures/"

            if not os.path.isfile(self.database_seed_base_dir + relative_path + image_dict['data']['filename']):
                with open(self.database_seed_base_dir + relative_path + image_dict['data']['filename'], 'wb') as file:
                    file.write(image_bytes)

            # Add to metadata
            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
                json_data[relative_path + image_dict['data']['filename']] = image_dict['data']['metadata']
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))
        except Exception as err:
            self.status_log("EXCEPTION (upload_image)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not upload picture {image_dict['data']['id']} due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "PICTURES",
                       "old": old, "new": image_dict['data']['filename'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Save Logos to file system under /app/images/ by overwriting old filename
    # Also used to write to footer-text.txt for footer info.
    def upload_logo_image(self, data_dict, user_email):
        response = [status.HTTP_200_OK, f"Picture {data_dict['data']['filename']} was uploaded.",
                    data_dict['data']['filename'], None]

        try:
            database_manager.get_database(self.database_id)
            file_data = json.dumps(data_dict['data']['val'])
            if data_dict['data']['filename'] == "footer-text.txt":
                try:
                    text_file = open(self.production_footer_text + "footer-text.txt", "w")
                    text_file.write(file_data.strip('""'))
                    text_file.close()
                except FileNotFoundError:
                    text_file = open(self.development_footer_text + "footer-text.txt", "w")
                    text_file.write(file_data.strip('""'))
                    text_file.close()
            else:
                try:
                    with open(self.production_logo_directory + data_dict['data']['filename'], 'wb') as file:
                        file.write(base64.b64decode(file_data.split(",")[1]))
                        file.close()
                except FileNotFoundError:
                    with open(self.development_logo_directory + data_dict['data']['filename'], 'wb') as file:
                        file.write(base64.b64decode(file_data.split(",")[1]))
                        file.close()

        except Exception as err:
            self.status_log("EXCEPTION (upload_logo_image)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not upload picture {data_dict['data']['filename']} due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "NONE", "old": data_dict['data']['filename'], "new": data_dict['data'],
                       "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    # Put an olt firmware into the database
    def upload_olt_firmware(self, firmware_dict, user_email):
        response = [status.HTTP_200_OK, f"OLT firmware {firmware_dict['data']['filename']} was uploaded.", None, {}]

        try:
            database_manager.get_database(self.database_id)
            file_id = os.path.splitext(firmware_dict['data']['filename'])[0]
            file = json.dumps(firmware_dict['data']['val'])
            fs = gridfs.GridFSBucket(db=self.database, bucket_name="OLT-FIRMWARE")
            if fs.find({"_id": firmware_dict['data']['filename']}).count() > 0:
                old = firmware_dict['data']['filename']
                fs.delete(file_id=file_id)
            else:
                response[0] = status.HTTP_201_CREATED
                old = None

            # Remove from chunks collection if necessary
            if self.database['OLT-FIRMWARE.chunks'].find({'files_id': file_id}).count() > 0:
                self.database['OLT-FIRMWARE.chunks'].delete_many({'files_id': file_id})

            file_data = base64.b64decode(file.split(",")[1])
            up = fs.open_upload_stream_with_id(file_id=file_id, filename=firmware_dict['data']['filename'],
                                               metadata=firmware_dict['data']['metadata'])
            up.write(file_data)
            up.close()

            # Add to defaults
            if not os.path.isfile(self.database_seed_base_dir + "firmwares/olts/" + firmware_dict['data']['filename']):
                with open(self.database_seed_base_dir + "firmwares/olts/" + firmware_dict['data']['filename'],
                          'wb') as file:
                    file.write(file_data)
            # Add to metadata
            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
                json_data["firmwares/olts/" + firmware_dict['data']['filename']] = firmware_dict['data']['metadata']
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))
            response[2] = firmware_dict['data']['filename']
        except Exception as err:
            self.status_log("EXCEPTION (upload_olt_firmware)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not upload OLT firmware {firmware_dict['data']['filename']} due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, {}]
            old = "ERROR"

        action_data = {"collection": "OLT-FIRMWARE",
                       "old": old, "new": firmware_dict['data']['filename'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Put an onu firmware into the database
    def upload_onu_firmware(self, firmware_dict, user_email):
        response = [status.HTTP_200_OK, f"ONU firmware {firmware_dict['data']['filename']} was uploaded."]

        try:
            database_manager.get_database(self.database_id)
            file = json.dumps(firmware_dict['data']['val'])
            fs = gridfs.GridFSBucket(db=self.database, bucket_name="ONU-FIRMWARE")
            if fs.find({"_id": firmware_dict['data']['filename']}).count() > 0:
                old = firmware_dict['data']['filename']
                fs.delete(file_id=firmware_dict['data']['filename'])
            else:
                response[0] = status.HTTP_201_CREATED
                old = None

            # Remove from chunks collection if necessary
            if self.database['ONU-FIRMWARE.chunks'].find({'files_id': firmware_dict['data']['filename']}).count() > 0:
                self.database['ONU-FIRMWARE.chunks'].delete_many({'files_id': firmware_dict['data']['filename']})

            file_data = base64.b64decode(file.split(",")[1])
            up = fs.open_upload_stream_with_id(file_id=firmware_dict['data']['filename'],
                                               filename=firmware_dict['data']['filename'],
                                               metadata=firmware_dict['data']['metadata'])
            up.write(file_data)
            up.close()

            # Add to defaults
            if not os.path.isfile(self.database_seed_base_dir + "firmwares/onus/" + firmware_dict['data']['filename']):
                with open(self.database_seed_base_dir + "firmwares/onus/" + firmware_dict['data']['filename'],
                          'wb') as file:
                    file.write(file_data)
            # Add to metadata
            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
                json_data["firmwares/onus/" + firmware_dict['data']['filename']] = firmware_dict['data']['metadata']
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))
        except Exception as err:
            self.status_log("EXCEPTION (upload_onu_firmware)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not upload ONU firmware {firmware_dict['data']['filename']} due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "ONU-FIRMWARE",
                       "old": old, "new": firmware_dict['data']['filename'], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    """
    Download files
    """

    # Put an image into the database
    def download_image(self, filename, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            database_manager.get_database(self.database_id)
            fs = gridfs.GridFSBucket(db=self.database, bucket_name="PICTURES")
            down = fs.find({"_id": os.path.splitext(filename)[0]})
            if down.count() > 0:
                for file in down:
                    response[1] = base64.b64encode(file.read())
            down.close()
        except Exception as err:
            self.status_log("EXCEPTION (download_image)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not download picture {filename} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def download_olt_debug_state(self, olt_id, user_email):
        """Returns a JSON file of the specified OLTs document from the the OLT-DEBUG-STATE collection.
        :param olt_id The ID of the OLT to get the debug state for
        :param user_email The email of the user requesting the OLT debug state"""
        response = [status.HTTP_200_OK, ""]
        collection_name = "OLT-DEBUG-STATE"

        try:
            database_manager.get_database(self.database_id)
            if collection_name in self.database.collection_names():
                collection = self.database[collection_name]
                olt_debug_state = collection.find_one({"_id": olt_id})
                if olt_debug_state is not None:
                    response[1] = olt_debug_state
                else:
                    response = [status.HTTP_404_NOT_FOUND,
                                f"The OLT {olt_id} does not have an {collection_name} entry. Click 'Generate' to create one."]
            else:
                response = [status.HTTP_404_NOT_FOUND,
                            f"The {collection_name} collection does not exist in the database. Click 'Generate' to create it."]
        except Exception as err:
            self.status_log("EXCEPTION (download_olt_debug_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not download OLT {olt_id} Debug State due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": collection_name, "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Delete files
    """

    # Delete the specified OLT Firmware file
    def delete_olt_firmware(self, file_id, user_email):
        response = [status.HTTP_200_OK, f"OLT firmware {file_id} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            filename_query = self.database["OLT-FIRMWARE.files"].find_one({"_id": file_id})
            if filename_query is not None and filename_query["filename"] is not None:
                filename = filename_query["filename"]
            else:
                filename = file_id
            fs = gridfs.GridFSBucket(db=self.database, bucket_name="OLT-FIRMWARE")
            fs.delete(file_id)

            if os.path.exists(self.database_seed_base_dir + "firmwares/olts/" + filename):
                os.remove(self.database_seed_base_dir + "firmwares/olts/" + filename)

            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
                if "firmwares/olts/" + filename in json_data:
                    json_data.pop("firmwares/olts/" + filename)
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))
        except Exception as err:
            self.status_log("EXCEPTION (delete_olt_firmware)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete OLT firmware {file_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-FIRMWARE",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete the specified ONU Firmware file
    def delete_onu_firmware(self, file_id, user_email):
        response = [status.HTTP_200_OK, f"ONU firmware {file_id} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            filename_query = self.database["ONU-FIRMWARE.files"].find_one({"_id": file_id})
            if filename_query is not None and filename_query["filename"] is not None:
                filename = filename_query["filename"]
            else:
                filename = file_id
            fs = gridfs.GridFSBucket(db=self.database, bucket_name="ONU-FIRMWARE")
            fs.delete(file_id)

            if os.path.exists(self.database_seed_base_dir + "firmwares/onus/" + filename):
                os.remove(self.database_seed_base_dir + "firmwares/onus/" + filename)

            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
                if "firmwares/onus/" + filename in json_data:
                    json_data.pop("firmwares/onus/" + filename)
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))
        except Exception as err:
            self.status_log("EXCEPTION (delete_onu_firmware)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete ONU firmware {file_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-FIRMWARE",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete the specified Picture file
    def delete_picture(self, filename, user_email):
        response = [status.HTTP_200_OK, f"Picture {filename} was deleted."]

        try:
            database_manager.get_database(self.database_id)
            file_id = os.path.splitext(filename)[0]
            fs = gridfs.GridFSBucket(db=self.database, bucket_name="PICTURES")
            metadata = self.database["PICTURES.files"].find_one({"_id": file_id}, {"_id": 0, "metadata.Device Type": 1})
            fs.delete(file_id)

            # Remove from defaults
            if metadata["metadata"]["Device Type"] == "CNTL":
                relative_path = "pictures/controllers/"
            elif metadata["metadata"]["Device Type"] == "SWI":
                relative_path = "pictures/switches/"
            elif metadata["metadata"]["Device Type"] == "OLT":
                relative_path = "pictures/olts/"
            else:
                relative_path = "pictures/onus/"

            if os.path.exists(self.database_seed_base_dir + relative_path + filename):
                os.remove(self.database_seed_base_dir + relative_path + filename)

            # Remove from metadata
            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
                if relative_path + filename in json_data:
                    json_data.pop(relative_path + filename)
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))
        except Exception as err:
            self.status_log("EXCEPTION (delete_picture)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete Picture {filename} due to {type(err).__name__}:{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES",
                       "deleted": response[0] == status.HTTP_200_OK, "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    """
    Modify File Metadata
    """

    def update_image(self, file_id, metadata, user_email):
        response = [status.HTTP_200_OK, status.HTTP_200_OK]

        try:
            database_manager.get_database(self.database_id)
            old = ""
            collection = self.database['PICTURES.files']
            file = collection.find_one({'_id': file_id})
            if file is not None:
                old = file['metadata']

            collection.update_one({'_id': file_id}, {"$set": {"metadata": metadata}})
        except Exception as err:
            self.status_log("EXCEPTION (update_image)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update Picture {file_id} due to {type(err).__name__}:{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES", "old": old, "new": metadata, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    # Updates the ONU Firmware file metadata
    def update_onu_fw_metadata(self, data, user_email):
        response = [status.HTTP_200_OK, f"ONU firmware {data['_id']} metadata was updated."]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-FIRMWARE.files"]
            old = collection.find_one({'_id': data['_id']}, {"metadata": 1})
            if old is not None:
                old = old["metadata"]
            inserted = collection.update_one({'_id': data['_id']}, {"$set": {"metadata": data["metadata"]}},
                                             upsert=True)

            # Add to metadata.json file
            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
            if "firmwares/onus/" + data['filename'] in json_data:
                json_data["firmwares/onus/" + data['filename']] = data["metadata"]
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))

            if inserted.upserted_id == data['_id']:
                response[0] = status.HTTP_201_CREATED
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update ONU firmware {data['_id']} due to No documents were modified::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (update_onu_fw_metadata)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update ONU firmware {data['_id']} due to {type(err).__name__}::{sys.exc_info()[1]}"]
            old = "ERROR"

        action_data = {"collection": "ONU-FIRMWARE.files",
                       "old": old, "new": data["metadata"], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    # Updates the ONU Firmware file metadata
    def update_olt_fw_metadata(self, data, user_email):
        response = [status.HTTP_200_OK, f"OLT firmware {data['_id']} metadata was updated.", None, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-FIRMWARE.files"]
            old = collection.find_one({'_id': data['_id']}, {"metadata": 1})
            if old is not None:
                old = old["metadata"]
            inserted = collection.update_one({'_id': data['_id']}, {"$set": {"metadata": data["metadata"]}},
                                             upsert=True)

            # Add to metadata.json file
            with open(self.database_seed_base_dir + "metadata.json", 'r') as file:
                json_data = json.load(file)
            if "firmwares/olts/" + data['filename'] in json_data:
                json_data["firmwares/olts/" + data['filename']] = data["metadata"]
            with open(self.database_seed_base_dir + "metadata.json", 'w') as file:
                file.write(json.dumps(json_data, sort_keys=True))

            if inserted.upserted_id == data['_id']:
                response[0] = status.HTTP_201_CREATED
                response[2] = data['filename']
            elif inserted.matched_count == 1:
                response[0] = status.HTTP_200_OK
                response[2] = data['filename']
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[
                    1] = f"Could not update OLT firmware {data['_id']} due to No documents were modified::No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (update_olt_fw_metadata)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT firmware {data['_id']} due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]
            old = "ERROR"

        action_data = {"collection": "OLT-FIRMWARE.files",
                       "old": old, "new": data["metadata"], "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK)

        return response

    """
    Get Firmware/Picture Information
    """

    # Retrieves information about the OLT firmware files
    def get_all_olt_firmwares_info(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-FIRMWARE.files"]
            firmwares = collection.find({}).sort('_id')
            for file in firmwares:
                response[1].append(file)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_firmwares_info)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT firmware information due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves information about the OLT firmware files
    def get_all_onu_firmwares_info(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-FIRMWARE.files"]
            firmwares = collection.find({})
            for file in firmwares:
                response[1].append(file)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_firmwares_info)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU firmware information due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the list of firmwares and its versions assigned to each ONU
    def get_first_assigned_onu_firmware(self, user_email, firmware, limit=1):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            limit = int(limit)
            if limit < 0:
                raise ValueError
            collection = self.database["ONU-CFG"]
            firmware_info = collection.find(
                {"ONU.FW Bank Files": {"$in": [firmware]}},
                {"_id": 1, "ONU.Name": 1},
            ).limit(limit)
            for fi in firmware_info:
                response[1].append(fi)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_firmwares)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU firmware due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the list of firmwares and its versions assigned to each ONU
    def get_first_assigned_onu_sla(self, user_email, sla):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            sla_info = collection.find(
                {"$or": [{"OLT-Service 0.SLA-CFG": sla}, {"OLT-Service 1.SLA-CFG": sla}, {"OLT-Service 2.SLA-CFG": sla},
                         {"OLT-Service 3.SLA-CFG": sla}, {"OLT-Service 4.SLA-CFG": sla}, {"OLT-Service 5.SLA-CFG": sla},
                         {"OLT-Service 6.SLA-CFG": sla}, {"OLT-Service 7.SLA-CFG": sla}]},
                {"_id": 1}
            ).limit(1)
            for fi in sla_info:
                response[1].append(fi)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_assigned_onu_slas)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU SLAs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the list of firmwares and its versions assigned to each ONU
    def get_first_assigned_olt_firmware(self, user_email, firmware, limit=1):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            limit = int(limit)
            if limit < 0:
                raise ValueError
            collection = self.database["OLT-CFG"]
            firmware_info = collection.find(
                {"OLT.FW Bank Files": {"$in": [firmware]}},
                {"_id": 1, "OLT.Name": 1},
            ).limit(limit)
            for fi in firmware_info:
                response[1].append(fi)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_firmwares)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT firmware due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves information about the Picture files
    def get_all_pictures_info(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["PICTURES.files"]
            firmwares = collection.find({})
            for file in firmwares:
                response[1].append(file)
        except Exception as err:
            self.status_log("EXCEPTION (get_all_pictures_info)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU firmware information due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "PICTURES.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    Get Compatibility Data
    """

    # Get list of all ONU/ONT manufacturers and models
    def get_onu_compatibility_data(self, user_email):
        response = [status.HTTP_200_OK, {"Vendors": [], "Models": []}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            vendors = collection.distinct("ONU.Vendor")
            models = collection.distinct("ONU.Model")
            collection = self.database["ONU-FIRMWARE.files"]
            vendors_from_firmwares = collection.distinct("metadata.Compatible Manufacturer")
            models_from_firmwares = collection.distinct("metadata.Compatible Model")

            # Combine Model array and values
            models.extend(model for model in models_from_firmwares if model not in models)
            # Combine Vendor array and values
            vendors.extend(vendor for vendor in vendors_from_firmwares if vendor not in vendors)

            response[1]['Vendors'] = vendors
            response[1]['Models'] = models
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_compatibility_data)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU compatibility data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE, ONU-FIRMWARE.files", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get list of all OLT models
    def get_olt_compatibility_data(self, user_email):
        response = [status.HTTP_200_OK, {"Models": []}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            models = collection.distinct("OLT.ASIC")
            collection = self.database["OLT-FIRMWARE.files"]
            models_from_firmwares = collection.distinct("metadata.Compatible Model")

            # Combine Model array and values
            models.extend(model for model in models_from_firmwares if model not in models)
            response[1]['Models'] = models
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_compatibility_data)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT compatibility data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get sys logs for specific user
    def get_sys_logs_for_user(self, user_email, since_utc_time, email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.user_database["SYSLOG-ACTIONS"]
            logs = collection.find({"Time": {"$gt": since_utc_time}, "User": user_email}, no_cursor_timeout=True)
            response[1] = list(logs)
        except Exception as err:
            self.status_log("EXCEPTION (get_sys_logs_for_user)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get user system logs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-ACTIONS", "user": email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Get sys logs for specific user
    def get_sys_log_returned(self, log_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.user_database["SYSLOG-ACTIONS"]
            response[1] = collection.find_one({'_id': log_id}, {"New": 1})
        except Exception as err:
            self.status_log("EXCEPTION (get_sys_log_returned)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get user system logs return data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SYSLOG-ACTIONS", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Log change user permission
    def change_user_permission(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log delete user
    def delete_user(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log change user last name
    def change_user_firstname(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log change user first name
    def change_user_lastname(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log change user password
    def change_user_password(self, user_email):
        action_data = {"collection": "auth_user", "user": user_email}
        self.sys_log(action_data, "get", True)
        return

    # Log change user password
    def create_user_log(self, user_created, user_email):
        action_data = {"collection": "auth_user", "old": {},
                       "new": user_created, "user": user_email}
        self.sys_log(action_data, "write", True)
        return

    """
    MIB Information
    """

    # Retrieve an array of all ONU MIB CUR States
    def get_onu_mib_cur_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-CUR-STATE"]
            cntls_cursor = collection.find()
            for state in cntls_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_mib_cur_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all MIB Current states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-MIB-CUR-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve ONU MIB CUR State for the specified ONU
    def get_onu_mib_cur_state(self, onu_id, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-CUR-STATE"]
            response[1] = collection.find_one({"_id": onu_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_mib_cur_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} MIB Current state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-MIB-CUR-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the ONUs firmware upgrade status and percentage
    def get_onu_upgrade_status(self, onu_id, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            onu_status = self.database["ONU-STATE"].find_one(
                {"_id": onu_id, "ONU.FW Upgrade Status.Status": {"$exists": True}})
            if onu_status is not None:
                cntl_status = self.database["CNTL-STATE"].find_one({"_id": onu_status["CNTL"]["MAC Address"]},
                                                                   {"_id": 0, "ONU.FW Upgrades": 1})
                if cntl_status is not None:
                    response[1] = {
                        "CNTL": cntl_status["ONU"]["FW Upgrades"],
                        "ONU": onu_status["ONU"]["FW Upgrade Status"]
                    }
                if onu_status['ONU']['FW Upgrade Status']['Status'] == 'Downloading' and onu_status['_id'] not in \
                        cntl_status['ONU']['FW Upgrades']:
                    response[1] = {
                        "CNTL": cntl_status["ONU"]["FW Upgrades"],
                        "ONU": {'Status': 'Failed'}
                    }
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_upgrade_status)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} upgrade status due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the ONUs firmware upgrade statuses and percentages for all ONUs under the OLT
    def get_onu_upgrade_statuses_for_olt(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            onus = collection.find({"OLT.MAC Address": olt_id}, {"_id": 1, "ONU.FW Upgrade Status": 1})
            olt_status = self.database["OLT-STATE"].find_one({"_id": olt_id}, {"_id": 0, "ONU.FW Upgrades": 1})
            for onu in onus:
                if 'Status' in onu["ONU"]["FW Upgrade Status"] and onu["ONU"]["FW Upgrade Status"][
                    'Status'] == 'Downloading' and onu['_id'] not in olt_status['ONU']['FW Upgrades']:
                    response[1].append({"_id": onu["_id"], "Upgrade Status": {'Status': 'Failed'}})
                else:
                    response[1].append({"_id": onu["_id"], "Upgrade Status": onu["ONU"]["FW Upgrade Status"]})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_upgrade_statuses_for_olt)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU upgrade statuses for OLT {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves the ONUs firmware upgrade statuses and percentages for all ONUs under the Controller
    def get_onu_upgrade_statuses_for_controller(self, cntl_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            onus = collection.find({"CNTL.MAC Address": cntl_id},
                                   {"_id": 1, "OLT.MAC Address": 1, "ONU.FW Upgrade Status": 1})
            cntl_status = self.database["CNTL-STATE"].find_one({"_id": cntl_id}, {"_id": 0, "ONU.FW Upgrades": 1})
            for onu in onus:
                if 'Status' in onu["ONU"]["FW Upgrade Status"] and onu["ONU"]["FW Upgrade Status"][
                    'Status'] == 'Downloading' and onu['_id'] not in cntl_status['ONU']['FW Upgrades']:
                    response[1].append({"_id": onu["_id"], "OLT ID": onu["OLT"]["MAC Address"],
                                        "Upgrade Status": {'Status': 'Failed'}})
                else:
                    response[1].append({"_id": onu["_id"], "OLT ID": onu["OLT"]["MAC Address"],
                                        "Upgrade Status": onu["ONU"]["FW Upgrade Status"]})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_upgrade_statuses_for_controller)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU upgrade statuses for PON Controller {cntl_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve an array of all ONU MIB RST States
    def get_onu_mib_rst_states(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-RST-STATE"]
            cntls_cursor = collection.find()
            for state in cntls_cursor:
                response[1].append(state)
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_mib_rst_states)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all MIB Reset states due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-MIB-RST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve ONU MIB RST State for the specified ONU
    def get_onu_mib_rst_state(self, onu_id, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-MIB-RST-STATE"]
            response[1] = collection.find_one({"_id": onu_id})
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_mib_rst_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU {onu_id} MIB Reset state due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-MIB-RST-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_custom_device_query(self, collection, attribute, operator, value, user_email):
        response = [status.HTTP_200_OK, []]
        num_value = None

        if value.isnumeric():
            num_value = int(value)
        elif value.lower() == 'false':
            value = False
        elif value.lower() == 'true':
            value = True

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[collection]
            devices = []
            if operator == "==":
                if num_value is not None:
                    devices = collection.find({attribute: {"$in": [value, num_value]}}, {attribute: 1})
                else:
                    devices = collection.find({attribute: {"$eq": value}}, {attribute: 1})
            elif operator == "!=":
                if num_value is not None:
                    devices = collection.find({attribute: {"$ne": [value, num_value]}}, {attribute: 1})
                else:
                    devices = collection.find({attribute: {"$ne": value}}, {attribute: 1})
            elif operator == "<=":
                devices = collection.find({attribute: {"$lte": num_value}}, {attribute: 1})
            elif operator == "<":
                devices = collection.find({attribute: {"$lt": num_value}}, {attribute: 1})
            elif operator == ">=":
                devices = collection.find({attribute: {"$gte": num_value}}, {attribute: 1})
            elif operator == ">":
                devices = collection.find({attribute: {"$gt": num_value}}, {attribute: 1})
            elif operator == 'like':
                devices = collection.find({attribute: {"$regex": value, "$options": 'i'}}, {attribute: 1})
            elif operator == 'all':
                devices = collection.find({attribute: {"$exists": True}}, {attribute: 1})

            for dev in devices:
                response[1].append(dev)
        except Exception as err:
            self.status_log("EXCEPTION (get_custom_device_query)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get custom query due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": collection, "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieves all the data for the test loop on the switch thermal test page
    def get_switch_thermal_test_data(self, switch_id, user_email):
        response = [status.HTTP_200_OK, []]
        stats = []

        try:
            database_manager.get_database(self.database_id)
            olt_state_collection = self.database["OLT-STATE"]
            onu_state_collection = self.database["ONU-STATE"]

            olts = olt_state_collection.find({"Switch.Chassis ID": switch_id}, {"_id": 1})

            for item in olts:
                olt = item["_id"]
                formatted_olt = olt.replace(":", "")
                stats_collection = self.database["STATS-OLT-{}".format(formatted_olt)]
                # stats data to return
                cursor = stats_collection.find().sort([('Time', -1)]).limit(8)
                # There will only ever be one item in cursor. Not sure of another way to write this.
                for x in cursor:
                    if "OLT-NNI" in x:
                        stats = x
                        break

                state = olt_state_collection.find_one({"_id": olt})
                bps = 0
                for onu in state["ONUs"]:
                    onu_state = onu_state_collection.find_one({"_id": onu})
                    if onu_state is not None:
                        for index in range(0, 8):
                            if f"OLT-PON Service {index}" in onu_state["STATS"]:
                                if "RX Rate bps" in onu_state["STATS"][f"OLT-PON Service {index}"]:
                                    bps += onu_state["STATS"][f"OLT-PON Service {index}"]["RX Rate bps"]
                                if "TX Rate bps" in onu_state["STATS"][f"OLT-PON Service {index}"]:
                                    bps += onu_state["STATS"][f"OLT-PON Service {index}"]["TX Rate bps"]

                response[1].append({"olt": olt, "state": state, "stats": stats, "traffic": bps})

        except Exception as err:
            self.status_log("EXCEPTION (get_switch_thermal_test_data)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get thermal test data due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE, ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_device_upgrade_hierarchy(self, user_email):
        response = [status.HTTP_200_OK, {'controllers': {}, 'switches': {}}]

        try:
            database_manager.get_database(self.database_id)
            # Collect all OLTs and ONUs beneath each Controller using Controllers System Status field
            cntl_ids = list(set().union(*[
                self.database['CNTL-STATE'].distinct('_id'),
                self.database['CNTL-CFG'].distinct('_id')
            ]))
            for cntl in cntl_ids:
                system_status = self.database['CNTL-STATE'].find_one({'_id': cntl}, {'System Status': 1, '_id': 0})
                if system_status is not None:
                    response[1]['controllers'][cntl] = {'olts': {}}
                    for olt in system_status['System Status'].keys():
                        response[1]['controllers'][cntl]['olts'][olt] = {'onus': []}
                        for onu in system_status['System Status'][olt]['ONUs'].keys():
                            response[1]['controllers'][cntl]['olts'][olt]['onus'].append(onu)

            # Get Switch Ids from OLT States then use Controller tree from above for OUNs
            switch_ids = self.database['OLT-STATE'].distinct("Switch.Chassis ID")
            for switch in switch_ids:
                response[1]['switches'][switch] = {'olts': {}}
                olts = self.database['OLT-STATE'].find({"Switch.Chassis ID": switch}, {"_id": 1, "CNTL.MAC Address": 1})
                for olt in olts:
                    if "CNTL" in olt and "MAC Address" in olt["CNTL"]:
                        cntl_mac = olt["CNTL"]["MAC Address"]
                        if cntl_mac in response[1]['controllers']:
                            response[1]['switches'][switch]['olts'] = response[1]['controllers'][cntl_mac]['olts']
                        else:
                            response[1]['switches'][switch]['olts'] = {}
        except Exception as err:
            self.status_log("EXCEPTION (get_device_upgrade_hierarchy)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get device upgrade tree due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-STATE, OLT-STATE, ONU-STATE, CNTL-CFG, SWI-CFG, OLT-CFG, ONU-CFG",
                       "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Upgrades the specified config files with the latest fields
    def upgrade_configurations_versions(self, new_config_info, user_email):
        response = [status.HTTP_200_OK, "Configuration versions were updated.", None, {}]
        all_collections = {}

        try:
            database_manager.get_database(self.database_id)
            new_data = {}

            # Get list of all collections and all document ids to be updated
            for data in new_config_info:
                collection_name = data["collection"]
                version = data["newVersion"]

                # PON Auto and documents managed by PON Controller have their CFG Versions is different places
                if collection_name in ['CNTL-AUTO-CFG', 'SWI-AUTO-CFG', 'OLT-AUTO-CFG', 'ONU-AUTO-CFG', 'AUTO-CFG']:
                    update_document = {"$set": {"AUTO.CFG Version": version}}
                else:
                    update_document = {"$set": {"CNTL.CFG Version": version}}

                if collection_name in ["CNTL-CFG", "SWI-CFG", "OLT-CFG", "ONU-CFG"]:
                    device_type = str(collection_name).split("-")[0].upper()
                    update_document["$inc"] = {f"{device_type}.CFG Change Count": 1}

                if collection_name not in all_collections:
                    # Loop through the new attributes to place in that document
                    for field in data["newFields"]:
                        document = list(field.keys())
                        value = list(field.values())
                        update_document["$set"][document[0]] = value[0]

                    all_collections[collection_name] = {
                        "ids": [],
                        "updateDoc": update_document
                    }

                all_collections[collection_name]["ids"].append(data["device"])

            # Preform the updates
            for collection_name, update_data in all_collections.items():
                collection = self.database[collection_name]
                collection.update_many(filter={"_id": {"$in": update_data["ids"]}}, update=update_data["updateDoc"],
                                       upsert=True)
                new_data[collection_name] = {
                    "affected": update_data["ids"],
                    "updates": update_data["updateDoc"]
                }

            response[2] = new_data
        except Exception as err:
            self.status_log("EXCEPTION (upgrade_configurations_versions)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update configuration versions due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": all_collections,
                       "returned": response[1], "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    """
    Utility functions
    """

    # Log the user action to the system
    # Deprecated as of R3.0.0. Logic removed as all logging is now handled by PonManagerLogHandler
    def sys_log(self, data, type, success):
        return True

    """
    Tibit Settings
    """

    # Get user session expiry age from user database
    def get_user_session_expiry(self, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            expiration_amounts = list(self.user_database["auth_user"].aggregate([
                {
                    '$match': {
                        'email': user_email
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'roles'
                    }
                }, {
                    '$unwind': {
                        'path': '$roles',
                        'includeArrayIndex': 'string',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group',
                        'localField': 'roles.group_id',
                        'foreignField': 'id',
                        'as': 'group'
                    }
                }, {
                    '$project': {
                        'Timeout': {
                            '$arrayElemAt': [
                                '$group.User Session Expiry Age Timeout', 0
                            ]
                        },
                        'Override': {
                            '$arrayElemAt': [
                                '$group.User Session Expiry Age Timeout Override', 0
                            ]
                        }
                    }
                }, {
                    '$sort': {
                        'Timeout': -1
                    }
                }
            ]))
            expiration = None

            if len(expiration_amounts) > 0:
                for exp_dict in expiration_amounts:
                    if "Timeout" in exp_dict:
                        expiration = exp_dict["Timeout"]
                        break

            if expiration is None:
                collection = self.user_database["tibit_settings"]
                expiration = collection.find_one({"_id": "Default"})["User Session Expiry Age Timeout"]

            if expiration is not None:
                response[1] = expiration

        except Exception as err:
            self.status_log("EXCEPTION (get_user_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get user session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "tibit_settings", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

        # Get user session expiry age from user database

    def set_role_session_expiry(self, role_id, override, timeout, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["auth_group"]
            old_document = collection.find_one({"id": role_id})
            update_document = {"$set": {"User Session Expiry Age Timeout": timeout, "User Session Expiry Age Timeout Override": override}}
            update_result = collection.update_one(filter={'id': role_id}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_user_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set user session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        action_data = {"collection": "tibit_settings",
                       "old": old_document, "new": update_document, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    def get_role_session_expiry(self, id):
        response = [status.HTTP_200_OK, ""]
        try:
            collection = self.user_database["auth_group"]
            role_info = collection.find_one({"id": id})
            if role_info and "User Session Expiry Age Timeout" in role_info:
                timeout = role_info["User Session Expiry Age Timeout"]
                timeout_override = role_info["User Session Expiry Age Timeout Override"]
                response[1] = {
                    "timeout": timeout,
                    "override": timeout_override
                }
            # Missing fields, use global
            else:
                collection = self.user_database["tibit_settings"]
                settings = collection.find_one({'_id': 'Default'})
                timeout = settings["User Session Expiry Age Timeout"]
                response[1] = {
                    "timeout": timeout,
                    "override": False
                }
        except Exception as err:
            self.status_log("EXCEPTION (get_role_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get role session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]
        return response

    def get_global_session_time(self):
        response = [status.HTTP_200_OK, ""]
        try:
            collection = self.user_database["tibit_settings"]
            settings = collection.find_one({'_id': 'Default'})
            if settings:
                timeout = settings["User Session Expiry Age Timeout"]
                response[1] = {
                    "timeout": timeout,
                }
        except Exception as err:
            self.status_log("EXCEPTION (get_global_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get global session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]
        return response

    def set_global_session_time(self, timeout, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["tibit_settings"]
            update_document = {"$set": {"User Session Expiry Age Timeout": timeout}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)

            # Set for all role timeouts that aren't overriden
            collection = self.user_database["auth_group"]
            group_update_document = {"$set": {"User Session Expiry Age Timeout": timeout, "User Session Expiry Age Timeout Override": False}}
            collection.update_many(filter={'User Session Expiry Age Timeout Override': {'$ne': True}}, update=group_update_document, upsert=True)

            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_user_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set user session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        action_data = {"collection": "tibit_settings",
                       "old": None, "new": timeout, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    def get_all_session_expiry(self):
        response = [status.HTTP_200_OK, "OK", None, {}]
        try:
            collection = self.user_database["auth_group"]
            roles = list(collection.find({}, {"User Session Expiry Age Timeout":1, "name":1}))
            timeoout_dictionary = {}
            global_timeout = None

            for role in roles:
                if "User Session Expiry Age Timeout" in role:
                    timeoout_dictionary[role['name']] = role["User Session Expiry Age Timeout"]
                else:
                    if global_timeout is None:
                        global_timeout = self.user_database["tibit_settings"].find_one({'_id': 'Default'})["User Session Expiry Age Timeout"]
                    timeoout_dictionary[role['name']] = global_timeout
            response[1] = timeoout_dictionary
        except Exception as err:
            self.status_log("EXCEPTION (get_role_session_expiry)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get role session expiry age due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]
        return response

    # Get locale timestamp format from user database
    def get_locale_timestamp_format(self, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            collection = self.user_database["tibit_settings"]
            expiration = collection.find_one({"_id": "Default"})

            if expiration is not None:
                response[1] = expiration
        except Exception as err:
            self.status_log("EXCEPTION (get_locale_timestamp_format)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get locale timestamp format due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "tibit_settings", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

        # Get user session expiry age from user database

    def set_locale_timestamp_format(self, locale_timestamp, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["tibit_settings"]
            update_document = {"$set": {"Locale Timestamp Format": locale_timestamp}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_locale_timestamp_format)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set locale_timestamp_format due to {type(err).__name__}::{sys.exc_info()[1]}", None,
                        None]

        action_data = {"collection": "tibit_settings",
                       "old": None, "new": locale_timestamp, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Get pon mode for the system from user database
    def get_pon_mode(self, user_email):
        response = [status.HTTP_200_OK, ""]

        try:
            collection = self.user_database["tibit_settings"]
            default_olt_settings = collection.find_one({"_id": "Default"})

            if default_olt_settings is not None:
                response[1] = default_olt_settings
        except Exception as err:
            self.status_log("EXCEPTION (get_pon_mode)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get pon mode due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "tibit_settings", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def set_pon_mode(self, pon_mode, user_email):
        response = [status.HTTP_200_OK, "OK", None, {}]

        try:
            collection = self.user_database["tibit_settings"]
            old = collection.find_one({'_id': 'Default'})
            update_document = {"$set": {"PON Mode": pon_mode}}
            update_result = collection.update_one(filter={'_id': 'Default'}, update=update_document, upsert=True)
            if update_result.upserted_id == "Default":
                response[0] = status.HTTP_201_CREATED
            else:
                response[0] = status.HTTP_200_OK
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (set_pon_mode)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not set pon_mode due to {type(err).__name__}::{sys.exc_info()[1]}", None, None]

        action_data = {"collection": "tibit_settings",
                       "old": old, "new": pon_mode, "user": user_email}
        self.sys_log(action_data, "write",
                     response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Utility Functions
    # Print message to the user
    @staticmethod
    def status_log(msg_type, message):
        print("PON MANAGER {} - {}".format(msg_type, message))

    @staticmethod
    def is_int(value):
        try:
            int(value)
            return True
        except ValueError:
            return False

    #Returns current time - given time
    @staticmethod
    def get_time_duration(timestamp, second_timestamp=None):
        time = datetime.datetime.strptime(timestamp, '%Y-%m-%d %H:%M:%S.%f')
        # time_delta = datetime.datetime.utcnow() - time
        time_delta_string = ""
        if second_timestamp:
            time2 = datetime.datetime.strptime(second_timestamp, '%Y-%m-%d %H:%M:%S.%f')
            time_delta = time2 - time
        else:
            time_delta = datetime.datetime.utcnow() - time

        if time_delta.days >= 1:
            time_delta_string += str(time_delta.days) + "d, "
        if time_delta.seconds >= 3600:
            time_delta_string += str(time_delta.seconds//3600) + "h, "
        if time_delta.seconds >= 60:
            time_delta_string += str((time_delta.seconds % 3600)//60) + "m"
        return time_delta_string

    @staticmethod
    def controller_time_is_online(cntl_state_timestamp, cntl_state_version):
        seconds_in_day = 60 * 60 * 24
        cntl_time = datetime.datetime.strptime(cntl_state_timestamp, '%Y-%m-%d %H:%M:%S.%f')
        time_delta = datetime.datetime.utcnow() - cntl_time
        max_time_delta = 6

        controller_version = cntl_state_version[1:].split("-")[0]
        controller_version_numbers = controller_version.split(".")

        if len(controller_version_numbers) > 2 and (int(controller_version_numbers[0]) >= 3 or (
                int(controller_version_numbers[0]) == 2 and int(controller_version_numbers[1]) >= 2)):
            max_time_delta = 2

        return divmod(time_delta.days * seconds_in_day + time_delta.seconds, 60)[0] < max_time_delta

    @staticmethod
    def automation_time_is_online(auto_state_timestamp):
        seconds_in_day = 60 * 60 * 24
        auto_time = datetime.datetime.strptime(auto_state_timestamp, '%Y-%m-%d %H:%M:%S.%f')
        time_delta = datetime.datetime.utcnow() - auto_time
        max_time_delta = 3

        return divmod(time_delta.days * seconds_in_day + time_delta.seconds, 60)[0] < max_time_delta

    """
    Angular 2 new endpoints
    """

    def make_bulk_change(self, changes, user_email):
        response = [status.HTTP_200_OK, {}, None, {}]
        changes = changes['changesObj']
        attribute = changes['attribute']
        value = changes['value']
        failed_devices = []
        do_change = False
        coll_prefix = ''

        try:
            database_manager.get_database(self.database_id)
            # First checks the type of device to save to
            if changes['collection'] == 'ONU Configurations':
                collection = self.database['ONU-CFG']
                coll_prefix = 'ONU'

                # Special case that needs special back-end validation
                if attribute == 'ONU.FW Bank Files' or attribute == 'ONU.FW Bank Versions':
                    onu_state = self.database['ONU-STATE']

                    for onu_id in changes['devices']:
                        fw_bank_files = ['', '']
                        fw_bank_versions = ['', '']
                        partial_success = False
                        save_fw = False
                        onu_state_vendor = onu_state.find_one({"_id": onu_id}, {"ONU.Vendor": 1})

                        # Check if the ONU has a state file and the state has a listed Vendor
                        if onu_state_vendor:
                            vendor_name = onu_state_vendor['ONU']['Vendor']

                            # Checks if user is configuring each bank. If they are and one fails it never saves.
                            # Check if user is configuring FW Bank 0
                            if 'fw0' in value:
                                if (value['fw0'][0]['_id'] == ''):
                                    fw_bank_files[0] = ('')
                                    fw_bank_versions[0] = ('')
                                    save_fw = True
                                # Ensure Manufacturer in the FW metadata matches that of the ONUs state Vendor
                                elif vendor_name == value['fw0'][0]['metadata']['Compatible Manufacturer']:
                                    fw_bank_files[0] = (value['fw0'][0]['_id'])
                                    fw_bank_versions[0] = (value['fw0'][0]['value'])
                                    save_fw = True
                                else:
                                    save_fw = False
                            else:
                                save_fw = True
                            # Check if user is configuring FW Bank 1
                            if 'fw1' in value and save_fw != False:
                                if (value['fw1'][0]['_id'] == ''):
                                    fw_bank_files[1] = ('')
                                    fw_bank_versions[1] = ('')
                                    save_fw = True
                                # Ensure Manufacturer in the FW metadata matches that of the ONUs state Vendor
                                elif vendor_name == value['fw1'][0]['metadata']['Compatible Manufacturer']:
                                    fw_bank_files[1] = (value['fw1'][0]['_id'])
                                    fw_bank_versions[1] = (value['fw1'][0]['value'])
                                    save_fw = True
                                else:
                                    save_fw = False
                        else:
                            # If the ONU doesn't have a state, the FW is saved since there is no validation check possible
                            if 'fw0' in value:
                                fw_bank_files[0] = (value['fw0'][0]['_id'])
                                fw_bank_versions[0] = (value['fw1'][0]['value'])
                                save_fw = True

                            # Check if user is configuring FW Bank 1
                            if 'fw1' in value:
                                fw_bank_files[1] = value['fw1'][0]['_id']
                                fw_bank_versions[1] = value['fw1'][0]['value']
                                save_fw = True

                        if save_fw:
                            # Save FW Bank Version to DB
                            update_result = collection.update_one({'_id': onu_id},
                                                                  {"$set":
                                                                      {
                                                                          'ONU.FW Bank Versions': fw_bank_versions,
                                                                          'ONU.FW Bank Files': fw_bank_files
                                                                      }, "$inc": {f"{coll_prefix}.CFG Change Count": 1}
                                                                  }
                                                                  )
                            if update_result.matched_count <= 0:
                                failed_devices.append(onu_id)
                        else:
                            failed_devices.append(onu_id)
                # End FW change

                # Check and increment reset count
                elif attribute == 'ONU.Reset Count':
                    for device_id in changes['devices']:
                        update_result = collection.update_one({'_id': device_id}, {"$inc": {"ONU.Reset Count": 1}})
                        if update_result.matched_count <= 0:
                            failed_devices.append(device_id)

                # All other changes
                else:
                    do_change = True

            elif changes['collection'] == 'OLT Configurations':
                collection = self.database['OLT-CFG']
                coll_prefix = 'OLT'
                if attribute == 'OLT.FW Bank Files' or attribute == 'OLT.FW Bank Versions':
                    # For each file, get version from OLT-FIRMWARE.files and save ther arrays of filenames and versions
                    fw_collection = self.database["OLT-FIRMWARE.files"]
                    version_array = []
                    filename_array = []

                    for version in value:
                        version_array.append(version)
                        if version != '':
                            fw_file = fw_collection.find_one({'metadata.Version': version}, {"filename": 1})
                            if fw_file is not None:
                                filename = fw_file['filename']
                                filename_array.append(filename)
                            else:
                                do_change = False
                                failed_devices = changes['devices']
                                break
                        else:
                            filename_array.append('')

                    # Save OLT FW Version and Filename to the db
                    for olt_id in changes['devices']:
                        update_result = collection.update_one({'_id': olt_id},
                                                              {"$set":
                                                                  {
                                                                      'OLT.FW Bank Versions': version_array,
                                                                      'OLT.FW Bank Files': filename_array
                                                                  }, "$inc": {f"{coll_prefix}.CFG Change Count": 1}
                                                              }
                                                              )
                        if update_result.matched_count <= 0:
                            failed_devices.append(olt_id)
                # Checks individual devices for pon mode then determines if value is within limits
                elif attribute == 'NNI.Max Frame Size':
                    for device_id in changes['devices']:
                        pon_mode = fw_file = collection.find_one({'_id': device_id}, {'OLT.PON Mode': 1})
                        if pon_mode is not None:
                            pon_mode = pon_mode['OLT']['PON Mode']
                            if pon_mode == 'GPON':
                                if value <= 9600:
                                    update_result = collection.update_one(
                                        {'_id': device_id}, {"$set": {attribute: value},
                                                             "$inc": {
                                                                 f"{coll_prefix}.CFG Change Count": 1}})
                                    if update_result.matched_count <= 0:
                                        failed_devices.append(device_id)
                                else:
                                    failed_devices.append(device_id)
                            else:
                                if value <= 12500:
                                    update_result = collection.update_one(
                                        {'_id': device_id}, {"$set": {attribute: value},
                                                             "$inc": {
                                                                 f"{coll_prefix}.CFG Change Count": 1}})
                                    if update_result.matched_count <= 0:
                                        failed_devices.append(device_id)
                                else:
                                    failed_devices.append(device_id)
                        else:
                            failed_devices.append(device_id)
                elif attribute == 'OLT.Reset Count':
                    for device_id in changes['devices']:
                        update_result = collection.update_one({'_id': device_id}, {"$inc": {"OLT.Reset Count": 1}})
                        if update_result.matched_count <= 0:
                            failed_devices.append(device_id)
                else:
                    do_change = True

            # No special cases for controllers or switches so those just save
            elif changes['collection'] == 'Controller Configurations':
                collection = self.database['CNTL-CFG']
                coll_prefix = 'CNTL'
                do_change = True

            elif changes['collection'] == 'Switch Configurations':
                collection = self.database['SWI-CFG']
                coll_prefix = 'SWI'
                do_change = True

            if do_change:
                # Potential for a field to be changed between the update and find
                # Only way to do an update many and determine which documents were actually updated (not just count)
                collection.update_many({"_id": {"$in": changes['devices']}},
                                       {"$set": {attribute: value}, "$inc": {f"{coll_prefix}.CFG Change Count": 1}})
                updated_devices = collection.find({"_id": {"$in": changes['devices']}}, {attribute: 1})

                for device in updated_devices:
                    result_value = device
                    keys = str(attribute).split(".")
                    for key in keys:
                        result_value = result_value[key]

                    if result_value != value:
                        failed_devices.append(device['_id'])

            response[1] = {'failed': failed_devices, 'newVal': value}
            response[2] = " ".join([collection.name, attribute, "=", str(value)])
        except Exception as err:
            self.status_log("EXCEPTION (make_bulk_change)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not make bulk change due to {type(err).__name__}::{sys.exc_info()[1]}", None, {}]

        action_data = {"collection": "ONU-CFG, OLT-CFG, CNTL-CFG, SWI-CFG", "user": user_email}
        if len(failed_devices) > 0:
            response[0] = status.HTTP_207_MULTI_STATUS
            self.sys_log(action_data, "post", response[0] == status.HTTP_207_MULTI_STATUS)
        else:
            response[0] = status.HTTP_200_OK
            self.sys_log(action_data, "post", response[0] == status.HTTP_200_OK)

        return response

    def get_controller_counts(self, user_email):
        response = [status.HTTP_200_OK, {"online": [], "offline": [], "pre-provisioned": []}]
        online_count = 0
        offline_count = 0
        pre_pro_count = 0

        try:
            database_manager.get_database(self.database_id)

            count_prepro = list(self.database["CNTL-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "CNTL-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    "$count": "prepro_count"
                }
            ]))
            pre_pro_count = get_nested_value(count_prepro, [0, "prepro_count"], 0)

            controller_states = self.database["CNTL-STATE"].find({}, {"System Status": 1, "Time": 1, "CNTL.Version": 1})
            for controller in controller_states:
                is_cntl_online = self.controller_time_is_online(controller["Time"], controller["CNTL"]["Version"])
                if not is_cntl_online:
                    offline_count += 1
                else:
                    online_count += 1

            response[1] = [{"name": "online", "value": online_count},
                           {"name": "offline", "value": offline_count},
                           {"name": "pre-provisioned", "value": pre_pro_count}]
        except Exception as err:
            self.status_log("EXCEPTION (get_controller_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get PON Controller counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG, CNTL-STATE", "returned": response[1], "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olt_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]
        online_count = 0
        offline_count = 0
        pre_pro_count = 0

        try:
            database_manager.get_database(self.database_id)
            olt_state_count = self.database["OLT-STATE"].count_documents({})  # number of documents in olt-state

            cntl_time_cutoff_r21 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=6))
            cntl_time_cutoff_r22 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=2))

            olt_counts = list(self.database["CNTL-STATE"].aggregate([
                {  # olderThanR220 field set to true if cntl is R221 or older
                    '$addFields': {
                        'olderThanR220': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.1.'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.0'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '1.'
                                                    ]
                                                }, 1
                                            ]
                                        }
                                    ]
                                }, True, False
                            ]
                        }
                    }
                }, {  # add controller status as online or offline
                    '$addFields': {
                        'status': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$and': [
                                                {
                                                    '$eq': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r22
                                                    ]
                                                }
                                            ]
                                        }, {
                                            '$and': [
                                                {
                                                    '$ne': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r21
                                                    ]
                                                }
                                            ]
                                        }
                                    ]
                                }, {
                                    'status': 'online'
                                }, {
                                    'status': 'offline'
                                }
                            ]
                        }
                    }
                }, {  # System Status object to array
                    '$project': {
                        '_id': 0,
                        'System Status': {
                            '$objectToArray': '$System Status'
                        },
                        'Status': '$status.status'
                    }
                }, {  # Each olt in System Status to their own document
                    '$unwind': {
                        'path': '$System Status',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {  # Project to only have olt id, olt state, controller state
                    '$project': {
                        'olt': '$System Status.k',
                        'state': '$System Status.v.OLT State',
                        'cntl': '$Status'
                    }
                }, {  # online: olt online and cntl online; else offline
                    '$group': {
                        '_id': 0,
                        'online': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$or': [
                                                    {
                                                        '$eq': [
                                                            '$state', 'Primary'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$state', 'Secondary'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$state', 'Unspecified'
                                                        ]
                                                    }
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$olt', '$$REMOVE'
                                ]
                            }
                        },
                        'offline': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$ne': [
                                                    '$state', 'Unspecified'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$state', 'Primary'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$state', 'Secondary'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$olt', {
                                        '$cond': [
                                            {
                                                '$eq': [
                                                    '$cntl', 'offline'
                                                ]
                                            }, '$olt', '$$REMOVE'
                                        ]
                                    }
                                ]
                            }
                        }
                    }
                }, {  # online count: online; offline count: offline - (offline intersect online)
                    '$project': {
                        'online count': {
                            '$size': '$online'
                        },
                        'offline count': {
                            '$subtract': [
                                {
                                    '$size': '$offline'
                                }, {
                                    '$size': {
                                        '$setIntersection': [
                                            '$offline', '$online'
                                        ]
                                    }
                                }
                            ]
                        }
                    }
                }
            ]))

            # Find pre-provisioned by getting OLT-CFG ids that don't have a state.
            agg_prepro = list(self.database["OLT-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "OLT-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    "$count": "prepro_count"
                }
            ]))
            pre_pro_count = get_nested_value(agg_prepro, [0, "prepro_count"], 0)
            online_count = get_nested_value(olt_counts, [0, "online count"], 0)
            offline_count = get_nested_value(olt_counts, [0, "offline count"], 0)

            if (offline_count + online_count) < olt_state_count:
                offline_count = olt_state_count - online_count

            response[1] = [{"name": "online", "value": online_count},
                           {"name": "offline", "value": offline_count},
                           {"name": "pre-provisioned", "value": pre_pro_count}]
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG, OLT-STATE", "returned": response[1], "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]
        online_count = 0
        offline_count = 0
        pre_pro_count = 0
        onu_state_count = 0

        try:
            database_manager.get_database(self.database_id)

            onu_state_count = self.database["ONU-STATE"].count_documents({})  # number of documents in onu-state

            cntl_time_cutoff_r21 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=6))
            cntl_time_cutoff_r22 = str(datetime.datetime.utcnow() - datetime.timedelta(minutes=2))

            onu_counts = list(self.database["CNTL-STATE"].aggregate([
                {   #add olderThanR220 field if cntl is R221 or older
                    '$addFields': {
                        'olderThanR220': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.1.'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '2.0'
                                                    ]
                                                }, 1
                                            ]
                                        }, {
                                            '$eq': [
                                                {
                                                    '$indexOfCP': [
                                                        '$CNTL.Version', '1.'
                                                    ]
                                                }, 1
                                            ]
                                        }
                                    ]
                                }, True, False
                            ]
                        }
                    }
                }, {# add status online or offline for each controller
                    '$addFields': {
                        'status': {
                            '$cond': [
                                {
                                    '$or': [
                                        {
                                            '$and': [
                                                {
                                                    '$eq': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r22
                                                    ]
                                                }
                                            ]
                                        }, {
                                            '$and': [
                                                {
                                                    '$ne': [
                                                        '$olderThanR220', False
                                                    ]
                                                }, {
                                                    '$gte': [
                                                        '$Time', cntl_time_cutoff_r21
                                                    ]
                                                }
                                            ]
                                        }
                                    ]
                                }, {
                                    'status': 'online'
                                }, {
                                    'status': 'offline'
                                }
                            ]
                        }
                    }
                }, {# project only system status and cntl status. System status obj to array
                    '$project': {
                        '_id': 0,
                        'System Status': {
                            '$objectToArray': '$System Status'
                        },
                        'Status': '$status.status'
                    }
                }, {# every olt in system status array becomes its own object
                    '$unwind': {
                        'path': '$System Status',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {# onu objects turned into array of onus
                    '$project': {
                        'onu': {
                            '$objectToArray': '$System Status.v.ONUs'
                        },
                        'cntl status': '$Status'
                    }
                }, {# separate documents for each onu
                    '$unwind': {
                        'path': '$onu',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {# if online cntl and registered/unspecified add to online set. Else add to offline set
                    '$group': {
                        '_id': 0,
                        'online': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$or': [
                                                    {
                                                        '$eq': [
                                                            '$onu.v', 'Registered'
                                                        ]
                                                    }, {
                                                        '$eq': [
                                                            '$onu.v', 'Unspecified'
                                                        ]
                                                    }
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl status', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$onu.k', '$$REMOVE'
                                ]
                            }
                        },
                        'offline': {
                            '$addToSet': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$ne': [
                                                    '$onu.v', 'Registered'
                                                ]
                                            }, {
                                                '$ne': [
                                                    '$onu.v', 'Unspecified'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$cntl status', 'online'
                                                ]
                                            }
                                        ]
                                    }, '$onu.k', {
                                        '$cond': [
                                            {
                                                '$eq': [
                                                    '$cntl status', 'offline'
                                                ]
                                            }, '$onu.k', '$$REMOVE'
                                        ]
                                    }
                                ]
                            }
                        }
                    }
                }, {# online count = count(online) offline count = count(offline - offline intersect online)
                    '$project': {
                        'online count': {
                            '$size': '$online'
                        },
                        'offline count': {
                            '$subtract': [
                                {
                                    '$size': '$offline'
                                }, {
                                    '$size': {
                                        '$setIntersection': [
                                            '$offline', '$online'
                                        ]
                                    }
                                }
                            ]
                        }
                    }
                }
            ]))

            count_prepro = list(self.database["ONU-CFG"].aggregate([
                {
                    "$lookup": {
                        "from": "ONU-STATE",
                        "localField": "_id",
                        "foreignField": "_id",
                        "as": "matched"
                    }
                },
                {
                    "$match": {
                        "$and": [{
                            "matched": {"$eq": []},
                            # This performs the function to get the ids from config not in state
                            "_id": {"$ne": "Default"}
                        }]
                    }
                },
                {
                    "$count": "prepro_count"
                }
            ]))
            pre_pro_count = get_nested_value(count_prepro, [0, "prepro_count"], 0)
            online_count = get_nested_value(onu_counts, [0, "online count"], 0)
            offline_count = get_nested_value(onu_counts, [0, "offline count"], 0)
            if (offline_count + online_count) < onu_state_count:
                offline_count = onu_state_count - online_count
            response[1] = [{"name": "online", "value": online_count},
                           {"name": "offline", "value": offline_count},
                           {"name": "pre-provisioned", "value": pre_pro_count}]
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG, ONU-STATE", "returned": response[1], "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_online_onus_for_controller(self, controller_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            cntl_state = self.database["CNTL-STATE"].find_one({"_id": controller_id},
                                                              {"System Status": 1, "_id": 0, "Time": 1,
                                                               "CNTL.Version": 1})
            if cntl_state is not None and "Time" in cntl_state and self.controller_time_is_online(cntl_state["Time"],
                                                                                                  cntl_state["CNTL"][
                                                                                                      "Version"]):
                for olt in cntl_state["System Status"]:
                    for onu in cntl_state["System Status"][olt]["ONUs"]:
                        if cntl_state["System Status"][olt]["ONUs"][onu] == "Unspecified" or \
                                cntl_state["System Status"][olt]["ONUs"][onu] == "Registered":
                            response[1].append(onu)
        except Exception as err:
            self.status_log("EXCEPTION (get_online_onus_for_controller)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONUs under PON Controller {controller_id} "
                        f"due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all controller config ids
    def get_all_cntl_cfg_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-CFG"]
            cfg_ids_cursor = collection.find({}, {"_id": 1})
            for id in cfg_ids_cursor:
                response[1].append(id['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_cntl_cfg_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all PON Controller configuration IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "CNTL-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all switch config ids
    def get_all_switch_cfg_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-CFG"]
            cfg_ids_cursor = collection.find({}, {"_id": 1})
            for id in cfg_ids_cursor:
                response[1].append(id['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_swi_cfg_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all Switch configuration IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "SWI-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all olt config ids
    def get_all_olt_cfg_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-CFG"]
            cfg_ids_cursor = collection.find({}, {"_id": 1})
            for id in cfg_ids_cursor:
                response[1].append(id['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_olt_cfg_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all OLT configuration IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve available OLTs to be selected as a Protection Peer
    def get_all_available_peer_olt_ids(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            cntl_cfg_collection = self.database["CNTL-CFG"]

            # Getting controller that the requesting OLT is inventoried on
            managing_cntl_inventory = list(cntl_cfg_collection.aggregate([
                {"$match": {"OLTs.Primary": olt_id}}]))

            # Get All OLT IDs that are on same controller inventory as requesting OLT, and have the field...
            # ...Protection.Peer set to A string of length 0
            if len(managing_cntl_inventory) > 0:
                all_inventoried_olts = managing_cntl_inventory[0]['OLTs']['Primary']

                olt_cfg_collection = self.database["OLT-CFG"]
                available_peer_olt_ids_cursor = olt_cfg_collection.find(
                    {"_id": {"$in": all_inventoried_olts}, "Protection.Peer": ''}, {'_id': 1})

                for id in available_peer_olt_ids_cursor:
                    # Not allowed to provision a protection peer as yourself
                    if id['_id'] != olt_id:
                        response[1].append(id['_id'])

        except Exception as err:
            self.status_log("EXCEPTION (get_all_available_peer_olt_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all available Protection Peer OLT IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Set PON Protection Automatic switchover to be Armed or Disarmed
    def put_protection_automatic_switchover(self, user_email, mac_address, auto_switchover_status):
        response = [status.HTTP_200_OK, f"OLT {mac_address} PON Protection Automatic Switchover status was updated.",
                    None, {}]

        # Must have peer configured for PON Controller to pickup action
        try:
            database_manager.get_database(self.database_id)
            olt_cfg_collection = self.database['OLT-CFG']
            olt = olt_cfg_collection.find_one({"_id": mac_address}, {"_id": 1, "Protection": 1})
            if olt is not None:
                olt_update_document = {"$set": {"Protection.Watch": auto_switchover_status},
                                       "$inc": {"Protection.Watch Count": 1, "OLT.CFG Change Count": 1}}
                peer_update_document = {"$set": {"Protection.Watch": auto_switchover_status},
                                        "$inc": {"Protection.Watch Count": 1, "OLT.CFG Change Count": 1}}
                olt_cfg_collection.update_one(filter={"_id": mac_address}, update=olt_update_document)
                olt_cfg_collection.update_one(filter={"_id": olt["Protection"]["Peer"]}, update=peer_update_document)
                # Compare old to new values for logging changes
                new_doc = {
                    mac_address: olt_update_document,
                    olt["Protection"]["Peer"]: peer_update_document
                }
                response[2] = new_doc
        except Exception as err:
            self.status_log("EXCEPTION (put_protection_automatic_switchover)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT {mac_address} PON Protection Automatic Switchover status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-CFG",
                       "old": "", "new": auto_switchover_status, "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Set PON Protection Forced switchover to be Armed or Disarmed
    def put_protection_forced_switchover(self, user_email, mac_address):
        response = [status.HTTP_200_OK, f"OLT {mac_address} PON Protection Forced Switchover status was updated.", None,
                    {}]

        # Must have peer configured and Protection.Watch set to Disabled for PON Controller to pickup action
        try:
            database_manager.get_database(self.database_id)
            olt_cfg_collection = self.database['OLT-CFG']
            update_document = {"$inc": {"Protection.Switchover Count": 1, "OLT.CFG Change Count": 1}}
            olt_cfg_collection.update_one({"_id": mac_address}, update_document, upsert=True)
            response[2] = update_document
        except Exception as err:
            self.status_log("EXCEPTION (put_protection_forced_switchover)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT {mac_address} PON Protection Forced Switchover status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-CFG",
                       "old": "", "new": "", "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Set PON Protection Forced switchover to be Armed or Disarmed
    def put_protection_cfg(self, user_email, mac_address, peer, inactivity_detection_time_active,
                           inactivity_detection_time_standby):
        response = [status.HTTP_200_OK, f"OLTs {mac_address} and {peer} PON Protection Configurations were updated.",
                    None, {}]

        try:
            database_manager.get_database(self.database_id)
            # Update current OLT config
            olt_cfg_collection = self.database['OLT-CFG']
            olt = olt_cfg_collection.find_one({"_id": mac_address}, {"_id": 1, "Protection": 1})
            olt_update_document = {"$set": {"Protection.Peer": peer,
                                            "Protection.Inactive Periods.Active": inactivity_detection_time_active,
                                            "Protection.Inactive Periods.Standby": inactivity_detection_time_standby},
                                   "$inc": {"OLT.CFG Change Count": 1}}
            peer_update_document = {"$set": {"Protection.Peer": mac_address,
                                             "Protection.Inactive Periods.Active": inactivity_detection_time_active,
                                             "Protection.Inactive Periods.Standby": inactivity_detection_time_standby,
                                             "Protection.Watch": olt["Protection"]['Watch']},
                                    "$inc": {"Protection.Watch Count": 1, "OLT.CFG Change Count": 1}}
            other_update_document = {"$set": {"Protection.Peer": '',
                                              "Protection.Watch": 'Disabled'},
                                     "$inc": {"Protection.Watch Count": 1, "OLT.CFG Change Count": 1}}
            olt_cfg_collection.update_one({"_id": mac_address}, olt_update_document)

            # Setting config of paired OLT peer
            if peer != '':
                olt_cfg_collection.update_one({"_id": peer}, peer_update_document)

            # If removing peer or changing to a different peer, we want to free up the partner peer
            if peer == '' or peer != olt['Protection']['Peer']:
                olt_cfg_collection.update_one({"_id": olt['Protection']['Peer']}, other_update_document)

            response[2] = {
                mac_address: olt_update_document,
                peer: peer_update_document,
                olt['Protection']['Peer']: other_update_document
            }
        except Exception as err:
            self.status_log("EXCEPTION (put_protection_forced_switchover)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLTs {mac_address} and {peer} PON Protection configurations due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-CFG",
                       "old": "", "new": "", "user": user_email}
        self.sys_log(action_data, "write", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Retrieve all onu config ids
    def get_all_onu_cfg_ids(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-CFG"]
            cfg_ids_cursor = collection.find({}, {"_id": 1})
            for id in cfg_ids_cursor:
                response[1].append(id['_id'])
        except Exception as err:
            self.status_log("EXCEPTION (get_all_onu_cfg_ids)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get all ONU configuration IDs due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    """
    ======================================================================
    =========================== PON Automation ===========================
    ======================================================================
    """

    # Retrieve automation states for ONUs under the OLT
    def get_onu_automation_states_for_olt(self, olt_id, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            olt_state_coll = self.database["OLT-STATE"]
            olt_state = olt_state_coll.find_one({"_id": olt_id})
            onu_ids = []
            if olt_state is not None and "ONU States" in olt_state:
                olt_onu_states = olt_state["ONU States"]
                if "Registered" in olt_onu_states:
                    onu_ids += olt_onu_states["Registered"]
                if "Unspecified" in olt_onu_states:
                    onu_ids += olt_onu_states["Unspecified"]
                if "Unprovisioned" in olt_onu_states:
                    onu_ids += olt_onu_states["Unprovisioned"]

            onu_auto_state_coll = self.database["ONU-AUTO-STATE"]
            response[1] = list(onu_auto_state_coll.find({"_id": {"$in": onu_ids}}, {"AUTO": 1}))
        except Exception as err:
            self.status_log("EXCEPTION (get_automation_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU Automation States for OLT {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve specified Automation config
    def get_automation_state(self, device_type, device_id, user_email):
        response = [status.HTTP_200_OK, []]
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type}-AUTO-STATE"]
            document = collection.find_one({"_id": device_id})

            if document is not None:
                response[1] = document
            else:
                response[1] = "Not found"
        except Exception as err:
            self.status_log("EXCEPTION (get_automation_state)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get {device_type} {device_id} Automation State due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve specified Automation config
    def get_automation_cfg(self, device_type, template_id, user_email):
        response = [status.HTTP_200_OK, []]
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type}-AUTO-CFG"]
            document = collection.find_one({"_id": template_id})

            if document is not None:
                response[1] = document
            else:
                response[1] = "Not found"
        except Exception as err:
            self.status_log("EXCEPTION (get_automation_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get {device_type} Automation configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Update specified Automation config
    def update_automation_cfg(self, device_type, template_id, new_document, user_email):
        response = [status.HTTP_200_OK, ["Document updated"]]
        old = None
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type}-AUTO-CFG"]
            old = collection.find_one({'id': template_id})
            if "_id" in new_document:
                del new_document['_id']

            updated = collection.replace_one({'_id': template_id}, new_document, upsert=True)
            if updated.upserted_id == template_id:
                response[0] = status.HTTP_201_CREATED
            elif updated.matched_count == 1:
                response[0] = status.HTTP_200_OK
            else:
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[1] = f"Could not update {template_id} {device_type} Automation. No documents were modified."
        except Exception as err:
            self.status_log("EXCEPTION (update_automation_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update {device_type} Automation configuration due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "user": user_email, "old": old, "new": new_document}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK)

        return response

    # Delete the specified automation configuration document
    def delete_automation_cfg(self, device_type, template_id, user_email):
        response = [status.HTTP_200_OK, f"{template_id} ONU Automation was deleted."]
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type.upper()}-AUTO-CFG"]
            collection.delete_one({'_id': template_id})
        except Exception as err:
            self.status_log("EXCEPTION (delete_automation_cfg)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not {device_type} Automation Configuration {template_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "deleted": response[0] == status.HTTP_200_OK,
                       "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Delete the specified reference from the Global mappings for the given device type
    def delete_automation_reference(self, device_type, automation_step, reference_name, user_email):
        response = [status.HTTP_200_OK, f"All ONU  data was deleted."]
        device_type = str(device_type).upper()

        try:
            database_manager.get_database(self.database_id)
            collection = self.database[f"{device_type}-AUTO-CFG"]
            if automation_step.upper() == "IDENTIFY":
                # Controller template uses Version as key, others use Description
                if device_type == "CNTL":
                    collection.update({'_id': "Global"},
                                      {"$pull": {"IDENTIFY.Mapping": {"[CNTL-STATE][CNTL][Version]": reference_name}}})
                else:
                    collection.update({'_id': "Global"},
                                      {"$pull": {"IDENTIFY.Mapping": {"Description": reference_name}}})
            elif automation_step.upper() == "SERVICE_PKG" and device_type == "ONU":
                collection.update({'_id': "Global"}, {"$unset": {
                    f"SERVICE.PACKAGES.{reference_name}": "",
                    f"AUTH.{reference_name}": "",
                    f"VLAN.{reference_name}": "",
                    f"SLA.{reference_name}": "",
                    f"DHCP.{reference_name}": ""
                }, })
            else:
                collection.update({'_id': "Global"}, {"$unset": {f"{automation_step.upper()}.{reference_name}": ""}})
        except Exception as err:
            self.status_log("EXCEPTION (delete_automation_reference)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not delete {device_type} Automation {automation_step.upper()} {reference_name} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": f"{device_type}-AUTO-CFG", "deleted": response[0] == status.HTTP_200_OK,
                       "user": user_email}
        self.sys_log(action_data, "delete", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all values of specific state fields
    def get_distinct_onu_state_identifiers(self, user_email):
        response = [status.HTTP_200_OK,
                    {"ONU.Manufacturer": [], "ONU.Model": [], "ONU.Vendor": [], "ONU.HW Version": [],
                     "ONU.Equipment ID": []}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-STATE"]
            response[1]["ONU.Manufacturer"] = collection.distinct("ONU.Manufacturer")
            response[1]["ONU.Model"] = collection.distinct("ONU.Model")
            response[1]["ONU.Vendor"] = collection.distinct("ONU.Vendor")
            response[1]["ONU.HW Version"] = collection.distinct("ONU.HW Version")
            response[1]["ONU.Equipment ID"] = collection.distinct("ONU.Equipment ID")
        except Exception as err:
            self.status_log("EXCEPTION (get_distinct_onu_state_identifiers)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get distinct ONU state fields due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Retrieve all values of specific state fields
    def get_distinct_olt_state_identifiers(self, user_email):
        response = [status.HTTP_200_OK, {"OLT.Manufacturer": [], "OLT.Model": [], "OLT.Vendor": []}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-STATE"]
            response[1]["OLT.Manufacturer"] = collection.distinct("OLT.Manufacturer")
            response[1]["OLT.Model"] = collection.distinct("OLT.Model")
            response[1]["OLT.Vendor"] = collection.distinct("OLT.Vendor")
        except Exception as err:
            self.status_log("EXCEPTION (get_distinct_olt_state_identifiers)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get distinct OLT state fields due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_switch_automation_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["SWI-AUTO-STATE"]
            response[1] = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Task': '$AUTO.TASK',
                        'status.Status': '$AUTO.STATUS',
                        'status.Enable': '$AUTO.Enable'
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'data': {
                            '$addToSet': '$status'
                        }
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'data': 1,
                        'DONE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'DONE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ERROR': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'ERROR'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR DEVICE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR DEVICE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR INPUT': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR INPUT'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ENABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', True
                                        ]
                                    }
                                }
                            }
                        },
                        'DISABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', False
                                        ]
                                    }
                                }
                            }
                        }
                    }
                }
            ]))[0]
        except Exception as err:
            self.status_log("EXCEPTION (get_switch_automation_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get switch automation counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_cntl_automation_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["CNTL-AUTO-STATE"]
            response[1] = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Task': '$AUTO.TASK',
                        'status.Status': '$AUTO.STATUS',
                        'status.Enable': '$AUTO.Enable'
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'data': {
                            '$addToSet': '$status'
                        }
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'data': 1,
                        'DONE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'DONE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ERROR': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'ERROR'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR DEVICE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR DEVICE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR INPUT': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR INPUT'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ENABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', True
                                        ]
                                    }
                                }
                            }
                        },
                        'DISABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', False
                                        ]
                                    }
                                }
                            }
                        }
                    }
                }
            ]))[0]
        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_automation_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get controller automation counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_olt_automation_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["OLT-AUTO-STATE"]
            counts = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Task': '$AUTO.TASK',
                        'status.Status': '$AUTO.STATUS',
                        'status.Enable': '$AUTO.Enable'
                    }
                }, {
                    '$lookup': {
                        'from': 'OLT-STATE',
                        'let': {
                            'id': '$status.Id'
                        },
                        'pipeline': [
                            {
                                '$match': {
                                    '$expr': {
                                        '$eq': [
                                            '$_id', '$$id'
                                        ]
                                    }
                                }
                            },
                            {
                                '$project': {
                                    '_id': 0,
                                    'CNTL.MAC Address': 1,
                                    'Switch.Chassis ID': 1
                                }
                            }
                        ],
                        'as': 'OLT-STATE'
                    }
                }, {
                    '$project': {
                        'status.Id': 1,
                        'status.Task': 1,
                        'status.Status': 1,
                        'status.Enable': 1,
                        'status.Parents': {'$arrayElemAt': ['$OLT-STATE', 0]}
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'data': {
                            '$addToSet': '$status'
                        }
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'data': 1,
                        'DONE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'DONE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ERROR': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'ERROR'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR DEVICE': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Status', 'WAITING FOR DEVICE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'WAITING FOR INPUT': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$$index.Task', 'WAITING FOR INPUT'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$$index.Enable', True
                                                ]
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        'ENABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', True
                                        ]
                                    }
                                }
                            }
                        },
                        'DISABLED': {
                            '$size': {
                                '$filter': {
                                    'input': '$data',
                                    'as': 'index',
                                    'cond': {
                                        '$eq': [
                                            '$$index.Enable', False
                                        ]
                                    }
                                }
                            }
                        }
                    }
                }
            ]))
            response[1] = counts[0]
        except Exception as err:
            self.status_log("EXCEPTION (get_olt_automation_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get olt automation counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    def get_onu_automation_counts(self, user_email):
        response = [status.HTTP_200_OK, {}]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database["ONU-AUTO-STATE"]
            data = list(collection.aggregate([
                            {
                                '$project': {
                                    'status.Id': '$_id',
                                    'status.Task': '$AUTO.TASK',
                                    'status.Status': '$AUTO.STATUS',
                                    'status.Enable': '$AUTO.Enable'
                                }
                            }, {
                                '$lookup': {
                                    'from': 'ONU-STATE',
                                    'let': {
                                        'id': '$status.Id'
                                    },
                                    'pipeline': [
                                        {
                                            '$match': {
                                                '$expr': {
                                                    '$eq': [
                                                        '$_id', '$$id'
                                                    ]
                                                }
                                            }
                                        },
                                        {
                                            '$project': {
                                                '_id': 0,
                                                'CNTL': 1,
                                                'OLT': 1
                                            }
                                        }
                                    ],
                                    'as': 'ONU-STATE'
                                }
                            }, {
                                '$project': {
                                    '_id': 0,
                                    'Id': '$status.Id',
                                    'Task': '$status.Task',
                                    'Status': '$status.Status',
                                    'Enable': '$status.Enable',
                                    'Parents': {'$arrayElemAt': ['$ONU-STATE', 0]}
                                }
                            }
                        ]))
            counts = list(collection.aggregate([
                {
                    '$project': {
                        'status.Id': '$_id',
                        'status.Task': '$AUTO.TASK',
                        'status.Status': '$AUTO.STATUS',
                        'status.Enable': '$AUTO.Enable'
                    }
                }, {
                    '$group': {
                        '_id': 0,
                        'DONE': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'DONE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$status.Enable', True
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'ERROR': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'ERROR'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$status.Enable', True
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'WAITING FOR DEVICE': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'WAITING FOR DEVICE'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$status.Enable', True
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'WAITING FOR INPUT': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$and': [
                                            {
                                                '$eq': [
                                                    '$status.Status', 'WAITING FOR INPUT'
                                                ]
                                            }, {
                                                '$eq': [
                                                    '$status.Enable', True
                                                ]
                                            }
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'ENABLED': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Enable', True
                                        ]
                                    }, 1, 0
                                ]
                            }
                        },
                        'DISABLED': {
                            '$sum': {
                                '$cond': [
                                    {
                                        '$eq': [
                                            '$status.Enable', False
                                        ]
                                    }, 1, 0
                                ]
                            }
                        }
                    }
                }
            ]))
            counts = counts[0]
            counts["data"] = data
            response[1] = counts
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_automation_counts)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get switch automation counts due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "AUTO-STATE", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response


    """
    ======================================================================
    ============================== Grafana ===============================
    ======================================================================
    """

    # Retrieve all grafana links for OLTs
    def get_grafana_olt_links(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            mydb = database_manager.get_database(self.database_id)
            collection = self.database["OLT-DASH-CFG"]
            all_collections = mydb.list_collection_names();

            if "OLT-DASH-CFG" in all_collections:
                links_cursor = collection.find()
                for link in links_cursor:
                    response[1].append(link)

            else:
                response[1] = None
        except Exception as err:
            self.status_log("EXCEPTION (get_grafana_links)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT Grafana links due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "OLT-DASH-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Save or Delete Grafana links for OLTs
    def put_grafana_olt_links(self, links, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database['OLT-DASH-CFG']

            for link in links['data']:
                id = link["_id"]
                if 'delete' in link:
                    collection.delete_one({"_id": id})
                else:
                    collection.replace_one({"_id": id}, link, upsert=True);

        except Exception as err:
            self.status_log("EXCEPTION (put_grafana_links)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update OLT-DASH-CFG status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "OLT-DASH-CFG", "user": user_email}
        self.sys_log(action_data, "put", response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    # Retrieve all grafana links for ONUs
    def get_grafana_onu_links(self, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            mydb = database_manager.get_database(self.database_id)
            collection = self.database["ONU-DASH-CFG"]
            all_collections = mydb.list_collection_names();

            if "ONU-DASH-CFG" in all_collections:
                links_cursor = collection.find()
                for link in links_cursor:
                    response[1].append(link)

            else:
                response[1] = None
        except Exception as err:
            self.status_log("EXCEPTION (get_grafana_links)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU Grafana links due to {type(err).__name__}::{sys.exc_info()[1]}"]

        action_data = {"collection": "ONU-DASH-CFG", "user": user_email}
        self.sys_log(action_data, "get", response[0] == status.HTTP_200_OK)

        return response

    # Save or Delete Grafana links for ONUs
    def put_grafana_onu_links(self, links, user_email):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            collection = self.database['ONU-DASH-CFG']

            for link in links['data']:
                id = link["_id"]
                if 'delete' in link:
                    collection.delete_one({"_id": id})
                else:
                    collection.replace_one({"_id": id}, link, upsert=True);

        except Exception as err:
            self.status_log("EXCEPTION (put_grafana_links)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not update ONU-DASH-CFG status due to {type(err).__name__}::{sys.exc_info()[1]}",
                        None, None]

        action_data = {"collection": "ONU-DASH-CFG", "user": user_email}
        self.sys_log(action_data, "put",
                     response[0] == status.HTTP_200_OK or response[0] == status.HTTP_201_CREATED)

        return response

    """
        ======================================================================
        ========================== Device Summary ============================
        ======================================================================
    """

    # retrieves online/offline status, state, and time stats (ex: online duration)
    def get_onu_summary(self, user_email, onu_id):
        response = [status.HTTP_200_OK, []]
        try:
            system_status = "Offline"
            is_offline = True
            is_resetting = False
            onu_data = self.database["ONU-STATE"].find_one({"_id": onu_id})
            config_reset_count = onu_data["ONU"]["Reset Count"]
            state_reset_count = onu_data["ONU"]["Reset Count"]
            olt_id = onu_data["OLT"]["MAC Address"]
            olt_state = self.database["OLT-STATE"].find_one({"_id": olt_id})
            cntl_id = onu_data["CNTL"]["MAC Address"]
            cntl_state = self.database["CNTL-STATE"].find_one({"_id": cntl_id})

            # Check that CNTL is online.
            if cntl_state is not None and "Time" in cntl_state and "CNTL" in cntl_state and "Version" in \
                    cntl_state["CNTL"] and self.controller_time_is_online(cntl_state["Time"],cntl_state["CNTL"]["Version"]):
                if olt_id in cntl_state["System Status"]:
                    if onu_id in cntl_state["System Status"][olt_id]["ONUs"]:
                        system_status = cntl_state["System Status"][olt_id]["ONUs"][onu_id]

                        # If ONU Status is Deregistered in CNTL-STATE, we must verify this is correct by checking OLT-STATE
                        if system_status == "Deregistered":
                            olt_state = get_nested_value(onu_data, ["OLT-STATE", 0])
                            if olt_state:
                                # Setting ONU status based on OLT-State ONU States
                                if onu_data in olt_state["ONU States"]["Disabled"]:
                                    system_status = "Disabled"
                                elif onu_data in olt_state["ONU States"]["Disallowed Admin"]:
                                    system_status = "Disallowed Admin"
                                elif onu_data in olt_state["ONU States"]["Disallowed Error"]:
                                    system_status = "Disallowed Error"
                                elif onu_data in olt_state["ONU States"]["Dying Gasp"]:
                                    system_status = "Dying Gasp"
                                elif onu_data in olt_state["ONU States"]["Unprovisioned"]:
                                    system_status = "Unprovisioned"

                        if cntl_state["System Status"][olt_id]["ONUs"][onu_id] == "Unspecified" or \
                                cntl_state["System Status"][olt_id]["ONUs"][onu_id] == "Registered" or \
                                cntl_state["System Status"][olt_id]["ONUs"][onu_id] == "FW Upgrade":
                            is_offline = False

            # Check if the ONU is resetting if online
            if not is_offline:
                if config_reset_count >= 0:
                    is_resetting = state_reset_count != config_reset_count

                if is_resetting:
                    system_status = "Pending Reset"

            last_status_update = onu_data["Time"]
            last_online = onu_data["ONU"]["Online Time"]
            last_offline = onu_data["Time"]
            uptime_duration = self.get_time_duration(last_online, last_offline)
            downtime_duration = self.get_time_duration(last_offline)

            response[1] = {"_id": onu_id,
                                "online": not is_offline,
                                "last_satus_update": last_status_update,
                                "last_online": last_online,
                                "last_offline": last_offline,
                                "uptime_duration": uptime_duration,
                                "downtime_duration": downtime_duration,
                                "state": system_status,
                                "state_collection": onu_data}
        except Exception as err:
            self.status_log("EXCEPTION (get_onu_summary)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get ONU Summary for {onu_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]

        return response

    # retrieves online/offline status, state, and time stats (ex: online duration)
    def get_olt_summary(self, user_email, olt_id):
        response = [status.HTTP_200_OK, []]
        try:
            database_manager.get_database(self.database_id)
            olt_state = self.database["OLT-STATE"].find_one({"_id": olt_id})
            cntl_id = olt_state["CNTL"]["MAC Address"]
            cntl_state = self.database["CNTL-STATE"].find_one({"_id": cntl_id})
            isOffline = False

            if cntl_state is not None and "Time" in cntl_state:
                if self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"]):
                    if olt_state is not None and olt_state["_id"] in cntl_state["System Status"]:
                        if cntl_state["System Status"][olt_state["_id"]]["OLT State"] == "Unspecified" or \
                                cntl_state["System Status"][olt_state["_id"]]["OLT State"] == "Primary" or \
                                cntl_state["System Status"][olt_state["_id"]]["OLT State"] == "Secondary":
                            isOffline = False
                        else:
                            isOffline = True
                    else:
                        isOffline = True
                else:
                    isOffline = True
            else:
                isOffline = True

            last_status_update = olt_state["Time"]
            last_online = olt_state["OLT"]["Online Time"]
            last_offline = olt_state["Time"]
            uptime_duration = self.get_time_duration(last_online, last_offline)
            downtime_duration = self.get_time_duration(last_offline)
            uptime = olt_state["OLT"]["Uptime"]

            response[1] = {"_id": olt_id,
                           "online": not isOffline,
                           "last_status_update": last_status_update,
                           "last_online": last_online,
                           "last_offline": last_offline,
                           "uptime_duration": uptime_duration,
                           "uptime": uptime,
                           "downtime_duration": downtime_duration,
                           "state_collection": olt_state}

        except Exception as err:
            self.status_log("EXCEPTION (get_olt_summary)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get OLT Summary for {olt_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
        return response

    # retrieves online/offline status and state
    def get_cntl_summary(self, user_email, cntl_id):
        response = [status.HTTP_200_OK, []]

        try:
            database_manager.get_database(self.database_id)
            cntl_state = self.database["CNTL-STATE"].find_one({"_id": cntl_id})
            online = self.controller_time_is_online(cntl_state["Time"], cntl_state["CNTL"]["Version"])
            last_status_update = cntl_state["Time"]
            paused = cntl_state["CNTL"]["Pause"]
            response[1] = {"online": online, "paused": paused, "last_status_update": last_status_update, "state_collection": cntl_state}

        except Exception as err:
            self.status_log("EXCEPTION (get_cntl_summary)", sys.exc_info()[1])
            response = [status.HTTP_500_INTERNAL_SERVER_ERROR,
                        f"Could not get CNTL Summary for {cntl_id} due to {type(err).__name__}::{sys.exc_info()[1]}"]
        return response

