"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import datetime
import json
import sys
import pymongo
import pymongo.errors
from threading import Lock
from typing import List, Tuple
from urllib.parse import quote_plus

from django.contrib.auth.models import User, Group
from pymongo import MongoClient, uri_parser, ReturnDocument
from pymongo.database import Database
from pymongo.results import UpdateResult
from rest_framework.exceptions import APIException

from api.settings import IN_PRODUCTION
from json_validator import JsonSchemaValidator
from log.DatabaseLogHandler import DatabaseLogHandler
from mongo_heartbeat import MongoServerHeartbeatHandler
from log.PonManagerLogger import pon_manager_logger
from manage import BUILDING_DOCUMENTATION

_DEFAULT_CONTROLLER_DATABASE_INFO = {
    "host": "127.0.0.1",
    "name": "tibit_pon_controller",
    "port": "27017",
    "auth_enable": False,
    "auth_db": "tibit_users",
    "username": "",
    "password": "",
    "tls_enable": False,
    "ca_cert_path": "",
    "replica_set_enable": False,
    "replica_set_hosts": [],
    "replica_set_name": "",
    "dns_srv": False,
    "db_uri": ""
}
_DEFAULT_USER_DATABASE_INFO = {
    "host": "127.0.0.1",
    "name": "tibit_users",
    "port": "27017",
    "auth_enable": False,
    "auth_db": "tibit_users",
    "username": "",
    "password": "",
    "tls_enable": False,
    "ca_cert_path": "",
    "replica_set_enable": False,
    "replica_set_hosts": [],
    "replica_set_name": "",
    "dns_srv": False,
    "db_uri": "",
    "django_key": ""
}


class DatabaseManager:
    """ Creates and handles database connections for PON Manager REST API """

    def __init__(self):
        """
        Initializes the Database Manager

        Reads the databases, user_database, and user_migration JSON files to load database connections
        Creates a single MongoClient to the user database, and one MongoClient PER entry in the databases file
        """
        self._session_selected_databases = {}
        self._databases = {}
        self._databases_lock = Lock()
        self._file_lock = Lock()
        self._json_schema_validator = None
        # Sets default SYSLOG-ACTIONS collection size
        self._COLLECTION_CAP_BYTES_DEFAULT = 50000000
        pon_manager_logger.info("PON Manager REST API started")

        if IN_PRODUCTION:
            self._DATABASES_FILE = "/var/www/html/api/databases.json"
            self._USER_DATABASE_FILE = "/var/www/html/api/user_database.json"
            self._USER_MIGRATION_FILE = "/var/www/html/api/user_migration.json"
            self._SCHEMA_FILES = "/var/www/html/api/schema_files"
        else:
            self._DATABASES_FILE = "databases.json"
            self._USER_DATABASE_FILE = "user_database.json"
            self._USER_MIGRATION_FILE = "user_migration.json"
            self._SCHEMA_FILES = "schema_files"

        self._user_migration_options = self.read_json_file(self._USER_MIGRATION_FILE)
        user_database_json = self.read_json_file(self._USER_DATABASE_FILE)

        # Connect to user database
        try:
            temp_user_db, heart_beat = self._create_connection(user_database_json, _DEFAULT_USER_DATABASE_INFO)
            if temp_user_db is not None:
                self._USER_DATABASE = temp_user_db, heart_beat
                pon_manager_logger.info("User database is active")
            else:
                pon_manager_logger.critical("User database is NOT active. Exiting...")
                sys.exit(1)
        except (pymongo.errors.ConfigurationError, pymongo.errors.ServerSelectionTimeoutError) as e:
            pon_manager_logger.critical(f"User database connection error: {e}\nExiting...")
            sys.exit(1)

        # Do migrations for user database
        try:
            self._perform_migrations()
        except Exception as e:
            pon_manager_logger.error(f"Error migrating user options to user database: {e}")

        # Add the database log handler to the logger
        database_logger = DatabaseLogHandler(self.user_database)
        pon_manager_logger.add_handler(database_logger)

        # Connect to each PON Controller database
        try:
            databases_json = self.read_json_file(self._DATABASES_FILE)
            for database_id in databases_json.keys():
                try:
                    client, heart_beat = self._create_connection(databases_json[database_id],
                                                                 _DEFAULT_CONTROLLER_DATABASE_INFO)
                    if client is not None:
                        self._databases[database_id] = client, heart_beat
                        pon_manager_logger.info(f"Database {database_id} active")
                except pymongo.errors.ConfigurationError as e:
                    pon_manager_logger.error(f"Failed to connect to database {database_id}. Database URI error: {e}")
        except FileNotFoundError:
            pon_manager_logger.error("Databases json file not found")

        # Instantiate JSON validators for each collection
        self._json_schema_validator = JsonSchemaValidator(self._SCHEMA_FILES)

    @property
    def user_database(self):
        return self._USER_DATABASE[0]

    def get_tibit_settings(self, query=None, projection=None):
        """ Retrieves the chosen tibit settings document """
        if query is None:
            query = {"_id": "Default"}
        collection = self._USER_DATABASE[0].get_collection("tibit_settings")
        return collection.find_one(query, projection)

    def read_json_file(self, file_path: str) -> dict:
        """ Reads the json contents of the file specified in the given path

        :param file_path: Path to the json file to read
        :return json file contents as dict
        :raises FileNotFoundError, JSONDecodeError
        """
        with self._file_lock:
            with open(file_path, 'r') as file:
                file_contents = json.load(file)

        return file_contents

    def update_json_file_key(self, file_path: str, key: str, new_dict):
        """ Updates the document of the given key in the JSON file at file_path

        :param file_path: Path to the json file to read
        :param key: The JSON key to modify or create
        :param new_dict: The JSON body. If None, deletes the key from the dictionary
        :raises FileNotFoundError, JSONDecodeError
        """
        with self._file_lock:
            with open(file_path, 'r') as file:
                file_contents = json.load(file)
                if new_dict is None:
                    del file_contents[key]
                else:
                    file_contents[key] = new_dict
            with open(file_path, 'w') as file:
                json.dump(obj=file_contents, fp=file, indent=4)

    def set_session_database_id(self, session_key, database_id):
        """ Sets the selected database for the given session """
        self._session_selected_databases[session_key] = database_id

    def get_session_database_id(self, session_key):
        """ Gets the selected database for the given session """
        return self._session_selected_databases[session_key]

    def remove_session_database_id(self, session_key):
        """ Deletes the selected database for the given session """
        del self._session_selected_databases[session_key]

    def set_users_selected_database(self, user_email, database_id) -> UpdateResult:
        """ Sets the users active database in the user database """
        collection = self.user_database.get_collection("auth_user")
        try:
            result = collection.update_one({"email": user_email}, {"$set": {"active_database": database_id}})
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")
        return result

    def get_users_selected_database(self, user_email):
        """ Get the users active PON Controller database selection from the user database

        :raises KeyError if the active_database fetched is not found or is not an active database connection
        """
        collection = self.user_database.get_collection("auth_user")

        try:
            result = collection.find_one({"email": user_email}, {"active_database": 1, "_id": 0})
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        if result is None or "active_database" not in result or result['active_database'] not in self._databases.keys():
            raise KeyError

        return result['active_database']

    def get_user_session_expiry(self, email) -> int:
        """ Get user session expiry age from user database """

        try:
            expiration_amounts = list(self.user_database["auth_user"].aggregate([
                {
                    '$match': {
                        'email': email
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_user_groups',
                        'localField': 'id',
                        'foreignField': 'user_id',
                        'as': 'roles'
                    }
                }, {
                    '$unwind': {
                        'path': '$roles',
                        'includeArrayIndex': 'string',
                        'preserveNullAndEmptyArrays': False
                    }
                }, {
                    '$lookup': {
                        'from': 'auth_group',
                        'localField': 'roles.group_id',
                        'foreignField': 'id',
                        'as': 'group'
                    }
                }, {
                    '$project': {
                        'Timeout': {
                            '$arrayElemAt': [
                                '$group.User Session Expiry Age Timeout', 0
                            ]
                        },
                        'Override': {
                            '$arrayElemAt': [
                                '$group.User Session Expiry Age Timeout Override', 0
                            ]
                        }
                    }
                }, {
                    '$sort': {
                        'Timeout': -1
                    }
                }
            ]))
            expiration = None

            if len(expiration_amounts) > 0:
                for exp_dict in expiration_amounts:
                    if "Timeout" in exp_dict:
                        expiration = exp_dict["Timeout"]
                        break
            if expiration is None:
                collection = self.user_database["tibit_settings"]
                expiration = collection.find_one({"_id": "Default"})["User Session Expiry Age Timeout"]

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return int(expiration)

    def list_databases(self):
        """ Gives a list of databases currently active

        :return List of database IDs available
        """
        return self._databases.keys()

    def get_all_databases(self):
        """ Gives a list of databases currently active

        :return Dictionary of database connections
        """
        return self._databases

    def get_databases_json(self):
        """ Gives a dictionary of all active databases parameters

        :return Dictionary of databases information
        """
        return self.read_json_file(self._DATABASES_FILE)

    def get_database(self, database_id: str) -> Database:
        """ Get the connection to the database specified by the database_id

        :param database_id: ID of the database reference to get
        :return Database object of the database connection
        :raises KeyError, APIException
        """
        if not self._databases[database_id][1].is_alive:
            raise APIException(detail=self._databases[database_id][1].details)
        return self._databases[database_id][0]

    def mongo_server_is_active(self, database_id: str) -> bool:
        """ Check the given database connection is alive

        :param database_id: ID of the database reference to check
        :return bool describing if the mongo server connection is ok
        :raises KeyError if the database_id is not found in the active list
        """
        return self._databases[database_id][1].is_alive

    def mongo_server_get_status(self, database_id: str) -> str:
        """ Check the status of the given mongo server

        :param database_id: ID of the database reference to check
        :return str status of mongo server connection
        :raises KeyError if the database_id is not found in the active list
        """
        status = self._databases[database_id][1].details
        heartbeat = self._databases[database_id][1]
        if heartbeat.is_alive:
            collections = self._databases[database_id][0].collection_names()

            if 'CNTL-CFG' not in collections:
                status = f"Database '{heartbeat.name}' is Not Populated"
            else:
                status = 'Online'
        return status

    def add_database(self, database_id: str, database_json: dict):
        """ Create a new database connection and add the information to the databases file

        :param database_id: ID of the new database
        :param database_json: Dictionary of the new database connection parameters
        :raises DuplicateKeyError if the database ID is already in use
        """
        with self._databases_lock:
            if database_id in self._databases.keys():
                raise pymongo.errors.DuplicateKeyError(error="Database ID already in use")
            else:
                new_database, heart_beat = self._create_connection(database_json, _DEFAULT_CONTROLLER_DATABASE_INFO)
                if new_database is not None:
                    self.update_json_file_key(self._DATABASES_FILE, database_id, database_json)
                    self._databases[database_id] = new_database, heart_beat
                else:
                    raise APIException("Failed to create database connection. See logs for details.")

    def edit_database(self, database_id: str, database_json: dict) -> str:
        """ Update an existing database connection and update the information in the databases file

        :param database_id: ID of the new database
        :param database_json: Dictionary of the new database connection parameters
        :return str Describing the status of the edit or error encountered
        """
        with self._databases_lock:
            if database_id in self._databases.keys():
                result = "Updated"
            else:
                result = "Created"

            new_database, heart_beat = self._create_connection(database_json, _DEFAULT_CONTROLLER_DATABASE_INFO)
            if new_database is not None:
                self.update_json_file_key(self._DATABASES_FILE, database_id, database_json)
                self._databases[database_id] = new_database, heart_beat
            else:
                raise APIException("Failed to edit database connection. See logs for details.")

        return result

    def remove_database(self, database_id: str):
        """ Deletes the given database reference from the list of connections

        :param database_id: ID of the database reference to delete
        """
        with self._databases_lock:
            self.update_json_file_key(file_path=self._DATABASES_FILE, key=database_id, new_dict=None)
            if database_id in self._databases.keys():
                # Make sure to stop the heartbeat and close the MongoClient
                # The explicit stop is only to prevent a failed heartbeat message after the connection is closed
                self._databases[database_id][1].stop()
                self._databases[database_id][0].client.close()
                del self._databases[database_id]

    def find(self, database_id: str, collection: str, query=None, projection=None, sort=None, limit=0, skip=0,
             next=None):
        """ Preforms the collection.find operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param projection: Query projection document
        :param sort: List of Tuples with the key and 1 or -1 in order of the key to sort on first
        :param limit: The total number of documents to return
        :param skip: The number of documents to omit from the beginning of the results
        :param next: The _id to being the query at
        :return List of retrieved documents
        """
        return self._find(database_id=database_id, collection=collection, query=query, projection=projection, sort=sort,
                          limit=limit, skip=skip, next=next, one=False)

    def find_one(self, database_id: str, collection: str, query: dict, projection=None):
        """ Preforms the collection.find_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param projection: Query projection document
        :return Dictionary of retrieved document
        """
        return self._find(database_id=database_id, collection=collection, query=query, projection=projection, one=True)

    def distinct(self, database_id: str, collection: str, query=None, distinct=None):
        """ Preforms the collection.distinct operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param distinct: The Field to get unique values of
        :return Dictionary with array of unique values
        """
        return self._distinct(database_id=database_id, collection=collection, query=query, field=distinct)

    def update_many(self, database_id: str, collection: str, query: dict, update_document: dict, upsert=False):
        """ Preforms the collection.update_many operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param update_document: The update options dictionary
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        return self._update(database_id=database_id, collection=collection, query=query,
                            update_document=update_document, upsert=upsert, many=True)

    def update_one(self, database_id: str, collection: str, query: dict, update_document: dict, upsert=False):
        """ Preforms the collection.update_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param update_document: The update options dictionary
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        return self._update(database_id=database_id, collection=collection, query=query,
                            update_document=update_document, upsert=upsert, many=False)

    def insert_many(self, database_id: str, collection: str, documents: List[dict]):
        """ Preforms the collection.insert_many operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param documents: List of documents to insert into the collection
        :return InsertManyResult object
        """
        return self._insert(database_id=database_id, collection=collection, documents=documents, many=True)

    def insert_one(self, database_id: str, collection: str, document: dict):
        """ Preforms the collection.insert_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param document: Document to insert into the collection
        :return InsertOneResult object
        """
        return self._insert(database_id=database_id, collection=collection, documents=document, many=False)

    def find_one_and_update(self, database_id: str, collection: str, query: dict, update_document: dict,
                            projection=None, return_doc=ReturnDocument.BEFORE, upsert=False):
        """ Preforms the collection.find_one_and_update operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param update_document: The update options dictionary
        :param projection: Projection document for the find operation
        :param return_doc: ReturnDocument.BEFORE or ReturnDocument.AFTER
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            return_doc = collection.find_one_and_update(filter=query, update=update_document, projection=projection,
                                                        return_document=return_doc, upsert=upsert)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return return_doc

    def find_one_and_replace(self, database_id: str, collection: str, query: dict, new_document: dict, projection=None,
                             return_doc=ReturnDocument.BEFORE, upsert=True):
        """ Preforms the collection.find_one_and_replace operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param new_document: The new document
        :param projection: Projection document for the find operation
        :param return_doc: ReturnDocument.BEFORE or ReturnDocument.AFTER
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            return_doc = collection.find_one_and_replace(filter=query, replacement=new_document, projection=projection,
                                                         return_document=return_doc, upsert=upsert)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return return_doc

    def replace_one(self, database_id: str, collection: str, query: dict, new_document: dict, upsert=False):
        """ Preforms the collection.replace_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        :param new_document: The document to insert
        :param upsert: Insert the document if it does not exist if set to true
        :return UpdateResult object
        """
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            replace_result = collection.replace_one(filter=query, replacement=new_document, upsert=upsert)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return replace_result

    def delete_many(self, database_id: str, collection: str, query: dict):
        """ Preforms the collection.delete_many operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        """
        self._delete(database_id=database_id, collection=collection, query=query, many=True)

    def delete_one(self, database_id: str, collection: str, query: dict):
        """ Preforms the collection.delete_one operation

        :param database_id: The string ID of the database
        :param collection: The string name of the collection to access
        :param query: Custom query filter document
        """
        self._delete(database_id=database_id, collection=collection, query=query, many=False)

    def validate(self, collection: str, document: dict, validate_required: bool = True, schema: dict = None) -> Tuple[
        bool, dict]:
        """ Validates a document against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param document: The document to validate
        :param validate_required: Check for missing required fields in the document (set to False for PATCH)
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the document
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the document fails JSON validation and '{}' otherwise.
        """
        schema_version = None
        if "CNTL" in document and "CFG Version" in document["CNTL"]:
            schema_version = self._json_schema_validator.get_schema_version_for_doc_version(
                document["CNTL"]["CFG Version"])
        else:
            # If the configuration version isn't present in the document, validate the document against
            # the 'current' (or latest) schema.
            schema_version = 'current'

        if schema_version:
            is_valid, details = self._json_schema_validator.validate(collection, document, validate_required,
                                                                     schema_version, schema)
            if not is_valid and details:
                print(f"ERROR: JSON validation failed for {details['collection']}, doc_id {details['id']}.")
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
            elif details and details['level'] == 'warning':
                print(
                    f"WARNING: JSON validation warning for {details['collection']}, doc_id {details['id']}, {details['message']}.")
                # Warnings are considered success
        else:
            # Skip validation if no validator exists for the specified document version
            is_valid = True
            details = {}

        return is_valid, details

    def validate_path(self, collection: str, path: str, schema: dict = None) -> Tuple[bool, dict]:
        """ Validates a MongoDB path against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param path: A path to validate
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the path
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the path fails JSON validation and '{}' otherwise.
        """
        valid = True
        details = {}
        valid, details = self._json_schema_validator.validate_path(collection, path, schema=schema)
        if not valid and 'bad value' in details:
            print(
                f"ERROR: Invalid query value for {details['collection']}, path {details['path']}, value {details['bad value']}.")
            print(f"ERROR: {details['message']}")
            err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
            err_msg = err_msg.replace("\n", "\nERROR: ")
            print(f"ERROR: details = {err_msg}")
        elif not valid:
            print(f"ERROR: Invalid query path for {details['collection']}, path {details['path']}.")
            err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
            err_msg = err_msg.replace("\n", "\nERROR: ")
            print(f"ERROR: details = {err_msg}")

        return valid, details

    def validate_query(self, collection: str, query_params: dict, schema: dict = None) -> Tuple[bool, dict]:
        """ Validates a MongoDB query path and value against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param query_params: A dictionary of query paths and values to validate
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the path or value
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the path or value fails JSON validation and '{}' otherwise.
        """
        valid = True
        details = {}
        for key, value in query_params.items():
            valid, details = self._json_schema_validator.validate_path(collection, key, value, schema=schema)
            if not valid and 'bad value' in details:
                print(
                    f"ERROR: Invalid query value for {details['collection']}, path {details['path']}, value {details['bad value']}.")
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break
            elif not valid:
                print(f"ERROR: Invalid query path for {details['collection']}, path {details['path']}.")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break

        return valid, details

    def validate_projection(self, collection: str, query_params: dict, schema: dict = None) -> Tuple[bool, dict]:
        """ Validates a MongoDB projection path and value against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param query_params: A dictionary of query paths and values to validate
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the path or value
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the path or value fails JSON validation and '{}' otherwise.
        """
        valid = True
        details = {}
        for key, value in query_params.items():
            valid, details = self._json_schema_validator.validate_path(collection, key, None, schema=schema)
            if not valid:
                # ERROR - Invalid projection path
                details['message'] = f"Invalid projection path for {details['collection']}, path {details['path']}."
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break
            elif not isinstance(value, int) or value < 0 or value > 1:
                # ERROR - Invalid projection value
                valid = False
                details = {
                    "level": "error",
                    "message": f"Invalid projection value for {collection}, path {key}, value {value}.",
                    "collection": collection,
                    "path": key,
                    "bad value": value
                }
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break

        return valid, details

    def validate_sort(self, collection: str, query_params: dict, schema: dict = None) -> Tuple[bool, dict]:
        """ Validates a MongoDB sort path and value against the JSON schema defined for a collection

        :param collection: The string name of the collection to validate against
        :param query_params: A dictionary of query paths and values to validate
        :param schema: Schema to validate against (overrides collection schema)
        :return
            A tuple (bool, dict), where the boolean returns 'True' if the path or value
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the path or value fails JSON validation and '{}' otherwise.
        """
        valid = True
        details = {}
        for key, value in query_params.items():
            valid, details = self._json_schema_validator.validate_path(collection, key, None, schema=schema)
            if not valid:
                # ERROR - Invalid sort path
                details['message'] = f"Invalid sort path for {details['collection']}, path {details['path']}."
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break
            elif not isinstance(value, int) or (value != -1 and value != 1):
                # ERROR - Invalid sort value
                valid = False
                details = {
                    "level": "error",
                    "message": f"Invalid sort value for {collection}, path {key}, value {value}.",
                    "collection": collection,
                    "path": key,
                    "bad value": value
                }
                print(f"ERROR: {details['message']}")
                err_msg = self._json_schema_validator.json_dumps(details, sort_keys=False)
                err_msg = err_msg.replace("\n", "\nERROR: ")
                print(f"ERROR: details = {err_msg}")
                break

        return valid, details

    def _find(self, database_id: str, collection: str, query: dict, projection=None, sort=None, limit=0, skip=0,
              next=None, one=True):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            if one:
                result = collection.find_one(filter=query, projection=projection)
            else:
                if sort is None or len(sort) == 0:
                    sort = [("_id", 1)]
                if next:
                    if query is None:
                        query = {}
                    if any(map(lambda tup: tup[0] == '_id' and tup[1] == -1, sort)):
                        query["_id"] = {"$lt": next}
                    else:
                        query["_id"] = {"$gt": next}
                result = list(collection.find(filter=query, projection=projection, limit=limit, skip=skip, sort=sort))
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return result

    def _distinct(self, database_id=None, collection=None, query=None, field=None):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            result = collection.distinct(field, query)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return result

    def _update(self, database_id: str, collection: str, query: dict, update_document: dict, upsert: bool, many: bool):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            if many:
                update_result = collection.update_many(filter=query, update=update_document, upsert=upsert)
            else:
                update_result = collection.update_one(filter=query, update=update_document, upsert=upsert)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return update_result

    def _insert(self, database_id: str, collection: str, documents, many: bool):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            if many:
                insert_result = collection.insert_many(documents=documents)
            else:
                insert_result = collection.insert_one(document=documents)
        except pymongo.errors.DuplicateKeyError as e:
            # Allow DuplicateKeyError to pass through for handling in the view
            raise e
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return insert_result

    def _delete(self, database_id: str, collection: str, query: dict, many=False):
        database = self.get_database(database_id)
        collection = database.get_collection(collection)

        try:
            if many:
                collection.delete_many(filter=query)
            else:
                collection.delete_one(filter=query)
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

    def _create_connection(self, parameters_json: dict, default_parameters: dict) -> Tuple[
        Database, MongoServerHeartbeatHandler]:
        """ Connect to the given Mongo server and return a database reference object
            NOTE: Returned database connection should be verified to not be None in case of configuration error

        :param parameters_json: JSON dict of the MongoDB connection information
        :return Database object of the specified Mongo database
        :raises ConfigurationError if final URI is invalid
        """
        default_parameters.update(parameters_json)
        parameters_json = default_parameters

        mongodb_uri = parameters_json['db_uri']
        auth_source = parameters_json['auth_db']
        auth_enable = parameters_json['auth_enable']
        username = parameters_json['username']
        password = parameters_json['password']
        dns_srv = parameters_json['dns_srv']
        host = parameters_json['host']
        port = parameters_json['port']
        name = parameters_json['name']
        replica_set_enable = parameters_json['replica_set_enable']
        replica_set_hosts = parameters_json['replica_set_hosts']
        replica_set_name = parameters_json['replica_set_name']
        tls_enable = parameters_json['tls_enable']
        ca_cert_path = parameters_json['ca_cert_path']

        # Use URI if defined
        if mongodb_uri == '':
            if dns_srv:
                mongodb_uri = "mongodb+srv://"
            else:
                mongodb_uri = "mongodb://"

            options_tokens = []

            # Auth options
            if auth_enable:
                mongodb_uri += "{}:{}@".format(
                    quote_plus(username),
                    quote_plus(password)
                )
                options_tokens.append("authSource={}".format(auth_source))

            # There are three types of connections
            #   Direct Connection   - connect directly to MongoDB server.
            #   Replica Set         - connect to a cluster of MongoDB servers providing redundancy.
            #   DNS Seed List (SRV) - connect using connection information from a DNS SRV record.
            if dns_srv:
                # DNS Seed List (SRV) - use the 'host' field only
                mongodb_uri += "{}".format(host)
            elif replica_set_enable:
                # Replica Set Connections - use the 'replica_set_hosts' field
                # The 'replica_set_hosts' field is a a list of hosts that are part of the Replica Set.
                mongodb_uri += ",".join(replica_set_hosts)
            else:
                # Direct Connection - use the 'host' and 'port' fields
                mongodb_uri += "{}:{}".format(host, port)

            # Append Replica Set information
            if replica_set_enable:
                options_tokens.append("replicaSet={}".format(replica_set_name))

            if tls_enable:
                options_tokens.append("ssl=true")
                options_tokens.append("ssl_ca_certs={}".format(ca_cert_path))
            elif dns_srv:
                # MongoDB automatically enables encryption (ssl=true) with SRV. If encryption is
                # disabled, explicitly disable encryption via the URI.
                options_tokens.append("ssl=false")

            if len(options_tokens) > 0:
                mongodb_uri += "/?"
                mongodb_uri += "&".join(options_tokens)

        mongo_client = None
        heart_beat = None

        # Validate the URI
        try:
            uri_parser.parse_uri(mongodb_uri)

            is_replica_set = replica_set_enable or "replicaSet=" in mongodb_uri
            mongo_server_heartbeat = MongoServerHeartbeatHandler(is_replica_set=is_replica_set, name=name)
            mongo_client = MongoClient(mongodb_uri, event_listeners=[mongo_server_heartbeat]).get_database(name=name)
            heart_beat = mongo_server_heartbeat
        except pymongo.errors.ConfigurationError as e:
            print(f"Failed to connect to MongoDB server at {mongodb_uri}. Database URI error: {e}")
        except FileNotFoundError:
            print(f"Failed to connect to MongoDB server at {mongodb_uri}. CA Certificate file was not found.")

        return mongo_client, heart_beat

    def _perform_migrations(self):
        """ Performing Django User Migration manually (instead of; python manage.py makemigrations(migrate) command(s)) if not already done """
        collections = self.user_database.list_collection_names()

        # SCHEMA
        if not "__schema__" in collections:  # If the collection does not exist, create it
            self.user_database.create_collection('__schema__')
        schema_collection = self.user_database['__schema__']
        for migration_document in self._user_migration_options[
            '__schema__']:  # Looping through all schema documents that need to exist, as defined in 'user_migration.json'
            schema_document = schema_collection.find_one(
                {'name': migration_document['name']})  # Find schema document in MongoDB by name
            if schema_document is None:  # If document was not found, insert it
                schema_collection.insert_one(migration_document)
            if migration_document[
                'name'] != 'django_session':  # Here, begin validating sequence number is correct. django_session schema document does not have a sequence.
                sequenced_collection = self.user_database[migration_document['name']]
                highest_sequence_document = sequenced_collection.find_one({"$query": {}, "$orderby": {"id": -1}}, {
                    "id": 1})  # Based on the name field of the schema document, search the associated collection for the highest 'id' value. Only return the id field as that is all I need. This is where the sequence should start
                if highest_sequence_document is not None:  # Continue if we found a document
                    if type(highest_sequence_document['id']) is int:  # Continue if 'id' is an integer
                        if int(highest_sequence_document["id"]) != schema_document['auto'][
                            'seq']:  # Only update schema document if the sequence value does not match what it should be, the highest value
                            schema_collection.update_one({'name': migration_document['name']}, {"$set": {
                                'auto': {'field_names': ['id'], 'seq': int(highest_sequence_document[
                                                                               "id"])}}})  # Update schema document with highest sequence value

        # SYSLOG-ACTIONS
        if not "SYSLOG-ACTIONS" in collections:
            self.user_database.create_collection('SYSLOG-ACTIONS')
        # This caps the SYSLOG-ACTIONS collection if it is not capped to 50MB
        collection_stats = self.user_database.command("collstats", "SYSLOG-ACTIONS")
        if "capped" not in collection_stats.keys() or not collection_stats["capped"]:
            collection_cap_size = self._COLLECTION_CAP_BYTES_DEFAULT
            settings_doc = self.user_database.get_collection("tibit_settings").find_one({"_id": "Default"},
                                                                                        {"SYSLOG-ACTIONS Max Bytes": 1})
            if settings_doc is not None and settings_doc != {}:
                if settings_doc["SYSLOG-ACTIONS Max Bytes"] is not None:
                    try:
                        collection_cap_size = int(settings_doc["SYSLOG-ACTIONS Max Bytes"])
                    except ValueError:
                        pon_manager_logger.warning(
                            f"Could not parse max collection size for \"SYSLOG-ACTIONS\" collection. Using default of {self._COLLECTION_CAP_BYTES_DEFAULT} bytes.")

            self.user_database.command({"convertToCapped": "SYSLOG-ACTIONS", "size": collection_cap_size})
            pon_manager_logger.info(
                f"Converted collection \"SYSLOG-ACTIONS\" to capped collection with max size of {collection_cap_size} bytes")

        # AUTH_GROUP
        if not "auth_group" in collections:
            self.user_database.create_collection('auth_group')
        collection = self.user_database['auth_group']
        for document in self._user_migration_options['auth_group']:
            exists = collection.find_one(document)
            if exists is None:
                collection.replace_one({'id': document['id']}, document, upsert=True)

        # AUTH_GROUP_PERMISSIONS
        if not "auth_group_permissions" in collections:
            self.user_database.create_collection('auth_group_permissions')
        collection = self.user_database['auth_group_permissions']
        try:
            max_id = collection.find_one(sort=[("id", pymongo.DESCENDING)])["id"] + 1
        except:
            max_id = 1
        for document in self._user_migration_options['auth_group_permissions']:
            exists = collection.find_one({"group_id": document["group_id"], "permission_id": document["permission_id"]})
            if exists is None:
                document["id"] = max_id
                max_id += 1
                collection.insert_one(document)

        # AUTH_PERMISSION
        if not "auth_permission" in collections:
            self.user_database.create_collection('auth_permission')
        collection = self.user_database['auth_permission']
        for document in self._user_migration_options['auth_permission']:
            exists = collection.find_one(document)
            if exists is None:
                collection.replace_one({'id': document['id']}, document, upsert=True)

        # AUTH_USER
        if not "auth_user" in collections:
            self.user_database.create_collection('auth_user')

        # AUTH_USER_GROUPS
        if not "auth_user_groups" in collections:
            self.user_database.create_collection('auth_user_groups')

        # AUTH_USER_USER_PERMISSIONS
        if not "auth_user_user_permissions" in collections:
            self.user_database.create_collection('auth_user_user_permissions')

        # DJANGO_ADMIN_LOG
        if not "django_admin_log" in collections:
            self.user_database.create_collection('django_admin_log')

        # DJANGO_CONTENT_TYPE
        if not "django_content_type" in collections:
            self.user_database.create_collection('django_content_type')
        collection = self.user_database['django_content_type']
        for document in self._user_migration_options['django_content_type']:
            exists = collection.find_one(document)
            if exists is None:
                collection.replace_one({'id': document['id']}, document, upsert=True)

        # DJANGO_MIGRATIONS
        if not "django_migrations" in collections:
            self.user_database.create_collection('django_migrations')
        collection = self.user_database['django_migrations']
        for document in self._user_migration_options['django_migrations']:
            exists = collection.find_one(document)
            if exists is None:
                collection.replace_one({'id': document['id']}, document, upsert=True)

        # DJANGO_SESSION
        if not "django_session" in collections:
            self.user_database.create_collection('django_session')

        # TIBIT_SETTINGS
        if not "tibit_settings" in collections:
            self.user_database.create_collection('tibit_settings')
        collection = self.user_database['tibit_settings']
        for migration_document in self._user_migration_options['tibit_settings']:
            found_document_in_db = collection.find_one({'_id': 'Default'})
            if found_document_in_db is None:
                collection.replace_one({'_id': migration_document['_id']}, migration_document, upsert=True)
                # Migrate old session expiry age if the deprecated record exists
                found_document_in_db = collection.find_one({'_id': 'USER_SESSION_EXPIRY_AGE'})
                if found_document_in_db and found_document_in_db['timeout']:
                    collection.update_one({'_id': migration_document['_id']}, {
                        "$set": {"User Session Expiry Age Timeout": found_document_in_db['timeout']}})
            else:  # Default document exists, need to verify all properties are present
                for key in migration_document:  # Looping through all values that should exist in the document from migration.json
                    if key not in found_document_in_db:  # Check if the document in DB doesn't have key:value pair that it should
                        collection.update_one({'_id': migration_document['_id']},
                                              {"$set": {key: migration_document[key]}})
                    else:  # The property in mongoDB does exist. We need to verify that the property has all the correct properties
                        if type(found_document_in_db[
                                    key]) is dict:  # We only need to verify contents of a property if it's an object/dict
                            for found_document_in_db_property_key in migration_document[
                                key]:  # Looping through all the properties of the nested object/dict
                                if found_document_in_db_property_key not in found_document_in_db[
                                    key]:  # If the nested object/dict is missing a property, insert it
                                    property_name = key + '.' + found_document_in_db_property_key  # Forming the property name to be used in Mongo query
                                    collection.update_one({'_id': migration_document['_id']}, {"$set": {
                                        property_name: migration_document[key][found_document_in_db_property_key]}})

        # TIBIT_MIGRATIONS
        if not "tibit_migrations" in collections:
            self.user_database.create_collection('tibit_migrations')

        # Migrating R1.3.1 and below users to R2.0.0 users
        collection = self.user_database['tibit_migrations']
        user_migration_0001 = collection.find_one({"_id": 'auth_user_1'})
        if user_migration_0001 is None:
            if len(User.objects.filter(groups__name='Administrators')) == 0:
                for user in User.objects.all():
                    user.groups.add(Group.objects.get(name="Administrators"))
                    user.is_staff = False
                    user.save()

            # Creating migration record
            collection = self.user_database['tibit_migrations']
            migration_record = self._user_migration_options['tibit_migrations']['auth_user_1']
            migration_record.update({"applied": datetime.datetime.now()})
            collection.insert_one(migration_record)

        # Migrating R2.2.0 and below users to have lowercase email and username
        user_migration_0002 = collection.find_one({"_id": 'auth_user_2'})
        if user_migration_0002 is None:
            for user in User.objects.all():
                user.email = user.email.lower()
                user.username = user.username.lower()
                user.save()

            # Creating migration record
            collection = self.user_database['tibit_migrations']
            migration_record = self._user_migration_options['tibit_migrations']['auth_user_2']
            migration_record.update({"applied": datetime.datetime.now()})
            collection.insert_one(migration_record)

        # Migrating #R3.1.0 and below groups to have view permissions
        auth_group_permission_migration_0001 = collection.find_one({"_id": 'auth_group_permissions_1'})
        if auth_group_permission_migration_0001 is None:
            group_ids = list(self.user_database['auth_group'].find({}, {"id": 1, "_id": 0}))
            permission_ids = [57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69,
                              70]  # permission_ids for newly added view permissions
            all_group_permissions = self.user_database['auth_group_permissions'].find()
            has_new_permissions = False
            for permission in all_group_permissions:
                if permission["group_id"] != 1 and permission["group_id"] != 2:  # not admin or read only
                    if permission["permission_id"] in permission_ids:
                        has_new_permissions = True
                        break

            if not has_new_permissions:  # Need to add view permissions
                # Grab highest 'id' as to not overwrite any ids
                permission_counter = \
                self.user_database['auth_group_permissions'].find_one(sort=[("id", pymongo.DESCENDING)])["id"] + 1

                for group in group_ids:  # Loop through each existing group/role
                    group_id = group["id"]  # id number of group/role

                    if group_id != 1 and group_id != 2:  # not admin or read only
                        has_admin_read = self.user_database['auth_group_permissions'].find_one(
                            {"group_id": group_id, "permission_id": 41})

                        for permission in permission_ids:  # Loop through new permissions to add
                            if (
                                    permission == 57 and has_admin_read) or permission != 57:  # only add 'admin view' if group has 'read admin'
                                new_permission = {
                                    "id": permission_counter,
                                    "group_id": group_id,
                                    "permission_id": permission
                                }
                                self.user_database['auth_group_permissions'].insert_one(new_permission)
                                permission_counter += 1

            # Creating migration record
            collection = self.user_database['tibit_migrations']
            migration_record = self._user_migration_options['tibit_migrations']['auth_group_permissions_1']
            migration_record.update({"applied": datetime.datetime.now()})
            collection.insert_one(migration_record)

        # R3.2.0 Adding Session Expiry Timeout and Override fields to Auth_Group
        collection = self.user_database['auth_group']
        missing_role_timeout_fields = list(collection.find({"User Session Expiry Age Timeout": {'$exists': False}}))
        if len(missing_role_timeout_fields) > 0:
            global_timeout = self.user_database['tibit_settings'].find_one({'_id': 'Default'})['User Session Expiry Age Timeout']
            update = {"$set": {"User Session Expiry Age Timeout": global_timeout, "User Session Expiry Age Timeout Override": False}}
            collection.update_many({"User Session Expiry Age Timeout": {'$exists': False}}, update, upsert=True)

        pon_manager_logger.info("User database migrations complete")


if BUILDING_DOCUMENTATION:
	print("Skipping Database Manager for Documentation")
	database_manager = None
else:
	database_manager = DatabaseManager()

