"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, inline_serializer
from drf_spectacular.types import OpenApiTypes
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import get_nested_value, PonManagerApiResponse, validate_data, validate_query_params, permission_required_any_of


# ==================================================
# ============= One Switch State View ==============
# ==================================================
class OneState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_switch_state",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'state', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    def get(self, request, swi_id, version):
        """Get the state for the specified Switch"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("OLT-STATE")
        cursor = collection.find({"Switch.Chassis ID": swi_id})
        res_data = None
        if cursor.count() > 0:
            res_data = cursor.sort("Time", pymongo.DESCENDING)[0]

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============== Switch States View ================
# ==================================================
class States(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_switch_states",
        parameters=[
            OpenApiParameter(name="scale_db", description="true/false sort by Time",
                             type=OpenApiTypes.BOOL, required=False),
        ],
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'state', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    def get(self, request, version):
        scale_db = request.GET.get('scale_db', None)
        """Get the states for all Switches"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("OLT-STATE")
        cursor = collection.find({}, {"_id": 0, "Switch": 1, "Time": 1})
        if cursor.count() > 0 and not scale_db:
            cursor = cursor.sort("Time", pymongo.DESCENDING)

        # Duplicate switch states get overwritten by most recent entry
        all_switch_states = {}
        for switch_state in cursor:
            if switch_state["Switch"]:
                switch_id = switch_state["Switch"]["Chassis ID"]
                all_switch_states[switch_id] = switch_state["Switch"]
                all_switch_states[switch_id]["Time"] = switch_state["Time"]
        res_data = all_switch_states.values()

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========= One Switch Configuration View ==========
# ==================================================
class OneConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_switch_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'config', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    def get(self, request, swi_id, version):
        """Get the config for the specified Switch"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="SWI-CFG", query={"_id": swi_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_one_switch_config",
        request=inline_serializer(name="SWI-CFG", fields={"data": JSONField(help_text="SWI-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'config', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_switches', 'network.can_create_network_switches'], raise_exception=True))
    @validate_data(collection="SWI-CFG", resource_id_param="swi_id")
    def put(self, request, data, swi_id, version):
        """Update the config for the specified Switch"""
        data['SWI']['CFG Change Count'] += 1
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SWI-CFG", query={"_id": swi_id}, new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_switch_config",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'config', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_switches', raise_exception=True))
    def delete(self, request, swi_id, version):
        """Delete the config of the specified Switch"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="SWI-CFG", query={"_id": swi_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== Switch Configurations View ============
# ==================================================
class Configurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_switch_configs",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'config', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    @validate_query_params(collection="SWI-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the configs for all Switches"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="SWI-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="SWI-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_switch_config",
        request=inline_serializer(name="SWI-CFG", fields={"data": JSONField(help_text="SWI-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['switch', 'config', 'post']
    )
    @method_decorator(permission_required('network.can_create_network_switches', raise_exception=True))
    @validate_data(collection="SWI-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided Switch config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="SWI-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            swi_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"Switch configuration with id {swi_id} already exists")

        return response


# ==================================================
# ============= One Switch Logs View ===============
# ==================================================
class Logs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_switch_logs",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: ResponseExample(200),
            400: ResponseExample(403),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'logs', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    def get(self, request, swi_id, version):
        """Get the logs of the specified Switch between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'start-time' is required")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("SYSLOG-SWI-{}".format(swi_id.replace(":", "")))
            if end_time is None:
                res_data = list(collection.find({"time": {"$gte": start_time}},
                                                {"_id": 0, "device ID": 0}).limit(10000))
            else:
                res_data = list(collection.find({"time": {"$gte": start_time, "$lte": end_time}},
                                                {"_id": 0, "device ID": 0}).limit(10000))
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    @extend_schema(
        operation_id="delete_one_switch_logs",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'logs', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_switches', raise_exception=True))
    def delete(self, request, swi_id, version):
        """Delete the Logs of the specified Switch"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("SYSLOG-SWI-{}".format(swi_id.replace(":", "")))
        collection.drop()

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ======== One Switch Automation State View ========
# ==================================================
class OneAutomationState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_switch_automation_state",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'state', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    def get(self, request, swi_id, version):
        """Get the Automation State of the specified Switch"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="SWI-AUTO-STATE", query={"_id": swi_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="delete_one_switch_automation_state",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'state', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_switches', raise_exception=True))
    def delete(self, request, swi_id, version):
        """Delete the specified Switch Automation State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="SWI-AUTO-STATE", query={"_id": swi_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== Switch Automation States View =========
# ==================================================
class AutomationStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_switch_automation_states",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'state', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    # TODO @validate_query_params(collection="SWI-AUTO-STATE")
    def get(self, request, version):
        """Get the Automation States of all Switches"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="SWI-AUTO-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ======== One Switch Automation Config View =======
# ==================================================
class OneAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_switch_automation_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'config', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified Switch Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_one_switch_automation_config",
        request=inline_serializer(name="SWI-AUTO-CFG", fields={"data": JSONField(help_text="SWI-AUTO-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'config', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_switches', 'network.can_create_network_switches'], raise_exception=True))
    # TODO @validate_data(collection="SWI-AUTO-CFG", resource_id_param="cfg_id")
    def put(self, request, cfg_id, version):
        """Update the config for the specified Switch Automation Config"""
        data = get_nested_value(request.data, ['data'])
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: <SWI-AUTO-CFG> }'")
        else:
            data['_id'] = cfg_id
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": cfg_id}, new_document=data)
            if old_document is None:
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=old_document)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="delete_one_switch_automation_config",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'config', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_switches', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified Switch Automation Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========= Switch Automation Configs View =========
# ==================================================
class AutomationConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_switch_automation_configs",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'config', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_switches', raise_exception=True))
    # TODO @validate_query_params(collection="SWI-AUTO-CFG")
    def get(self, request, version):
        """Get the Automation Configs of all Switches"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="SWI-AUTO-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_switch_automation_config",
        request=inline_serializer(name="SWI-AUTO-CFG", fields={"data": JSONField(help_text="SWI-AUTO-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['switch', 'automation', 'config', 'post']
    )
    @method_decorator(permission_required('network.can_create_network_switches', raise_exception=True))
    # TODO @validate_data(collection="SWI-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided Switch Automation Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            swi_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"Switch configuration with id {swi_id} already exists")

        return response


# ==================================================
# ====== Global Switch Automation Config View ======
# ==================================================
class GlobalAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_global_switch_automation_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the Global Switch Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": "Global"})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_global_switch_automation_config",
        request=inline_serializer(name="Global SWI-AUTO-CFG", fields={"data": JSONField(help_text="SWI-AUTO-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'config', 'put']
    )
    @method_decorator(permission_required_any_of(['global_config.can_update_automation', 'global_config.can_create_automation'], raise_exception=True))
    # TODO @validate_data(collection="SWI-AUTO-CFG", resource_id_param=None)
    def put(self, request, version):
        """Update the Global Switch Automation Config"""
        data = get_nested_value(request.data, ['data'])
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: <SWI-AUTO-CFG> }'")
        else:
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", query={"_id": "Global"}, new_document=data)
            if old_document is None:
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=old_document)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="post_global_swi_automation_config",
        request=inline_serializer(name="Global SWI-AUTO-CFG", fields={"data": JSONField(help_text="SWI-AUTO-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['swi', 'automation', 'config', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_automation', raise_exception=True))
    # TODO @validate_data(collection="SWI-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided Switch Automation Global Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="SWI-AUTO-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"Switch Automation configuration with id Global already exists")

        return response

    @extend_schema(
        operation_id="delete_global_switch_automation_config",
        parameters=[
            OpenApiParameter(name="step", description="Automation step to delete template from",
                             type=OpenApiTypes.STR, required=True),
            OpenApiParameter(name="name", description="Automation template to delete from the specified step",
                             type=OpenApiTypes.STR, required=True)
        ],
        responses={
            204: ResponseExample(204),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['switch', 'automation', 'config', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_automation', raise_exception=True))
    def delete(self, request, version):
        """Delete the Global Switch Automation Config"""
        step = request.GET.get('step', None)
        name = request.GET.get('name', None)

        # Return missing parameter response if step or name are undefined
        if step is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'step' is required")
        elif name is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'name' is required")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("SWI-AUTO-CFG")
            try:
                if step.upper() == "IDENTIFY":
                    collection.update({'_id': "Global"}, {"$pull": {"IDENTIFY.Mapping": {"Description": name}}})
                else:
                    collection.update({'_id': "Global"}, {"$unset": {f"{step.upper()}.{name}": ""}})
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

        return response
