# MCMS REST API Curl Examples
For ease of use, user details are stored in the _user-info.json_ file for authentication

The examples below store the HTTP headers provided on successful authentication in _cookies.txt_. This file is not needed to be edited and will be created by the login request.

## Usage Notes
Common curl command options used throughout these example cases:

- '-k' must be used if using a self signed certificate with HTTPS
- '-e' sets the request referrer. Required if using HTTPS
- '-i' will display the response headers
- '-X' will specify the HTTP method. GET is the default if '-X' is not used
- '-H' sets the following request header(s)
- '-d' provides body data. You may also pass a JSON file like so: '-d "@my-file.json"'
- '-c' will write any cookies specified in the response "Set-Cookies" headers to the filename provided. Must have permissions to create/write to a file
- '-b' reads the cookies from the file created by using '-c'

All options above and more may be found by running

    curl --help

The following examples use the jquery library jq to edit JSON.
The necessary edits may also be completed by hand.
jq can be installed using the apt package manager.

    apt install jq

## Common Actions

### Setting Variables
The following environment variables are used throughout the following examples.
Update them as needed.

    BASE_URL="https://10.1.20.102/api" \
    DATABASE="Default" \
    OLT="70:b3:d5:52:35:ac" \
    ONU="e8:b4:70:7f:f8:a0" \
    ONU_FW_VERSION="R2.3.0" \
    ONU_FW_FILE="R2.3.0-ONU-FW.bin" \
    OLT_TAG=200 \
    ONU_TAG=25 \
    GPON_ALLOC_ID=1154 \
    GPON_ONU_ID=1 \
    DHCPv4="pass" \
    DHCPv6="pass" \
    DHCP_CIRCUIT_ID="" \
    DHCP_REMOTE_ID="" \
    START_TIME=$(date -u "+%m/%d/%Y%%20%H:%M:%S.%6N" -d '1 hour ago')

### Logging In
    curl -X POST ${BASE_URL}/v1/users/authenticate/ \
    -d @user-info.json \
    -H "Content-Type: application/json" \
    -c cookies.txt -k
    
This request uses the user email and password stored in _user-info.json_ to login to the API server.
The cookies needed for subsequent requests are stored in _cookies.txt_.

For a successful authentication request, a document listing the users information, assigned roles, and permissions is returned.

### Select a Database
    curl -X PUT ${BASE_URL}/v1/databases/selection/ \
    -d "{\"data\":\"${DATABASE}\"}" \
    -H "Content-Type: application/json" \
    -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
    -b cookies.txt -k \
    -e ${BASE_URL}

### Logging Out
    curl -X GET ${BASE_URL}/v1/users/logout/ \
    -b cookies.txt -k

## Common Use Cases
This section presents a selection of examples for some of the more common use cases.
All examples below assume that the user is already logged in and has selected their desired database.
Use the following sequence to run the CURL examples listed in this section.

1. [Log in](#logging-in) to authenticate the user with the web server.

2. (Optional) [Select the database](#select-a-database) base to use for the session.
If skipping this step, the user’s default database (typically ‘Default’) is used for the session.

3. Run one or more examples from the Common Use Cases listed in the sections below.

4. [Log out](#logging-out) of the web server.

### OLT Use Cases

#### List OLTs with Switch Port Information

    curl -X GET ${BASE_URL}/v1/olts/states/?projection=_id=1,Switch.Chassis%20ID=1,Switch.Port%20ID=1 \
    -b cookies.txt -k

#### Get an OLT's statistics
Retrieves the statistics for a given OLT from the start time as well as the OLT's current
state statistics data

1. Get the OLT's statistics

       curl -X GET ${BASE_URL}/v1/olts/stats/${OLT}/?start-time=${START_TIME} \
       -b cookies.txt -k \
       > STATS-OLT-"$(sed 's/://g' <<<"${OLT}")".json

2. Get the OLT's state statistics

       curl -X GET ${BASE_URL}/v1/olts/states/?query=_id=${OLT}\&projection=OLT=1 \
       -b cookies.txt -k \
       > OLT-STATE.json

#### Reset an OLT

    curl -X PUT ${BASE_URL}/v1/olts/${OLT}/reset/ \
    -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
    -b cookies.txt -k \
    -e ${BASE_URL}

### ONU Use Cases

#### List ONUs for an OLT

    curl -X GET ${BASE_URL}/v1/onus/states/?query=OLT.MAC%20Address=${OLT}\&projection=_id=1 \
    -b cookies.txt -k

#### Configure Add CTag Service for an ONU
Configures a double-tagged VLAN service on an ONU, where the OLT adds/removes an outer STag (0x88A8) and the ONU adds/removes an inner CTag (0x8100).
The example configures DHCP Relay and an SLA for the service.

1. Get the OLT Configuration (OLT-CFG)

       curl -X GET ${BASE_URL}/v1/olts/configs/${OLT}/ \
       -b cookies.txt -k \
       > OLT-CFG.json

2. Configure the GPON ONU ID and ALLOC ID for the service by adding the ONU to the OLT’s ONU Inventory

       echo "$(jq --arg onu "${ONU}" --arg allocId "${GPON_ONU_ID}" --arg invVal "${GPON_ALLOC_ID}" '
       if .data.ONUs | has($onu) then . else .data.ONUs[$onu] = {"Disable": false} end |
       .data.ONUs[$onu]."ALLOC ID (OMCC)" = ($allocId|tonumber) |
       .data.ONUs[$onu]."OLT-Service 0" = ($invVal|tonumber)
       ' OLT-CFG.json)" > OLT-CFG.json
       
3. Add the VLAN for the service to the OLT’s NNI Networks configuration

    a. Get the index within the OLT's NNI Networks array of the existing or new entry

       NNI_INDEX=$(echo "$(jq --arg onu "${ONU}" --arg nniTags "s${OLT_TAG}.c${ONU_TAG}.c0" '
       .data as $root |
       [.data."NNI Networks"[]] | map(."TAG MATCH" == $nniTags) | index(true) as $index |
       if $index == null then ($root."NNI Networks" | length) else $index end
       ' OLT-CFG.json)")

    b. Update/insert the NNI Networks entry
    
       echo "$(jq --arg index "${NNI_INDEX}" --arg nniTags "s${OLT_TAG}.c${ONU_TAG}.c0" --arg invVal "${GPON_ALLOC_ID}" --arg v4 "${DHCPv4}" --arg v6 "${DHCPv6}" '
       .data."NNI Networks"[$index|tonumber] = {
       "Filter":
       {
       "EAPOL":"pass",
       "PPPoE":"pass",
       "DHCPv4":$v4,
       "DHCPv6":$v6
       },
       "Learning Limit": 2046,
       "SLA-CFG":{"Source":"N/A"},
       "TAG MATCH":$nniTags,
       "PON FLOOD ID":$invVal,
       }' OLT-CFG.json)" > OLT-CFG.json

4. Save the updated OLT-CFG

       curl -X PUT ${BASE_URL}/v1/olts/configs/${OLT}/ \
       -d @OLT-CFG.json \
       -H "Content-Type: application/json" \
       -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
       -b cookies.txt -k \
       -e ${BASE_URL}

5. Get the ONU Configuration (ONU-CFG)

    _If pre-provisioning an ONU:_
    
    a. Get the Default ONU Configuration (ONU-CFG)
       
       curl -X GET ${BASE_URL}/v1/onus/configs/Default/ \
       -b cookies.txt -k \
       > ONU-CFG.json

    b. Update the ONU-CFG to use the new ID and NETCONF Name

       echo "$(jq --arg onu "${ONU}" '
       .data._id = $onu |
       .data.NETCONF.Name = $onu
       ' ONU-CFG.json)" > ONU-CFG.json

    _If provisioning a service on an existing ONU:_

       curl -X GET ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -b cookies.txt -k \
       > ONU-CFG.json

6. Enable and configure the ONU's service configuration, including VLANs, SLA, and DHCP Relay

       echo "$(jq --arg circuitId "${DHCP_CIRCUIT_ID}" --arg remoteId "${DHCP_REMOTE_ID}" --arg v4 "${DHCPv4}" --arg v6 "${DHCPv6}" --arg oltTag "${OLT_TAG}" --arg onuTag "${ONU_TAG}" --arg olt "${OLT}" '
       .data."OLT-Service 0".Enable = true |
       .data."OLT-Service 0".DHCP."Circuit ID" = $circuitId |
       .data."OLT-Service 0".DHCP."Remote ID" = $remoteId |
       .data."OLT-Service 0".Filter.DHCPv4 = $v4 |
       .data."OLT-Service 0".Filter.DHCPv6 = $v6 |
       .data."OLT-Service 0"."NNI Network" = ["s" + $oltTag + ".c" + $onuTag + ".c0"] |
       .data."OLT-Service 0"."PON Network" = ["s0.c" + $onuTag + ".c0"] |
       .data."OLT-Service 0"."SLA-CFG" = "Min" |
       .data.ONU."SRV-CFG" = "Add CTag" |
       .data.ONU.CVID = ($onuTag|tonumber) |
       .data.OLT."MAC Address" = [$olt]
       ' ONU-CFG.json)" > ONU-CFG.json

7. Save the updated/new ONU-CFG

       curl -X PUT ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -d @ONU-CFG.json \
       -H "Content-Type: application/json" \
       -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
       -b cookies.txt -k \
       -e ${BASE_URL}

#### Disable an ONU service
Disables service on an ONU, which includes removing the ONU from the OLT and disabling and clearing the service configuration such that subscriber traffic is no longer forwarded

1. Get the ONU Configuration (ONU-CFG)

       curl -X GET ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -b cookies.txt -k \
       > ONU-CFG.json

2. Disable and clear the ONU's service configuration

       echo "$(jq '
       .data."OLT-Service 0".Enable = false |
       .data."OLT-Service 0".DHCP."Circuit ID" = "" |
       .data."OLT-Service 0".DHCP."Remote ID" = "" |
       .data."OLT-Service 0".Filter.DHCPv4 = "pass" |
       .data."OLT-Service 0".Filter.DHCPv6 = "pass" |
       .data."OLT-Service 0"."NNI Network" = [] |
       .data."OLT-Service 0"."PON Network" = [] |
       .data."OLT-Service 0"."SLA-CFG" = "Min" |
       .data.ONU."SRV-CFG" = "Disabled" |
       .data.ONU.CVID = 0 |
       .data.OLT."MAC Address" = []
       ' ONU-CFG.json)" > ONU-CFG.json

3. Save the updated ONU-CFG

       curl -X PUT ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -d @ONU-CFG.json \
       -H "Content-Type: application/json" \
       -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
       -b cookies.txt -k \
       -e ${BASE_URL}

4. Get the OLT Configuration (OLT-CFG)

       curl -X GET ${BASE_URL}/v1/olts/configs/${OLT}/ \
       -b cookies.txt -k \
       > OLT-CFG.json

5. Delete the ONU from the OLT's ONU Inventory

       echo "$(jq --arg onu "${ONU}" '
       del(.data.ONUs | .[$onu])
       ' OLT-CFG.json)" > OLT-CFG.json

6. Delete the OLT's NNI Networks for the service

       echo "$(jq --arg tag "s${OLT_TAG}.c${ONU_TAG}.c0" '
       del(.data."NNI Networks"[] | select(."TAG MATCH" == $tag))
       ' OLT-CFG.json)" > OLT-CFG.json

7. Save the updated OLT-CFG

       curl -X PUT ${BASE_URL}/v1/olts/configs/${OLT}/ \
       -d @OLT-CFG.json \
       -H "Content-Type: application/json" \
       -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
       -b cookies.txt -k \
       -e ${BASE_URL}

#### Update an ONU's Firmware
Configure an ONU to upgrade its firmware bank 0 to the new file

1. Get the ONU Configuration (ONU-CFG)

       curl -X GET ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -b cookies.txt -k \
       > ONU-CFG.json

2. Update the ONU-CFG firmware bank to the new file

       echo "$(jq --arg version "${ONU_FW_VERSION}" --arg file "${ONU_FW_FILE}" '
       .data.ONU."FW Bank Versions"[0] = $version |
       .data.ONU."FW Bank Files"[0] = $file
       ' ONU-CFG.json)" > ONU-CFG.json

3. Save the updated ONU-CFG

       curl -X PUT ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -d @ONU-CFG.json \
       -H "Content-Type: application/json" \
       -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
       -b cookies.txt -k \
       -e ${BASE_URL}

#### Activate an ONU's Firmware
Update an ONU's configuration to select the active firmware bank to use

1. Get the ONU Configuration (ONU-CFG)

       curl -X GET ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -b cookies.txt -k \
       > ONU-CFG.json

2. Select the ONU firmware bank to activate

       echo "$(jq '
       .data.ONU."FW Bank Ptr" = 0
       ' ONU-CFG.json)" > ONU-CFG.json

3. Save the updated ONU-CFG

       curl -X PUT ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -d @ONU-CFG.json \
       -H "Content-Type: application/json" \
       -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
       -b cookies.txt -k \
       -e ${BASE_URL}

#### Disable Firmware Upgrade for an ONU
Clear the ONU's firmware configurations and disable the firmware upgrade

1. Get the ONU Configuration (ONU-CFG)

       curl -X GET ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -b cookies.txt -k \
       > ONU-CFG.json

2. Clear all of the ONU's firmware bank and pointer configurations

       echo "$(jq '
       .data.ONU."FW Bank Versions" = ["", ""] |
       .data.ONU."FW Bank Files" = ["", ""] |
       .data.ONU."FW Bank Ptr" = 65535
       ' ONU-CFG.json)" > ONU-CFG.json

3. Save the updated ONU-CFG

       curl -X PUT ${BASE_URL}/v1/onus/configs/${ONU}/ \
       -d @ONU-CFG.json \
       -H "Content-Type: application/json" \
       -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
       -b cookies.txt -k \
       -e ${BASE_URL}
        
#### Get an ONU's statistics
Retrieves the statistics for a given ONU from the start time as well as the ONU's current
state statistics data

1. Get the ONU's statistics

       curl -X GET ${BASE_URL}/v1/onus/stats/${ONU}/?start-time=${START_TIME} \
       -b cookies.txt -k \
       > STATS-ONU-"$(sed 's/://g' <<<"${ONU}")".json

2. Get the ONU's state statistics

       curl -X GET ${BASE_URL}/v1/onus/states/?query=_id=${ONU}\&projection=ONU=1,STATS=1 \
       -b cookies.txt -k \
       > ONU-STATE.json

#### Reset an ONU

    curl -X PUT ${BASE_URL}/v1/onus/${ONU}/reset/ \
    -H "X-CSRFToken:$(grep csrftoken cookies.txt | sed 's/^.*csrftoken\s*//')" \
    -b cookies.txt -k \
    -e ${BASE_URL}
