#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#  Distributed as Tibit-Customer confidential.                             #
#                                                                          #
#--------------------------------------------------------------------------#
""" MCMS REST API Client.

The MCMS REST API Client object is a wrapper around the Python requests
package (https://docs.python-requests.org/en/latest/). The ApiClient class
includes utility methods for login, logout, and database selection and
security cookie handling on behalf of the application.

Sample script structure:

    # Instantiate an API Client Connection.
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server.
    api_client.login(user, password)

    # Select the database to use for this session.
    api_client.select_database(database)

    # Send an API request to configure or retrieve state for a device.
    status, data = api_client.request("GET", "/v1/onus/states/")

    # Logout of the web server to terminate the session.
    api_client.logout()

"""

import json
import re
import urllib.parse
import urllib3
import requests
from simplejson import JSONDecodeError

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

class ApiClient:
    """ API Client

    The API Client class is a wrapper around the Python requests that
    handles the connection to the web server. The ApiClient
    includes utility methods for login, logout, database selection,
    as well as, security cookie handling on behalf of the application.

    """
    def __init__(self, base_url, verbose=False):
        """ API Client constructor.

        Args:
            base_url: The base URL used to access the server (e.g., https://10.2.10.29/api).
            verbose: Enable verbose logging for API requests.
        """
        self.base_url = base_url
        self.verbose = verbose
        self.host = urllib.parse.urlparse(self.base_url).netloc
        self.session_id = None
        self.csrf_token = None
        self.headers = {
            'Content-Type': 'application/json',
            'Referer': self.base_url
        }


    def login(self, user, password):
        """ Log in to the web server and establish an HTTPS session for the application.

        Args:
            user: Username/email address that identifies the API user.
            password: Password for the API user.
        """
        if self.verbose:
            print(f"Authenticating user {user}...")

        # Send a request for authentication
        method = "POST"
        url = "/v1/users/authenticate/"
        response = requests.request(
            method="POST",
            url=f"{self.base_url}{url}",
            headers={'Content-Type': 'application/json'},
            data=json.dumps({
                "email": user,
                "password": password
            }),
            verify=False)

        # Logging
        status_summary = self._status_summary(response.status_code)
        print(f"{method: <5} {url} [ status: {response.status_code} ({status_summary}) ]")

        # Parse the Session ID and CSRF token cookies from the auth response
        cookies = response.headers['Set-Cookie']
        # Debug
        # if self.verbose:
        #     print(f"cookies={cookies}")

        # CSRF Token Cookie - check for https cookies first, http second
        cookie = ""
        if "__Host-csrftoken" in cookies:
            csrf_token = re.search('__Host-csrftoken=([a-zA-Z0-9]*);', cookies).group(1)
            cookie += "__Host-csrftoken=" + csrf_token
        else:
            csrf_token = re.search('csrftoken=([a-zA-Z0-9]*);', cookies).group(1)
            cookie += "csrftoken=" + csrf_token
        self.csrf_token = csrf_token

        # Session ID Cookie - check for https cookies first, http second
        if "__Host-sessionid" in cookies:
            session_id = re.search('__Host-sessionid=([a-zA-Z0-9]*);', cookies).group(1)
            cookie += "; __Host-sessionid=" + session_id
        else:
            session_id = re.search('sessionid=([a-zA-Z0-9]*);', cookies).group(1)
            cookie += "; sessionid=" + session_id
        self.session_id = session_id

        # Add authentication cookie to the headers
        self.headers['Cookie'] = cookie

        # Add CSRF Token to the headers
        if self.csrf_token:
            self.headers['X-CSRFToken'] = self.csrf_token


    def logout(self):
        """ Log out from the web server and terminate the HTTPS session. """
        # Log out the user
        if self.verbose:
            print("Logging out user...")
        return self.request(
            method="GET",
            url="/v1/users/logout/",
            data=None)


    def select_database(self, database_id):
        """ Select a database to use for the session.

        Note: the selection applies to the current active session only.

        Args:
            database_id: Database identifier to be selected.
        """
        return self.request(
            method="PUT",
            url="/v1/databases/selection/",
            data={"data": database_id})


    def request(self, method, url, data=None, extract_response_data=True):
        """ Send an HTTP request to the web server.

        This method sends an HTTP request to the web server
        and waits for a response from the server. The HTTP status
        code and response data are returned to the application.

        Args:
            method: The HTTP method: GET, POST, PUT, DELETE.
            url: The URL identifying the endpoint and resource.
            data: A dictionary containing the data for use in the request body.
            extract_response_data: Return the response data extracted from response_body['data']

        Returns:
            result: True if the write operation succeeds, False otherwise.

        Args:
            status_code: The HTTP status code returned by the web server.
            response_body: Data from the response body returned by the web server.
        """
        # Update the URL with the base_url
        if url[0] == '/':
            request_url = self.base_url + url
        else:
            request_url = self.base_url + '/' + url

        # Send the request
        response = requests.request(
            method=method,
            url=request_url,
            headers=self.headers,
            data=json.dumps(data),
            verify=False)

        try:
            response_body = response.json()
        except JSONDecodeError:
            response_body = response.text

        # Logging
        status_summary = self._status_summary(response.status_code)
        print(f"{method: <5} {url} [ status: {response.status_code} ({status_summary}) ]")
        if self.verbose:
            if data:
                print("Request Body:")
                print(json.dumps(data, indent=4, sort_keys=True))
            if response_body:
                print("Response Body:")
                print(json.dumps(response_body, indent=4, sort_keys=True))

        # Extract the data from the Response Body to make it easier for the
        # application to process.
        if extract_response_data:
            if 'data' in response_body:
                response_body = response_body['data']
            else:
                response_body = None

        return response.status_code, response_body


    def get_ids(self, url, path="data", key="_id"):
        """ Helper function for retrieving the resource IDs for a collection.

        This helper method sends an HTTP request to the web server
        to retrieve the list of resource IDs for the collection specified in
        the URL.

        Args:
            url: The URL identifying the endpoint and collection.
            path: The path in the response data where the IDs are listed.
            key: The name of the field used to identify the resource in the response data.

        Returns:
            ids: List of resource identifiers.
        """
        ids = []
        status, response = self.request(
            method="GET",
            url=f"{url}?projection=_id=1",
            data=None)
        if status == 200:
            data = json.loads(response.text)
            if path and data and path in data:
                data = data[path]
                for item in data:
                    if key in item:
                        ids.append(item[key])
        return ids


    def _status_summary(self, status):
        """ Convert an HTTP status code to a summary string.

        Args:
            status: The HTTP status code returned by the web server.
        """
        status_to_summary_map = {
            200: "OK",
            201: "OK",
            204: "OK",
            400: "FAILED - Bad Request",
            403: "FAILED - Forbidden",
            404: "FAILED - Not Found",
            500: "FAILED - Server Error",
        }
        try:
            status_summary = status_to_summary_map[status]
        except KeyError:
            status_summary = ""
        return status_summary
