#!/usr/bin/env python3
#--------------------------------------------------------------------------#
#  Copyright (c) 2021-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

""" Configure Firmware Upgrade for an ONU.

This Tibit YANG Example script configures firmware upgrade for an ONU. The script
provides examples for activating a specific version of firmware.

Example - Activate the firmware in bank 1:

  ./firmware_upgrade/edit_config_onu_firmware.py --onu ALPHe30cadcf --activate-bank 1


usage: edit_config_onu_firmware.py [--activate-bank ACTIVATE_BANK] [--help]
                                   [-h HOST] --onu ONU [-w PASSWD] [-p PORT]
                                   [-u USER] [-v]

optional arguments:
  --activate-bank ACTIVATE_BANK
                        Activate the firmware in the specified bank. (default:
                        None)
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
from lxml import etree
import os
import sys
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--activate-bank", action="store", dest="activate_bank", default=None, required=True, help="Activate the firmware in the specified bank.")
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    options = {
        "{{ACTIVE_BANK}}" : args.activate_bank,
        "{{ONU}}" : args.onu,
    }

    # Configure the active firmware bank pointer
    nc.edit_config(filename="2-onu-cfg-firmware-bank-ptr.xml", options=options)

    # Display a summary of what was configured
    print(f"\nConfigured Firmware Upgrade for ONU {options['{{ONU}}']}")
    print(f" Firmware Bank Pointer:  {options['{{ACTIVE_BANK}}']}")
