#!/usr/bin/env python3
#--------------------------------------------------------------------------#
#  Copyright (c) 2020-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

""" Get optical levels for an ONU.

This Tibit YANG Example script retrieves optical levels for an ONU.

Example:

  ./get_onu_optical_levels.py --onu ALPHe30cadcf


usage: get_onu_optical_levels.py [--help] [-h HOST] --onu ONU [-w PASSWD]
                                 [-p PORT] [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys
from lxml import etree
from netconf_driver import NetconfDriver

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    options = {
        "{{ONU}}" : args.onu
    }

    # Send a Netconf <get> request to retreive the optical measurements for the ONU
    ONU_OPTICAL_LEVELS = '''
    <rpc
        xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
        xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db"
        message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:pon-state>
                    <tibitcntlr:onu-state>
                        <tibitcntlr:onu>
                            <name>{{ONU}}</name>
                            <tibitcntlr:statistics>
                                <tibitcntlr:olt-pon/>
                                <tibitcntlr:onu-pon/>
                            </tibitcntlr:statistics>
                        </tibitcntlr:onu>
                    </tibitcntlr:onu-state>
                </tibitcntlr:pon-state>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=ONU_OPTICAL_LEVELS, options=options, message="/tibit-pon-controller-db::pon-state/tibit-pon-controller-db:onu-state/tibit-pon-controller-db:onu/tibit-pon-controller-db:statistics")

    # Parse the Netconf response and retrieve the OLT PON statistics from the XML response data.
    olt_optical_levels = {}
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'tibitcntlr' : "urn:com:tibitcom:ns:yang:controller:db",
            }
        root = etree.fromstring(rsp_xml)
        olt_pon_stats = root.findall("nc:data/tibitcntlr:pon-state/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:statistics/tibitcntlr:olt-pon/*", namespaces=NSMAP)
        if olt_pon_stats is not None:
            for stat in olt_pon_stats:
                # Strip the XML namespace from the tag
                tag = stat.tag.split("}")[1][0:]
                olt_optical_levels[tag] = stat.text

    # Parse the Netconf response and retrieve the ONU PON statistics from the XML response data.
    onu_optical_levels = {}
    if rsp_xml:
        onu_pon_stats = root.findall("nc:data/tibitcntlr:pon-state/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:statistics/tibitcntlr:onu-pon/*", namespaces=NSMAP)
        if onu_pon_stats is not None:
            for stat in onu_pon_stats:
                # Strip the XML namespace from the tag
                tag = stat.tag.split("}")[1][0:]
                onu_optical_levels[tag] = stat.text

    # Display the optical levels for the ONU
    print(f"\nOptical Levels for ONU {args.onu}:")
    print("Downstream:")
    if olt_optical_levels and onu_optical_levels:
        print(f"  OLT TX: {olt_optical_levels['tx-optical-level']:>{7}s} dB")
        print(f"  ONU RX: {onu_optical_levels['rx-optical-level']:>{7}s} dB")
    print("Upstream:")
    if olt_optical_levels and onu_optical_levels:
        print(f"  OLT RX: {olt_optical_levels['rx-optical-level']:>{7}s} dB")
        print(f"  ONU TX: {onu_optical_levels['tx-optical-level']:>{7}s} dB")