#!/usr/bin/env python3
#--------------------------------------------------------------------------#
#  Copyright (c) 2022-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

""" BBF YANG Notifications Example

This script script subscribes to notifications from the Netconf Server and
listens for BBF onu-presence-state-change change events.

For reference, an example of a raw onu-presence-state-change event is shown below.

<notification xmlns="urn:ietf:params:xml:ns:netconf:notification:1.0">
  <eventTime>2021-01-28T14:19:57Z</eventTime>
  <interfaces-state xmlns="urn:ietf:params:xml:ns:yang:ietf-interfaces">
    <interface>
      <name>channeltermination.1/0/1</name>
      <type xmlns:bbf-xponift="urn:bbf:yang:bbf-xpon-if-type">bbf-xponift:channel-termination</type>
      <channel-termination xmlns="urn:bbf:yang:bbf-xpon">
        <onu-presence-state-change xmlns="urn:bbf:yang:bbf-xpon-onu-state">
          <detected-serial-number>BFWS00123193</detected-serial-number>
          <last-change>2021-01-28T14:19:57.545976Z</last-change>
          <onu-presence-state xmlns:bbf-xpon-onu-types="urn:bbf:yang:bbf-xpon-onu-types">bbf-xpon-onu-types:onu-present</onu-presence-state>
          <onu-id>1</onu-id>
          <detected-registration-id></detected-registration-id>
          <v-ani-ref>vani-BFWS00123193</v-ani-ref>
        </onu-presence-state-change>
      </channel-termination>
    </interface>
  </interfaces-state>
</notification>

"""

import argparse
import sys
import json
from lxml import etree
import ncclient
from ncclient import manager
import os
from getpass import getpass
from datetime import datetime, timezone
import time
from netconf_driver import NetconfDriver

if __name__ == '__main__':

    # Example: ./notification_listener.py

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, attempt to read it from .nc_edit_auth.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    # Connect to the Netconf Server
    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)

    # Subscribe to notifications from the Netconf Server
    nc.subscribe()

    # Listen for notification events (type CTRL-C to terminate)
    while True:
        # Check for connection failures
        if not nc.is_connected():
            print("ERROR: Connection to server failed.")
            filter = ("subtree", '<pon-onu-state-change xmlns="urn:com:tibitcom:ns:yang:controller:db"/>')
            nc.subscribe(stream="NETCONF", filter=filter)
            time.sleep(1)
        else:
            # Get the next notification
            notification = nc.get_notification()
            if not notification:
                continue

            if args.verbose:
                print("\nNOTIFICATION:")
                print(etree.tostring(notification.notification_ele, pretty_print=True, encoding="unicode").rstrip())

            NSMAP = {
                'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
                'ncEvent' : "urn:ietf:params:xml:ns:netconf:notification:1.0",
                'tibitcntlr' : "urn:com:tibitcom:ns:yang:controller:db",
            }

            # Get the etree node for the event notification
            #import pdb; pdb.set_trace()
            event_node = notification.notification_ele

            # Get the event time
            event_time = ""
            if event_node is not None:
                node = event_node.find("ncEvent:eventTime", namespaces=NSMAP)
                if node is not None:
                    event_time = node.text

            # Convert the UTC time to local time
            if event_time:
                try:
                    event_utc_time = datetime.strptime(event_time, "%Y-%m-%dT%H:%M:%SZ")
                    event_time = event_utc_time.replace(tzinfo=timezone.utc).astimezone(tz=None).strftime("%Y-%m-%d %H:%M:%S")
                except ValueError:
                    # Ignore date/time conversion failures
                    pass

            #
            # OLT Device State Change Events
            #
            # Get the OLT ID
            olt_id = None
            if event_node is not None:
                node = event_node.find("tibitcntlr:pon-olt-state-change/tibitcntlr:olt-id", namespaces=NSMAP)
                if node is not None:
                    olt_id = node.text

            if olt_id:
                # Get the PON Controller ID
                controller_id = None
                if event_node is not None:
                    node = event_node.find("tibitcntlr:pon-olt-state-change/tibitcntlr:controller-id", namespaces=NSMAP)
                    if node is not None:
                        controller_id = node.text

                # Get the OLT Device State
                olt_state = None
                if event_node is not None:
                    node = event_node.find("tibitcntlr:pon-olt-state-change/tibitcntlr:state", namespaces=NSMAP)
                    if node is not None:
                        olt_state = node.text

                # Print the OLT State Change Event
                print(f"\n{event_time} OLT Device State Change Event")
                print(f"  OLT = {olt_id}")
                print(f"  PON Controller = {controller_id}")
                print(f"  State = {olt_state}")


            #
            # ONU Device State Change Events
            #
            # Get the ONU ID
            onu_id = None
            if event_node is not None:
                node = event_node.find("tibitcntlr:pon-onu-state-change/tibitcntlr:onu-id", namespaces=NSMAP)
                if node is not None:
                    onu_id = node.text

            if onu_id:
                # Get the PON Controller ID
                controller_id = None
                if event_node is not None:
                    node = event_node.find("tibitcntlr:pon-onu-state-change/tibitcntlr:controller-id", namespaces=NSMAP)
                    if node is not None:
                        controller_id = node.text

                # Get the OLT ID
                olt_id = None
                if event_node is not None:
                    node = event_node.find("tibitcntlr:pon-onu-state-change/tibitcntlr:olt-id", namespaces=NSMAP)
                    if node is not None:
                        olt_id = node.text

                # Get the ONU Device State
                onu_state = None
                if event_node is not None:
                    node = event_node.find("tibitcntlr:pon-onu-state-change/tibitcntlr:state", namespaces=NSMAP)
                    if node is not None:
                        onu_state = node.text

                # Print the OLT State Change Event
                print(f"\n{event_time} ONU Device State Change Event")
                print(f"  ONU = {onu_id}")
                print(f"  PON Controller = {controller_id}")
                print(f"  OLT = {olt_id}")
                print(f"  State = {onu_state}")

            # Debug Logging
            if args.verbose:
                print("---\n")


