#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2021 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

import argparse
import sys
import json
from lxml import etree
import ncclient
from ncclient import manager
import os
from getpass import getpass
import time

class NetconfDriver:
    def __init__(self, host="127.0.0.1", port=830, user=None, passwd=None, hostkey_verify=False, verbose=False):
        self.host = host
        self.port = port
        self.user = user
        self.passwd = passwd
        self.hostkey_verify = hostkey_verify
        self.verbose = verbose
        self.conn = None

    def _process_options(self, data_xml, options):

        for key, value in options.items():
            data_xml = data_xml.replace(f"{key}", str(value))

        return data_xml

    def connect(self):
        # Connect to the NETCONF Server
        # Attempt with the host key + 3 retries asking for password
        rc = False
        print(f"Connecting to Netconf server {self.host}:{self.port}")
        max_retries = 4
        for retry in range(0,max_retries):
            password=self.passwd
            if retry > 0:
                password = getpass()
            try:
                self.conn = manager.connect(host=self.host, port=self.port, username=self.user, password=password, hostkey_verify=False)
                # Break out of the retry loop on success
                break
            except ncclient.transport.errors.AuthenticationError:
                if retry == 0:
                    print("SSH key authentication failed. Attempting password authentication.")
                else:
                    print("SSH password authentication failed.")
                if retry == max_retries-1:
                    print("ERROR: SSH authentication failed.")
                    sys.exit()
                pass
            except:
                # Server connection error
                print("ERROR: Connect failed.")
                break

        return rc

    def is_connected(self):
        rc = False
        if self.conn:
            if self.conn.connected:
                rc = True
        return rc

    def close(self):
        # Close the connection to the the NETCONF Server
        self.conn.close()
        self.conn = None

    def edit_config(self, data_xml=None, filename=None, options=None, message=None, retries=4, timeout_ms=1000):
        # Check the connection to the Netconf server
        if not self.is_connected():
            self.connect()

        if filename:
            with open("{}/{}".format(os.path.dirname(os.path.realpath(sys.argv[0])),filename), 'r') as f:
                # Read in the XML from the file
                data_xml = f.read()
                print("Sending  <edit-config> from '{}'.".format(filename))
        else:
            print("Sending  <edit-config> for '{}'.".format(message))

        # Send the <edit-config> request to the Netconf Server and process the response
        #
        # Retry the <edit-config> request if a failure is returned. The retry is required to
        # work around an issue in the R2.0 Netconf Server that may return an error on an
        # modify/update request that immediately follows a create request for the same device.
        if data_xml:
            for retry in range(1,retries+1):
                # Update the XML with options from the script command line
                data_xml = self._process_options(data_xml, options)

                # Parse the XML and find the <edit-config> node
                NSMAP = {'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0"}
                root = etree.fromstring(data_xml)
                # Debug Logging
                if self.verbose:
                    print("REQUEST:")
                    print(etree.tostring(root, pretty_print=True, encoding="unicode").rstrip())
                config = root.find("nc:edit-config", namespaces=NSMAP)
                if config is not None and len(config):
                    config_xml = etree.tostring(config, pretty_print=True, encoding="unicode")
                    if config_xml:
                        print("# Send the <edit-config> RPC to the server")
                        try:
                            rsp = self.conn.dispatch(ncclient.xml_.to_ele(config_xml)).xml
                            # Debug Logging
                            if self.verbose:
                                print("RESPONSE:")
                                root = etree.fromstring(bytes(rsp, encoding='utf-8'))
                                print(etree.tostring(root, pretty_print=True, encoding='unicode').rstrip())
                            print("Success.")
                            # Exit retry loop
                            break
                        except ncclient.operations.rpc.RPCError as err:
                            if retry < retries:
                                print(f"Retrying <edit-config> request ({retry})")
                                time.sleep(timeout_ms/1000)
                            else:
                                print("ERROR: <edit-config> failed; err={}".format(err))
        # Debug Logging
        if self.verbose:
            print("---\n")

    def get(self, data_xml=None, filename=None, options=None, message=None):
        # Check the connection to the Netconf server
        if not self.is_connected():
            self.connect()

        rsp_xml = None
        if filename:
            with open("{}/{}".format(os.path.dirname(__file__),filename), 'r') as f:
                # Read in the XML from the file
                data_xml = f.read()
                message = filename

        print("Sending <get> for '{}'.".format(message))

        # Send the <get> request to the Netconf Server and process the response
        if data_xml:
            # Update the XML with options from the script command line
            data_xml = self._process_options(data_xml, options)
            # Parse the XML and find the <edit-config> node
            NSMAP = {'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0"}
            root = etree.fromstring(data_xml)
            # Debug Logging
            if self.verbose:
                print("REQUEST:")
                print(etree.tostring(root, pretty_print=True, encoding="unicode").rstrip())
            config = root.find("nc:get", namespaces=NSMAP)
            if config is not None and len(config):
                config_xml = etree.tostring(config, pretty_print=True, encoding="unicode")
                if config_xml:
                    # Send the <edit-config> RPC to the server
                    try:
                        rsp = self.conn.dispatch(ncclient.xml_.to_ele(config_xml)).xml
                        root = etree.fromstring(bytes(rsp, encoding='utf-8'))
                        rsp_xml = etree.tostring(root, pretty_print=True, encoding='unicode')
                        # Debug Logging
                        if self.verbose:
                            print("RESPONSE:")
                            print(rsp_xml.rstrip())
                        print("Success.")
                    except ncclient.operations.rpc.RPCError as err:
                        print("ERROR: <get> failed; err={}".format(err))

        # Debug Logging
        if self.verbose:
            print("---\n")

        return rsp_xml

    def rpc(self, data_xml, options):

        # Connect to the NETCONF Server
        with manager.connect(host=self.host, port=self.port, username=self.user, password=self.passwd, hostkey_verify=False) as conn:
            # Update the XML with options from the script command line
            data_xml = self._process_options(data_xml, options)

            # Debug
            if options.verbose:
                print(data_xml)

            # Parse the XML and find the <edit-config> node
            NSMAP = {
                'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
                'yang-1_1' : "urn:ietf:params:xml:ns:yang:1"
            }
            root = etree.fromstring(data_xml)
            config = root.find("nc:rpc", namespaces=NSMAP)
            if config is not None and len(config):
                config_xml = etree.tostring(config, pretty_print=True, encoding="unicode")
                if config_xml:
                    # Debug Logging
                    if self.verbose:
                        print("<rpc>:")
                        print(config_xml)
                    # Send the <edit-config> RPC to the server
                    try:
                        rsp = conn.dispatch(ncclient.xml_.to_ele(config_xml)).xml
                        if self.verbose:
                            print("<rpc-reply>:")
                            root = etree.fromstring(bytes(rsp, encoding='utf-8'))
                            print(etree.tostring(root, pretty_print=True, encoding='unicode'))
                        print("Success.")
                    except ncclient.operations.rpc.RPCError as err:
                        print("ERROR: rpc failed; err={}".format(err))

    def subscribe(self, filter_xml=None):
        # Check the connection to the Netconf server
        if not self.is_connected():
            self.connect()

        # Subscribe to notifications from the Netconf server
        if self.conn:
            try:
                self.conn.create_subscription()
                print("\nListening for event notifications...")
            except:
                print("ERROR: Create subscription failed.")

    def get_notification(self):
        return self.conn.take_notification(block=True,timeout=1)
