#!/usr/bin/env python3
# --------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# --------------------------------------------------------------------------#

# Usage: ./merge_collections.py --db <mongodb_uri>/<database>
# e.g., ./merge_collections.py --db mongodb://10.2.10.222:27017/tibit_pon_controller

import sys
import json
import time
import pymongo
from bson.json_util import dumps
from pymongo.write_concern import WriteConcern
import argparse
import csv

parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
parser.add_argument("-d",
                    "--db",
                    dest="db_uri",
                    action="store",
                    default='mongodb://127.0.0.1:27017/tibit_pon_controller',
                    required=False,
                    help="Database uri (e.g., mongodb://127.0.0.1:27017/tibit_pon_controller)")
args = parser.parse_args()

# Default max collection size in R3.1.0 for syslog and stats
MAX_COLL_SIZE = 1000000000

try:
    # Connect to the MongoDB Server as specified in the mongodb:// connection URI
    db_client = pymongo.MongoClient(args.db_uri)
    print("{}".format(db_client))

    # Get a handle to the database reference in the mongodb:// connection URI
    db = db_client.get_default_database()
    print("{}".format(db))

    # All collections in DB
    all_collections = db.list_collection_names()

    # Get the old stat and syslog collection names
    # OLT
    olt_stat_colls = list(filter(lambda x: "STATS-OLT-" in x, all_collections))
    olt_syslog_colls = list(filter(lambda x: "SYSLOG-OLT-" in x, all_collections))
    # ONU
    onu_stat_colls = list(filter(lambda x: "STATS-ONU-" in x, all_collections))
    onu_syslog_colls = list(filter(lambda x: "SYSLOG-ONU-" in x, all_collections))
    # Controller
    cntl_stat_colls = list(filter(lambda x: "STATS-CNTL-" in x, all_collections))
    cntl_syslog_colls = list(filter(lambda x: "SYSLOG-CNTL-" in x, all_collections))
    # PON Auto
    pa_stat_colls = list(filter(lambda x: "STATS-AUTO-" in x, all_collections))
    pa_syslog_colls = list(filter(lambda x: "SYSLOG-AUTO-" in x, all_collections))


    def transfer_collections(collections, new_coll_name, type):
        # Loops through all the old collections of a given type, adds a field with the devices name and places
        # the docs in a temporary collection for sorting and merging.
        for collection_name in collections:
            deviceArr = collection_name.split('-')[2:]
            device = '-'.join(deviceArr)
            db[collection_name].aggregate([
                {"$addFields": {"device ID": device, "valid": True}},
                {"$out": "temp"}
            ])

            # Ignores duplicates
            db["joined"].with_options(write_concern=WriteConcern(w=0)).insert_many(db["temp"].find(), ordered=False)

        # Drop the temporary collection
        db["temp"].drop()

        # If the new collection already exists then add it to the temporary join collection
        if new_coll_name in all_collections:
            db["joined"].with_options(write_concern=WriteConcern(w=0)).insert_many(db[new_coll_name].find(), ordered=False)
            db[new_coll_name].drop()

        print(("Sorting - {}").format(new_coll_name))

        # Sort the documents by Time field
        # Creates the new collection and inserts all sorted documents
        if type == "stat":
            db["joined"].create_index("Time")
            db[new_coll_name].insert_many(db["joined"].find().sort("Time", pymongo.ASCENDING))
        else:
            db["joined"].create_index("time")
            db[new_coll_name].insert_many(db["joined"].find().sort("time", pymongo.ASCENDING))

        db["joined"].drop()

        # Cap the collection
        db.command('convertToCapped', new_coll_name, size=MAX_COLL_SIZE)


    if len(olt_stat_colls) > 0:
        print(("Transferring OLT stats from - {} - collections").format(len(olt_stat_colls)))
        transfer_collections(olt_stat_colls, "STATS-OLT", "stat")

    if len(olt_syslog_colls) > 0:
        print(("Transferring OLT system logs from - {} - collections").format(len(olt_syslog_colls)))
        transfer_collections(olt_syslog_colls, "SYSLOG-OLT", "log")

    if len(onu_stat_colls) > 0:
        print(("Transferring ONU stats from - {} - collections").format(len(onu_stat_colls)))
        transfer_collections(onu_stat_colls, "STATS-ONU", "stat")

    if len(onu_syslog_colls) > 0:
        print(("Transferring ONU system logs from - {} - collections").format(len(onu_syslog_colls)))
        transfer_collections(onu_syslog_colls, "SYSLOG-ONU", "log")

    if len(cntl_stat_colls) > 0:
        print(("Transferring CNTL stats from - {} - collections").format(len(cntl_stat_colls)))
        transfer_collections(cntl_stat_colls, "STATS-CNTL", "stat")

    if len(cntl_syslog_colls) > 0:
        print(("Transferring CNTL system logs from - {} - collections").format(len(cntl_syslog_colls)))
        transfer_collections(cntl_syslog_colls, "SYSLOG-CNTL", "log")

    if len(pa_stat_colls) > 0:
        print(("Transferring AUTO stats from - {} - collections").format(len(pa_stat_colls)))
        transfer_collections(pa_stat_colls, "STATS-AUTO", "stat")

    if len(pa_syslog_colls) > 0:
        print(("Transferring AUTO system logs from - {} - collections").format(len(pa_syslog_colls)))
        transfer_collections(pa_syslog_colls, "SYSLOG-AUTO", "log")


except Exception as err:
    print("An exception has occurred!")
    print(err)
    db["joined"].drop()
    db["temp"].drop()
else:
    print("************* Collection transfers sucessful! *************")
