"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#

Maps url endpoints to database connector
"""

import json
import os
import traceback
import tarfile
import threading
import hashlib

import django.contrib.auth.models
from django.contrib.auth.mixins import LoginRequiredMixin
from django.http import JsonResponse, FileResponse
from django.http import HttpResponse
from django.contrib import messages

# Djongo imports
# from django.contrib.auth.models import User, Permission, ContentType, Group
# from django.contrib.sessions.models import Session

# MongoEngine imports
from api.models import User, Group, Permission, ContentType
from backends.mongo_sessions import MongoSession
from mongoengine import DoesNotExist

from django.contrib.auth import authenticate, login, logout, update_session_auth_hash
from djongo.sql2mongo import SQLDecodeError
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework import status, generics
from django.conf import settings
import django.core.serializers as serializers
import uuid
from datetime import datetime, timedelta, timezone
from django.core.mail import send_mail
import sys
from django.forms.models import model_to_dict
from django.template import loader
from django.contrib.auth.hashers import make_password, check_password
from database_manager import database_manager
from zxcvbn import zxcvbn

from log.PonManagerLogger import pon_manager_logger
from utils.tools import PonManagerApiResponse

sys.path.append('DBdjango/')  # Allows use of module outside of IDE
from DBdjango.database_connector import MongoDBConnector
from utils.tools import get_nested_value, validate_data
from manage import BUILDING_DOCUMENTATION

dbs = {}
sessionPurgeTime = datetime.now()
radiusUserPurgeTime = datetime.now()
recoveryCodePurgeTime = datetime.now()
development_databases_files = "databases.json"
development_omci_info = "omci.json"
development_search_states = [
    "searchColls/CNTL-STATE.json",
    "searchColls/OLT-STATE.json",
    "searchColls/ONU-STATE.json",
    "searchColls/ONU-MIB-CUR-STATE.json",
    "searchColls/ONU-MIB-RST-STATE.json",
    "searchColls/CPE-STATE.json",
    "searchColls/CNTL-STATS.json",
    "searchColls/OLT-STATS.json",
    "searchColls/ONU-STATS.json",
]
production_databases_files = "/var/www/html/api/databases.json"
production_omci_info = "/var/www/html/api/omci.json"
production_search_states = [
    "/var/www/html/api/searchColls/CNTL-STATE.json",
    "/var/www/html/api/searchColls/OLT-STATE.json",
    "/var/www/html/api/searchColls/ONU-STATE.json",
    "/var/www/html/api/searchColls/ONU-MIB-CUR-STATE.json",
    "/var/www/html/api/searchColls/ONU-MIB-RST-STATE.json",
    "/var/www/html/api/searchColls/CPE-STATE.json",
    "/var/www/html/api/searchColls/CNTL-STATS.json",
    "/var/www/html/api/searchColls/OLT-STATS.json",
    "/var/www/html/api/searchColls/ONU-STATS.json",
]
PASSCODE_TIME_LIMIT = 10
DAYS_TILL_NEXT_PURGE = 1
DAYS_TILL_RADIUS_PURGE = 3
use_mongo_engine = True
if 'django.contrib.auth.backends.ModelBackend' in settings.AUTHENTICATION_BACKENDS:
    use_mongo_engine = False


def start_up():
    """Creates an instance of MongoDBConnector for each database defined in databases.json"""
    all_dbs = database_manager.list_databases()
    for db in list(dbs.keys()):
        if db not in all_dbs:
            del dbs[db]
    for db in all_dbs:
        dbs[db] = MongoDBConnector(db)
    return True


if BUILDING_DOCUMENTATION:
    print('Skipping Startup for Documentation')
else:
    start_up()


# Returns the ID of the database that the given client is using
def get_client_db(db, client):
    try:
        if db not in dbs:
            if db not in database_manager.list_databases():
                db = database_manager.get_users_selected_database(client)
            if db not in dbs:
                dbs[db] = MongoDBConnector(db)
    except KeyError:
        db = "Default"

    return dbs[db]


# Purges expired sessions IF one day or more has elapsed since last purging. Triggered by API call(s)
def purge_expired_sessions(request):
    global sessionPurgeTime
    if datetime.now() >= sessionPurgeTime:
        request.session.clear_expired()
        sessionPurgeTime = datetime.now() + timedelta(
            days=DAYS_TILL_NEXT_PURGE)  # Updating purge time to be one day from now

        try:
            request.session.update({"database": ""})
        except KeyError:
            pass


# Purge radius users that don't have an active session
def purge_inactive_radius_users(request):
    global radiusUserPurgeTime
    if datetime.now() >= radiusUserPurgeTime:
        request.session.clear_expired()
        radiusUserPurgeTime = datetime.now() + timedelta(
            days=DAYS_TILL_RADIUS_PURGE)  # Updating purge time to be three days from now
        try:
            database_manager.purge_radius_users()
        except Exception as err:
            pon_manager_logger.error(f'Error purging Radius users. {err}')


def purge_inactive_recovery_code():
    """
    Deletes all recovery code documents more than 10 minutes old.
    Runs once every 24 hours and is triggered when someone logs in.
    """
    global recoveryCodePurgeTime
    try:
        currentDate = datetime.now()
        if currentDate >= recoveryCodePurgeTime:
            user_database = database_manager.user_database
            collection = user_database["user_recovery"]
            query = {"Time": {"$lt": (currentDate - timedelta(minutes=PASSCODE_TIME_LIMIT))}}
            collection.delete_many(query)
            recoveryCodePurgeTime = currentDate + timedelta(
                days=DAYS_TILL_NEXT_PURGE)  # Updating purge time to be one day from now
    except Exception as err:
        pon_manager_logger.error(f'Error purging inactive recovery code. {err}')


# Returns dictionary for Headers to be added to HTTP Response
def set_response_headers(request):
    cookie_id = "__Host-sessionexpire="
    secure_flag = "Secure"

    if "__Host-" not in settings.CSRF_COOKIE_NAME:
        cookie_id = "sessionexpire="
        secure_flag = ""
    headers = {'set-cookie': cookie_id + request.session.get_expiry_date().strftime(
        '%Y-%m-%dT%H:%M:%SZ') + "; expires=" + request.session.get_expiry_date().strftime(
        '%a, %d %b %Y %H:%M:%S GMT') + "; Max-Age=31449600; Path=/; SameSite=Strict; " + secure_flag}
    return headers


# Create your views here.
# Django HTTP status codes reference: https://www.django-rest-framework.org/api-guide/status-codes/

class GetAllOnuLabels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_labels(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltLabels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_labels(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetFirstAssignedOltFirmware(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, firmware):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_first_assigned_olt_firmware(request.user.email, firmware, request.GET.get('limit', 1))
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllControllerLabels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_cntl_labels(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSwitchLabels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_all_switch_labels(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSwitchRegionsPops(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_all_switch_regions_pops(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitchRegionPop(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_switch_region_pop(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


UPLOAD_LOGO_IMAGE_SCHEMA = {
    "properties": {
        "filename": {
            "type": "string",
            "enum": ["corner-logo.png", "footer-logo.png", "login-background.jpg", "footer-text.txt"]
        },
        "val": {
            "type": "string"
        }
    },
    "required": [
        "filename",
        "val"
    ]
}


class UploadLogoImage(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="tibit_branding", resource_id_param=None, validate_required=True,
                   schema=UPLOAD_LOGO_IMAGE_SCHEMA)
    def post(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_create_other_branding" in db.get_user_permissions(request.user.email):
            response = db.upload_logo_image(data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class GetControllerSummary(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if 'can_read_network_controllers' in db.get_user_permissions(request.user.email):
            response = db.get_cntl_summary(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltSummary(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if 'can_read_network_olts' in db.get_user_permissions(request.user.email):
            response = db.get_olt_summary(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuSummary(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if 'can_read_network_onus' in db.get_user_permissions(request.user.email):
            response = db.get_onu_summary(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPonAutoStateCount(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if 'can_read_automation' in db.get_user_permissions(request.user.email):
            response = db.get_pon_auto_state_count()
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPonAutoServicesForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if 'can_read_automation' in db.get_user_permissions(request.user.email):
            response = db.get_pon_auto_services_for_tree()
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPonAutoStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if 'can_read_automation' in db.get_user_permissions(request.user.email):
            response = db.get_pon_auto_status(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class GetMgmtLansForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_mgmt_lans_for_tree(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlsForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mgmt_lan):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_cntls_for_tree(request.user.email, mgmt_lan)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitchRegionsForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_regions_for_tree(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPopsUnderRegionForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, region):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_pops_under_region_for_tree(request.user.email, region)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitchesUnderPopForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, region, pop):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_switches_under_pop_for_tree(request.user.email, region, pop)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsUnderSwitchForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, switch_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_olts_under_switch_for_tree(request.user.email, switch_id)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsUnderControllerForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, controller_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_olts_under_controller_for_tree(request.user.email, controller_id)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnusUnderOltForTree(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onus_under_olt_for_tree(request.user.email, olt_id)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltModels(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            response = db.get_olt_compatibility_data(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            response = db.get_all_olts(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltFirmwareMetaData(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_firmwares_info(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuFirmwareMetaData(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_firmwares_info(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsForUnknownSwitch(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olts_for_unknown_switch(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetCntls(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_controllers(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_cntl_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsForSwitch(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olts_for_switch(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOltsInSwitchInventory(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_all_olts_in_switch_inventory(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltsForController(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olts_for_controller(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltsWithVersions(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olts_with_versions(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitches(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_all_switches(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSwitchesWithVersions(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_all_switches_with_versions(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetSwitchesForCntl(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_all_switches_for_cntl(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltDebugState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.download_olt_debug_state(mac_address, request.user.email)

            if response[0] == status.HTTP_200_OK:
                return FileResponse(json.dumps(response[1]), status=response[0], content_type="text/json",
                                    as_attachment=True,
                                    filename=f"OLT-{mac_address.replace(':', '_')}-DEBUG-STATE.json")
            else:
                return Response(status=response[0], data=response[1], headers=set_response_headers(request))
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
            return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onus(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


AUTO_SWITCHOVER_SCHEMA = {
    "properties": {
        'Auto Switchover Status': {
            "type": "string",
            "enum": ["Enabled", "Disabled", "Enabled Switchover"]
        }
    },
    "required": [
        'Auto Switchover Status'
    ]
}


class ProtectionAutomaticSwitchover(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="olt-cfg", resource_id_param="mac_address", validate_required=True,
                   schema=AUTO_SWITCHOVER_SCHEMA)
    def put(self, request, data, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_network_olts" in db.get_user_permissions(request.user.email):
            auto_switchover_status = data['Auto Switchover Status']
            response = db.put_protection_automatic_switchover(request.user.email, mac_address, auto_switchover_status)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class ProtectionForcedSwitchover(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_network_olts" in db.get_user_permissions(request.user.email):
            response = db.put_protection_forced_switchover(request.user.email, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


PROTECTION_CFG_SCHEMA = {
    "properties": {
        "Peer": {
            "type": "string",
            "pattern": "^$|^([0-9a-f]{2}(:[0-9a-f]{2}){5})$",
            "description": "An OLT mac address"
        },
        "Inactivity Detection Time When Active": {
            "type": "number",
            "enum": [50, 100, 200]
        },
        "Inactivity Detection Time When Standby": {
            "type": "number",
            "enum": [10, 50, 100, 200]
        },
        "ONU Fail Limit": {
            "type": "number"
        },
        "NNI RX LOS Pulse": {
            "type": "number"
        }
    },
    "required": [
        "Peer",
    ]
}


class ProtectionCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="olt-cfg", resource_id_param="mac_address", validate_required=True,
                   schema=PROTECTION_CFG_SCHEMA)
    def put(self, request, data, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_network_olts" in db.get_user_permissions(request.user.email):
            other_configs = data.get('otherConfigs', None)
            peer = data['Peer']
            response = db.put_protection_cfg(request.user.email, mac_address, peer, other_configs)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class GetAllGponOnuSerialNumbers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_gpon_onu_serial_numbers(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnusWithVersions(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onus_with_versions(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetUnspecifiedOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_unspecified_onus(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnusForOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onus_for_olt(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnlineOnusForController(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_online_onus_for_controller(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnusUnderOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onus_under_olt(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetEponOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params

        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_epon_onus(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuStatesForOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_states_for_olt(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetFirstAssignedOnuFirmware(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, firmware):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_first_assigned_onu_firmware(request.user.email, firmware, request.GET.get('limit', 1))
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetFirstAssignedOnuSla(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, sla):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_first_assigned_onu_sla(request.user.email, sla)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CntlState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_cntl_state(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class SwitchState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_switch_state(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_olt_state(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_state(mac_address, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ControllerStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_cntl_stats(mac_address, request.user.email, since_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetRangedControllerStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time, to_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_cntl_stats(mac_address, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_olt_stats(mac_address, request.user.email, since_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetRangedOltStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time, to_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_olt_stats(mac_address, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_stats(request, since_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_stats(mac_address, request.user.email, since_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetRangedOnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time, to_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_stats(mac_address, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetManyOnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        onus = params["onus"]
        since_utc_time = params["start-time"]
        to_utc_time = get_nested_value(params, ["end-time"], None)

        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_many_onu_stats(onus, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetManyDistinctOnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        onus = params["onus"]
        stats = params["stats"]
        since_utc_time = params["start-time"]
        to_utc_time = get_nested_value(params, ["end-time"], None)

        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_many_distinct_onu_stats(onus, stats, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuStatisticsTotals(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        onus = params["onus"]
        since_utc_time = params["start-time"]
        to_utc_time = get_nested_value(params, ["end-time"], None)

        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_stats_totals(onus, request.user.email, since_utc_time, to_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuStatistics(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_stats(request, since_utc_time)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlCfgs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_cntl_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltCfgs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuCfgs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CntlLog(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_cntl_log(mac_address, since_utc_time, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltLog(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_olt_log(mac_address, since_utc_time, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuLog(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_log(mac_address, since_utc_time, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_cntl_alarms(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_alarms(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_alarms(request.user.email, request.GET.get('attribute', None))
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuRegistrationAllow(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    # Checking if ONU is disallowed on any OLT AND if there is a different Reg Allow ONU pending on the given OLT
    def get(self, request, id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_registration_allow(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to reade this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    # Performs a Reg Allow ONU an all OLTs an OLT is disallowed on
    def put(self, request, id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_network_onus" in db.get_user_permissions(request.user.email):
            response = db.put_onu_registration_allow(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class GetAllCntlAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_alarms" in db.get_user_permissions(request.user.email):
            response = db.get_all_cntl_alarms_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlAlarmsIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_alarms" in db.get_user_permissions(request.user.email):
            response = db.get_all_cntl_alarms_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_alarms" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_alarms_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltAlarmsIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_alarms" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_alarms_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuAlarmsCfg(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_alarms" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_alarms_cfgs(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuAlarmsIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_alarms" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_alarms_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSlaIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_slas" in db.get_user_permissions(request.user.email):
            response = db.get_all_sla_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllServiceIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_services" in db.get_user_permissions(request.user.email):
            response = db.get_all_service_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllDownstreamMapIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_services" in db.get_user_permissions(request.user.email):
            response = db.get_all_downstream_map_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOmciServiceIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_services" in db.get_user_permissions(request.user.email):
            response = db.get_all_omci_service_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class Images(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, filename):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            response = db.download_image(filename, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ImageNames(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            response = db.get_image_names(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeviceImageNames(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            if request.query_params and request.query_params['image_name'] and request.query_params['device_type']:
                params = request.query_params
                response = db.get_devices_from_image(request, params['image_name'], params['device_type'])
            else:
                response = db.get_device_image_names(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class Cpes(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_onus" in db.get_user_permissions(request.user.email):
            response = db.delete_cpe(request, id)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CpesForOnu(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):  # Get all for ONU
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_cpes_for_onu(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteManyCpes(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        cpe_ids = request.GET.get('cpeIds').split(',')
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_onus" in db.get_user_permissions(request.user.email):
            response = db.delete_many_cpes(request, cpe_ids, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CpesExist(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.check_cpes_exist(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetCpe(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_cpe_state(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ExpiredCpesForOnu(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):  # Delete All Expired for ONU
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_onus" in db.get_user_permissions(request.user.email):
            response = db.delete_all_expired_cpes_for_onu(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetVcmStateForOnu(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_vcm_state_for_onu(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetVcmStateExist(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_vcm_state_exist(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltFwNames(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            response = db.get_olt_fw_filenames(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuFwNames(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_files" in db.get_user_permissions(request.user.email):
            response = db.get_onu_fw_filenames(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class Names(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id, device_type):
        db = get_client_db(request.user.user_db, request.user.email)
        if device_type == 'onu':
            if "can_read_network_onus" in db.get_user_permissions(request.user.email):
                response = db.get_onu_name(request, id)
            else:
                response = [status.HTTP_403_FORBIDDEN,
                            "You are not permitted to read this data. See a system administrator."]
        elif device_type == 'olt':
            if "can_read_network_olts" in db.get_user_permissions(request.user.email):
                response = db.get_olt_name(request, id)
            else:
                response = [status.HTTP_403_FORBIDDEN,
                            "You are not permitted to read this data. See a system administrator."]
        else:
            if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
                response = db.get_cntl_name(request, id)
            else:
                response = [status.HTTP_403_FORBIDDEN,
                            "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllMibCurStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_mib_cur_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetMibCurState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_mib_cur_state(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuFirmwareUpgradeStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_upgrade_status(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltOnuFirmwareUpgradeStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_upgrade_statuses_for_olt(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ControllerOnuFirmwareUpgradeStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_upgrade_statuses_for_controller(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllMibRstStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_mib_rst_states(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetMibRstState(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_mib_rst_state(id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class CustomDeviceQuery(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, collection, attribute, operator, value):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_custom_device_query(collection, attribute, operator, value, request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


UPGRADE_CONFIGURATIONS_SCHEMA = {
    "type": "array",
    "additionalItems": True,
    "items": {
        "type": "object",
        "properties": {
            "collection": {
                "type": "string",
                "description": "The database collection that is being upgraded"
            },
            "newVersion": {
                "type": "string",
                "description": "The version to upgrade to"
            },
            "device": {
                "type": "string",
                "description": "Unique identifier for the device"
            },
            "newFields": {
                "type": "array",
                "items": {
                    "type": "object",
                    "description": "New field as {key:value} pair"
                }
            }
        },
        "required": [
            "collection",
            "newVersion",
            "device",
            "newFields"
        ]
    }
}


class UpgradeConfigurationsToNewVersion(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_databases" in db.get_user_permissions(request.user.email):
            response = db.get_device_upgrade_hierarchy(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    @validate_data(collection="upgrade_configurations", resource_id_param=None, validate_required=True,
                   schema=UPGRADE_CONFIGURATIONS_SCHEMA)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_global_config_databases" in db.get_user_permissions(request.user.email):
            response = db.upgrade_configurations_versions(data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class SwitchThermalTestData(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, switch_id):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_switch_thermal_test_data(switch_id, request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOnu(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_onus" in db.get_user_permissions(request.user.email):
            delete_from_olt_inv = request.GET.get('deleteFromOltInv')
            response = db.delete_onu_all(request, mac_address, delete_from_olt_inv)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteManyOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_onus" in db.get_user_permissions(request.user.email):
            delete_from_olt_inv = request.GET.get('deleteFromAllDevices')
            ids = request.GET.get('ids').split(',')
            response = db.delete_many_onus(request, ids, delete_from_olt_inv)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOnuLogs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_onus" in db.get_user_permissions(request.user.email):
            response = db.delete_onu_logs(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOnuStats(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_onus" in db.get_user_permissions(request.user.email):
            response = db.delete_onu_stats(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_olts" in db.get_user_permissions(request.user.email):
            delete_from_all_devices = request.GET.get('deleteFromAllDevices')
            response = db.delete_olt_all(request, mac_address, delete_from_all_devices)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteManyOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_olts" in db.get_user_permissions(request.user.email):
            delete_from_all_devices = request.GET.get('deleteFromAllDevices')
            ids = request.GET.get('ids').split(',')
            response = db.delete_many_olts(request, ids, delete_from_all_devices)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOltLogs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_olts" in db.get_user_permissions(request.user.email):
            response = db.delete_olt_logs(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteOltStats(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_olts" in db.get_user_permissions(request.user.email):
            response = db.delete_olt_stats(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteController(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.delete_cntl(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteControllerLogs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.delete_cntl_logs(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteControllerStats(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.delete_cntl_stats(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DeleteSwitch(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def delete(self, request, mac_address):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_network_switches" in db.get_user_permissions(request.user.email):
            response = db.delete_switch_all(request, mac_address)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to delete this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


DATABASE_SEEDING_SCHEMA = {
    "properties": {
        "toSeed": {
            "type": "string",
            "description": "database to seed"
        },
        "toInclude": {
            "type": "object",
            "description": "Collections, Firmwares, Files to be seeded",
        }
    },
    "required": [
        "toSeed",
        "toInclude"
    ]
}


class DatabaseSeeding(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_databases" in db.get_user_permissions(request.user.email):
            response = db.get_seed_files(request, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    @validate_data(collection="database_seeding", resource_id_param=None, validate_required=True,
                   schema=DATABASE_SEEDING_SCHEMA)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_global_config_databases" in db.get_user_permissions(request.user.email):
            db_to_seed = data["toSeed"]
            to_include = data["toInclude"]
            db = dbs[db_to_seed]
            response = db.seed_database(request, to_include)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class UserDatabasePasswordType(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_databases" in db.get_user_permissions(request.user.email):
            try:
                response = [status.HTTP_200_OK, database_manager.get_user_database_password_opts()['type']]
            except:
                response = [status.HTTP_200_OK, 'password']
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class DatabaseSeedOltCFG(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="PONMGR-CFG", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_global_config_databases" in db.get_user_permissions(request.user.email):
            pon_mode = data["PON Mode"]
            response = db.set_seed_pon_mode(self.request, pon_mode)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class CheckCollectionExistence(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        collection_names = params["collection_names"]

        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_global_config_databases" in db.get_user_permissions(request.user.email):
            response = db.check_collection_existence(request, request.user.email, collection_names)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllUnattachedControllers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_unattached_controllers(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllControllersWithVersions(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_controllers_with_versions(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllUnattachedSwitches(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_all_unattached_switches(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllUnattachedOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_unattached_olts(request)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPreProvisionedOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_pre_provisioned_olts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllUnattachedOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_unattached_onus(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllUnattachedOnuConfigs(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_unattached_onu_configs(request)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSlaOnuUsage(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, sla):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_slas" in db.get_user_permissions(request.user.email):
            response = db.get_all_sla_onu_usage(request, sla)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSrvOnuUsage(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, srv):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_services" in db.get_user_permissions(request.user.email):
            response = db.get_all_srv_onu_usage(request, srv)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OmciInfo(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_global_config_services" in db.get_user_permissions(request.user.email):
            response = [status.HTTP_200_OK, ""]
            try:
                try:
                    with open(production_omci_info, 'r', encoding='utf-8') as omci_file:
                        omci_data = json.load(omci_file)
                except FileNotFoundError:
                    with open(development_omci_info, 'r', encoding='utf-8') as omci_file:
                        omci_data = json.load(omci_file)

                response[1] = omci_data
            except Exception as err:
                messages.error(self.request, f"{type(err)}")
                messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
                response[1] = "Could not get OMCI information due to {}::{}".format(type(err).__name__,
                                                                                    sys.exc_info()[1])
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


CREATE_USERS_SCHEMA = {
    "type": "object",
    "properties": {
        "email": {
            "$ref": "COMMON-TYPES.json#/definitions/Email"
        },
        "fName": {
            "type": "string",
            "maxLength": 64
        },
        "lName": {
            "type": "string",
            "maxLength": 64
        },
        "roles": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "password": {
            "type": "string",
            "maxLength": 128
        }
    },
    "required": [
        "email",
        "fName",
        "lName",
        "roles",
        "password"
    ]
}


# Create User
# POST support only
# /user/
# This will create a user
# Returns error message if it fails, otherwise, nothing
class CreateUsers(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        pass

    @validate_data(collection="create_users", resource_id_param=None, validate_required=True,
                   schema=CREATE_USERS_SCHEMA)
    def post(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_create_accounts_admin" in db.get_user_permissions(request.user.email):
            try:
                user_email = data['email'].lower()
                user_password = hashlib.md5(data['password'].encode('utf-8')).hexdigest()

                # Verify email is not in use
                if use_mongo_engine:
                    email_in_use = User.objects(email=user_email).count() != 0
                else:
                    email_in_use = User.objects.filter(email=user_email).exists()

                # creating of user
                if email_in_use:
                    return Response(status=status.HTTP_409_CONFLICT,
                                    data="The provided email address is already in use.")

                if use_mongo_engine:
                    user = User.create_user(username=user_email, email=user_email,
                                            password=user_password)
                else:
                    user = User.objects.create_user(username=user_email, email=user_email,
                                                    password=user_password)


            except SQLDecodeError as e:
                print(e)
                if 'duplicate key error' in e.args[0]:
                    msg = {
                        'err': 'User Already exists',
                        'code': 1}
                else:
                    msg = {
                        'err': 'Unable to connect to database',
                        'code': 0}
                return JsonResponse(msg)
            user.first_name = data['fName']
            user.last_name = data['lName']
            for role in data['roles']:
                if (use_mongo_engine and Group.objects.filter(name=role).count() != 0) or (
                        not use_mongo_engine and Group.objects.filter(name=role).exists()):
                    user.groups.add(Group.objects.get(name=role))

            if use_mongo_engine:
                user.backend = 'backends.mongoengine.MongoBackend'
            else:
                user.backend = 'django.contrib.auth.backends.ModelBackend'

            user.save()

            db.create_user_log(user, request.user.email)
            if use_mongo_engine:
                user_info = {"Email": user_email, "First Name": user.first_name, "Last Name": user.last_name,
                             "Password": "******", "Roles": user.groups.names()}
            else:
                user_info = {"Email": user_email, "First Name": user.first_name, "Last Name": user.last_name,
                             "Password": "******", "Roles": list(map(lambda u: u['fields']['name'], json.loads(
                        serializers.serialize('json', user.groups.all()))))}
            response = [status.HTTP_201_CREATED, None, user_info, None]
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to write this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], new_data=response[2], old_data=response[3])


class UserPermissions(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_accounts_admin" in db.get_user_permissions(request.user.email):
            status_code = status.HTTP_200_OK
            try:
                content_types = {}
                if use_mongo_engine:
                    for content_type in ContentType.objects.all():
                        content_types[content_type.ctid] = (content_type.app_label, content_type.model)
                else:
                    for content_type in json.loads(serializers.serialize('json', ContentType.objects.all())):
                        content_types[content_type['pk']] = (
                            content_type['fields']['app_label'], content_type['fields']['model'])

                data = {}

                if use_mongo_engine:
                    for permission in Permission.objects.all():
                        # Build permissions response
                        app_label = ""
                        for token in content_types[permission.content_type][0].split("_"):
                            app_label += token.capitalize() + " "
                        app_label = app_label.strip()
                        model = ""
                        for token in content_types[permission.content_type][1].split("_"):
                            model += token.capitalize() + " "
                        model = model.strip()
                        if app_label not in data:
                            data[app_label] = {}
                        if model not in data[app_label]:
                            data[app_label][model] = {}
                        if "read" in permission.codename:
                            data[app_label][model]['read'] = permission.name
                        elif "create" in permission.codename:
                            data[app_label][model]['create'] = permission.name
                        elif "update" in permission.codename:
                            data[app_label][model]['update'] = permission.name
                        elif "delete" in permission.codename:
                            data[app_label][model]['delete'] = permission.name
                        elif "view" in permission.codename:
                            data[app_label][model]['view'] = permission.name
                else:
                    for permission in json.loads(serializers.serialize('json', Permission.objects.all())):
                        # Build permissions response
                        app_label = ""
                        for token in content_types[permission['fields']['content_type']][0].split("_"):
                            app_label += token.capitalize() + " "
                        app_label = app_label.strip()
                        model = ""
                        for token in content_types[permission['fields']['content_type']][1].split("_"):
                            model += token.capitalize() + " "
                        model = model.strip()
                        if app_label not in data:
                            data[app_label] = {}
                        if model not in data[app_label]:
                            data[app_label][model] = {}
                        if "read" in permission['fields']['codename']:
                            data[app_label][model]['read'] = permission['fields']['name']
                        elif "create" in permission['fields']['codename']:
                            data[app_label][model]['create'] = permission['fields']['name']
                        elif "update" in permission['fields']['codename']:
                            data[app_label][model]['update'] = permission['fields']['name']
                        elif "delete" in permission['fields']['codename']:
                            data[app_label][model]['delete'] = permission['fields']['name']
                        elif "view" in permission['fields']['codename']:
                            data[app_label][model]['view'] = permission['fields']['name']
            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                messages.error(self.request, f"{type(err)}")
                messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not get permissions list due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to read this data. See a system administrator."
        return Response(status=status_code, data=data)


USER_ROLES_SCHEMA = {
    "type": "object",
    "properties": {
        "name": {
            "type": "string"
        },
        "permissions": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "can read databases",
                    "can create databases",
                    "Can update databases",
                    "Can delete databases",
                    "Can read controllers",
                    "Can create controllers",
                    "Can update controllers",
                    "Can delete controllers",
                    "Can read olts",
                    "Can create olts",
                    "Can update olts",
                    "Can delete olts",
                    "Can read onus",
                    "Can create onus",
                    "Can update onus",
                    "Can delete onus",
                    "Can read switches",
                    "Can create switches",
                    "Can update switches",
                    "Can delete switches",
                    "Can read alarms",
                    "Can create alarms",
                    "Can update alarms",
                    "Can delete alarms",
                    "Can read slas",
                    "Can create slas",
                    "Can update slas",
                    "Can delete slas",
                    "Can read services",
                    "Can create services",
                    "Can update services",
                    "Can delete services",
                    "Can read files",
                    "Can create files",
                    "Can update files",
                    "Can delete files",
                    "Can read global config databases",
                    "Can create global config databases",
                    "Can update global config databases",
                    "Can delete global config databases",
                    "Can read accounts admin",
                    "Can create accounts admin",
                    "Can update accounts admin",
                    "Can delete accounts admin",
                    "Can read other branding",
                    "Can create other branding",
                    "Can update other branding",
                    "Can delete other branding",
                    "Can read dashboard",
                    "Can create dashboard",
                    "Can update dashboard",
                    "Can delete dashboard",
                    "Can read automation",
                    "Can create automation",
                    "Can update automation",
                    "Can delete automation",
                    "Can view accounts admin",
                    "Can view dashboard",
                    "Can view alarms",
                    "Can view automation",
                    "Can view global config databases",
                    "Can view files",
                    "Can view services",
                    "Can view slas",
                    "Can view controllers",
                    "Can view olts",
                    "Can view onus",
                    "Can view switches",
                    "Can view branding",
                    "Can view global config devices",
                    "Can create search filters",
                    "Can read search filters",
                    "Can update search filters",
                    "Can delete search filters",
                    "Can view search filters",
                    "Can read cascading configurations",
                    "Can create cascading configurations",
                    "Can update cascading configurations",
                    "Can delete cascading configurations",
                    "Can view cascading configurations",
                    "Can create tasks",
                    "Can read tasks",
                    "Can update tasks",
                    "Can delete tasks",
                    "Can view tasks",
                ]
            }
        },
        "users": {
            "type": "array",
            "items": {
                "type": "string"
            }
        }
    },
    "required": [
        "name",
        "permissions",
        "users"
    ]
}


class UserRoles(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_accounts_admin" in db.get_user_permissions(request.user.email):
            status_code = status.HTTP_200_OK
            try:
                data = db.get_all_roles_with_users_permissions_and_timeout(request.user.email)
            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                messages.error(self.request, f"{type(err)}")
                messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not get roles list due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to read this data. See a system administrator."
        return Response(status=status_code, data=data)

    @validate_data(collection="create_role", resource_id_param=None, validate_required=True, schema=USER_ROLES_SCHEMA)
    def post(self, request, body):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_create_accounts_admin" in db.get_user_permissions(request.user.email):
            status_code = status.HTTP_201_CREATED
            try:
                name = body["name"]
                users = body["users"]
                permissions = body["permissions"]

                if use_mongo_engine:
                    try:
                        # get_or_create no longer in mongoengine. Manually check
                        Group.objects.get(name=name)
                        status_code = status.HTTP_409_CONFLICT
                        data = f"Role '{name}' cannot be created because it already exists."
                    except DoesNotExist:
                        # role does not exist... create new role
                        new_group = Group(name=name)

                        # Add users
                        added_users = []
                        for user in users:
                            if user is not None and User.objects(email=user).count() > 0:
                                new_group.user_set.add(User.objects.get(email=user))
                                added_users.append(user)

                        # Add permissions
                        for permission in permissions:
                            if permissions is not None and Permission.objects(name=permission).count() > 0:
                                new_group.permissions.add(Permission.objects.get(name=permission))
                        new_group.save()
                        data = f"Role '{name}' was created with permissions {permissions}"
                        if added_users:
                            data += f", added users {added_users}"
                    else:
                        status_code = status.HTTP_409_CONFLICT
                        data = f"Role '{name}' cannot be created because it already exists."
                else:
                    new_group, created = Group.objects.get_or_create(name=name)
                    if created:
                        # Add users
                        added_users = []
                        for user in users:
                            if user is not None and User.objects.filter(email=user).exists():
                                new_group.user_set.add(User.objects.get(email=user))
                                added_users.append(user)

                        # Add permissions
                        for permission in permissions:
                            if permissions is not None and Permission.objects.filter(name=permission).exists():
                                new_group.permissions.add(Permission.objects.get(name=permission))
                        new_group.save()
                        data = f"Role '{name}' was created with permissions {permissions}"
                        if added_users:
                            data += f", added users {added_users}"

            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                messages.error(self.request, f"{type(err)}")
                messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not create role due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to write this data. See a system administrator."
        return PonManagerApiResponse(status=status_code, new_data=data)

    @validate_data(collection="update_roles", resource_id_param=None, validate_required=False, schema=USER_ROLES_SCHEMA)
    def put(self, request, body):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_accounts_admin" in db.get_user_permissions(request.user.email):
            status_code = status.HTTP_200_OK
            try:
                name = body["name"]
                group = Group.objects.get(name=name)
                if "users" in body:
                    if name == "Administrators" and request.user.email not in body['users']:
                        status_code = status.HTTP_403_FORBIDDEN
                        data = "A user assigned the Administrator role may not remove themselves from that role."
                        old_data = None
                    else:
                        old_data = {name: {"Users": list(map(str, group.user_set.all()))}}
                        for user in group.user_set.all():
                            if user.email not in body['users']:
                                group.user_set.remove(User.objects.get(email=user.email))
                        for user in body['users']:
                            if user is not None:
                                group.user_set.add(User.objects.get(email=user))
                        data = {name: {"Users": list(map(str, group.user_set.all()))}}
                        if use_mongo_engine:
                            group.save()
                elif "permissions" in body:
                    old_data = {name: {"Permissions": list(map(str, group.permissions.all()))}}
                    for permission in group.permissions.all():
                        if permission.name not in body['permissions']:
                            group.permissions.remove(Permission.objects.get(name=permission.name))
                    for permission in body['permissions']:
                        if permission is not None:
                            group.permissions.add(Permission.objects.get(name=permission))
                    data = {name: {"Permissions": list(map(str, group.permissions.all()))}}
                    if use_mongo_engine:
                        group.save()
                else:
                    status_code = status.HTTP_400_BAD_REQUEST
                    data = f"Role {name} could not be updated."
                    old_data = None
            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                messages.error(self.request, f"{type(err)}")
                messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data = "Could not update role due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
                old_data = None
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to update this data. See a system administrator."
            old_data = None
        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_data)

    def delete(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_accounts_admin" in db.get_user_permissions(request.user.email):
            status_code = status.HTTP_200_OK
            try:
                user = User.objects.get(username=request.user.email)
                body = json.loads(request.body)
                name = body["name"]
                if (use_mongo_engine and Group.objects(name=name).count() > 0) or (
                        not use_mongo_engine and Group.objects.filter(name=name).exists()):
                    group = Group.objects.get(name=name)
                    if name == "Administrators":
                        status_code = status.HTTP_403_FORBIDDEN
                        data_change = f"The Administrators role cannot be deleted."
                    elif group.user_set.count() > 0:
                        status_code = status.HTTP_428_PRECONDITION_REQUIRED
                        data_change = f"The role '{name}' cannot be deleted because it contains users."
                    else:
                        group.delete()
                        data_change = f"deleted role: {name}"
                else:
                    data_change = f"deleted role: {name}"

            except Exception as err:
                traceback.print_exc(file=sys.stdout)
                messages.error(self.request, f"{type(err)}")
                messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
                status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                data_change = "Could not delete role due to {}::{}".format(type(err).__name__, sys.exc_info()[1])
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data_change = "You are not permitted to delete this data. See a system administrator."

        return PonManagerApiResponse(status=status_code, data_change=data_change, headers=set_response_headers(request))


# Create Initial User
# POST support only
# /user/initialize/
# This will create a user ONLY when no other users exist
# Returns error message if it fails, otherwise, nothing
class CreateInitialUser(APIView):
    def get(self, request):
        pass

    @validate_data(collection="auth_user", resource_id_param=None, validate_required=False)
    def post(self, request, data):
        if len(User.objects.all()) == 0:
            try:
                # Initializing user parameters to be used for password validation
                user_email = ''
                user_first_name = ''
                user_last_name = ''

                # Verifying that request has all parameters for password validation, and setting local vars
                if 'email' in data:
                    user_email = data['email'].lower()

                if 'first_name' in data:
                    user_first_name = data['first_name']

                if 'last_name' in data:
                    user_last_name = data['last_name']

                # Setting and hashing password
                user_password = hashlib.md5(data['password'].encode('utf-8')).hexdigest()

                # Validating password meets minimum requirements
                password_requirement_response = validateMinimumPasswordRequirements(request, data['password'],
                                                                                    user_email, user_first_name,
                                                                                    user_last_name)

                # If password does not meet requirements, return
                if password_requirement_response['password_meets_requirements'] is False:
                    return Response(status=status.HTTP_400_BAD_REQUEST,
                                    data=password_requirement_response['missing_criteria_message'])

                # creating of user
                if use_mongo_engine:
                    user = User.create_user(username=user_email, email=user_email, password=user_password,
                                            is_initial_user=True)
                else:
                    user = User.objects.create_user(username=user_email, email=user_email, password=user_password)

            except SQLDecodeError as e:
                print(e)
                if 'duplicate key error' in e.args[0]:
                    data = {
                        'err': 'User Already exists',
                        'code': 1}
                else:
                    data = {
                        'err': 'Unable to connect to database',
                        'code': 0}
                return JsonResponse(data)

            # If password DOES meet minimum reqs, configure and safe the user, else return 400
            user.first_name = data['first_name']
            user.last_name = data['last_name']
            user.backend = 'django.contrib.auth.backends.ModelBackend'

            user.groups.add(Group.objects.get(name="Administrators"))

            if use_mongo_engine:
                user.save(is_initial_user=True)
            else:
                user.save()

            user_db = "Default"
            db = get_client_db(user_db, user_email)
            db.create_user_log(user, user_email)

            return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT, new_data=user_email)
        else:
            return PonManagerApiResponse(status=status.HTTP_403_FORBIDDEN)


# Get All Users
# GET support only
# /getAllUsers
# This will return all users
class GetAllUsers(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        all_users = []
        try:
            db = get_client_db(request.user.user_db, request.user.email)
            if "can_read_accounts_admin" in db.get_user_permissions(request.user.email):
                all_users = db.get_all_users_with_groups()

                for user in all_users:
                    if 'database_type' not in user:
                        user['database_type'] = 'local'

                data = {'users': all_users}
                return JsonResponse(data, status=status.HTTP_200_OK)
            else:
                return JsonResponse({}, status=status.HTTP_403_FORBIDDEN)
        except Exception as err:
            data = {'err': 'Server Error'}
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            return JsonResponse(data, status=status.HTTP_500_INTERNAL_SERVER_ERROR)  # if failed, we return error message# Get All Users


# GET support only
# /user/active/
# Returns an array of all current active users, by ID
class GetAllActiveUsers(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        all_users = set()
        try:
            db = get_client_db(request.user.user_db, request.user.email)
            if "can_read_accounts_admin" in db.get_user_permissions(request.user.email):
                request.session.clear_expired()

                if use_mongo_engine:
                    session_objects = MongoSession.objects()
                else:
                    session_objects = Session.objects.all()

                for session in session_objects:
                    user_auth_id = session.get_decoded().get('_auth_user_id')
                    if user_auth_id is not None:
                        try:
                            if use_mongo_engine:
                                user_email = User.objects.get(uid=user_auth_id)
                            else:
                                user_email = User.objects.get(pk=user_auth_id)

                            if user_email is not None:
                                all_users.add(str(user_email))
                        except User.DoesNotExist:  # handle session with user that no longer exists
                            continue

                data = {'users': list(all_users)}
                return JsonResponse(data, status=status.HTTP_200_OK)
            else:
                return JsonResponse({}, status=status.HTTP_403_FORBIDDEN)
        except Exception as err:
            data = {'err': 'Server Error'}
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            return JsonResponse(data, status=status.HTTP_500_INTERNAL_SERVER_ERROR)  # if failed, we return error message


# Get users exist (bool)
# GET support only
# /user/exist/
# This will return True if one or more users exist in database, false if not
class GetUsersExist(APIView):
    def get(self, request):
        try:
            users_exist = len(User.objects.all()) > 0
            data = {'usersExist': users_exist}
            return JsonResponse(data)
        except:
            data = {'err': 'Server Error'}
            return JsonResponse(data)  # if failed, we return error message


# Updates user session
# GET support only
# /user/session/
# Session is updated automatically in middleware
class RefreshSession(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        return Response(status=status.HTTP_200_OK, data={'session': True}, headers=set_response_headers(request))


# Global Session Age Setting
# GET/PUT support
# /user/session/age/
class GlobalSessionAgeSetting(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_global_session_time()
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    @validate_data(collection="PONMGR-CFG", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_accounts_admin" in db.get_user_permissions(request.user.email):
            response = db.set_global_session_time(data['User Session Expiry Age Timeout'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class UserSessionAgeSetting(LoginRequiredMixin, APIView):
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_user_session_expiry(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    @validate_data(collection="auth_group", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_accounts_admin" in db.get_user_permissions(request.user.email):
            body = request.data
            response = db.set_role_session_expiry(body['data']['id'],
                                                  body['data']['User Session Expiry Age Timeout Override'],
                                                  body['data']['User Session Expiry Age Timeout'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class GetAllSessionAges(LoginRequiredMixin, APIView):
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_all_session_expiry()
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllSessions(LoginRequiredMixin, APIView):
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_accounts_admin" in db.get_user_permissions(request.user.email):
            sessionId = request.COOKIES['__Host-sessionid']
            response = db.get_all_user_sessions(request.user.email, sessionId)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator.", None, None]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def delete(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_accounts_admin" in db.get_user_permissions(request.user.email):
            body = json.loads(request.body)
            response = db.delete_user_sessions(body['sessions'])
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to delete this data. See a system administrator.", None, None]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Global timestamp locale format setting
# GET/PUT support
# /user/locale/timestamp/
class LocaleTimestampFormat(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_locale_timestamp_format(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    @validate_data(collection="PONMGR-CFG", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_accounts_admin" in db.get_user_permissions(request.user.email):
            response = db.set_locale_timestamp_format(data['Locale Timestamp Format'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


# Global open street location enabling
# GET/PUT support
# /user/locale/location/
class LocationPref(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_location_pref(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    @validate_data(collection="PONMGR-CFG", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_accounts_admin" in db.get_user_permissions(request.user.email):
            response = db.set_location_pref(data['Enable Open Street Location'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


# Global pon mode setting
# GET/PUT support
# /user/pon/mode
class PonMode(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_pon_mode(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    @validate_data(collection="PONMGR-CFG", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_accounts_admin" in db.get_user_permissions(request.user.email):
            response = db.set_pon_mode(data['PON Mode'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class RadiusSettings(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_radius_settings(request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    @validate_data(collection="PONMGR-CFG", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_accounts_admin" in db.get_user_permissions(request.user.email):
            response = db.set_radius_settings(data['settings'], request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class SideNav(APIView):
    def get(self, request):
        status_code = status.HTTP_200_OK
        response = {}
        try:
            config = database_manager.get_ponmgr_cfg()
            if "Enable Collapsed Navigation Panel" in config:
                response["Enable Collapsed Navigation Panel"] = config["Enable Collapsed Navigation Panel"]
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
        return Response(status=status_code, data=response, headers=set_response_headers(request))


class PonMgrSettings(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        status_code = status.HTTP_200_OK
        try:
            response = database_manager.get_ponmgr_cfg()
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
        return Response(status=status_code, data=response, headers=set_response_headers(request))

    @validate_data(collection="PONMGR-CFG", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        status_code = status.HTTP_200_OK
        try:
            response = database_manager.set_ponmgr_cfg(data)
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR

        return PonManagerApiResponse(status=status_code, new_data=response, old_data={})


# Authenticate User based on session token, not credentials
# POST support only
# /user/authenticate/token/
# This will authenticate a user
# Returns success status(bool) and userdata(only on success)
class AuthenticateUsersToken(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def post(self, request):
        response = [status.HTTP_200_OK, {'auth': 'False'}]
        purge_expired_sessions(request)
        request.user.last_login = datetime.now(timezone.utc)
        request.user.save()
        db = get_client_db(request.user.user_db, request.user.email)  # TODO: verify

        response[1] = {'auth': 'True', 'fName': request.user.first_name, 'lName': request.user.last_name,
                       'email': request.user.email, 'lastLogin': request.user.last_login,
                       'dateJoined': request.user.date_joined,
                       'roles': list(db.get_user_groups(request.user.email)),
                       'permissions': list(db.get_user_permissions(
                           request.user.email))}  # returning user data to be used in view context

        try:
            session_age_response = db.get_user_session_expiry(request.user.email)
            timeout_in_minutes = session_age_response[
                                     1] * 60  # Database value is in minutes. However, the set_expiry method takes seconds.
            request.session.set_expiry(timeout_in_minutes)
            db.sys_log({"user": request.user.email, "collection": "users"}, "login", True)
        except Exception as err:  # Failed to retrieve database associated with user / possibly does not exist anymore
            try:
                db = dbs['Default']  # Will attempt to write log using Default database connector
                db.sys_log({"user": request.user.email, "collection": "users"}, "login", True)
            except Exception as defaultErr:
                print("Failed to write log. Action: Authenticate: ", err)

        return PonManagerApiResponse(status=response[0], data=response[1], headers=set_response_headers(request))


# Logout User
# GET support only
# user/logout/
# This will logout a user
# Returns nothing. Might want to change, to return a status of some kind, but i'm skeptical there will ever be a need. USER not existing              Thu, 22 Jul 2021 21:17:58 GMT
class LogoutUser(APIView):

    def get(self, request):
        # Logout always return successfully regardless if user is logged in/exists
        logout(request)
        response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
        response.delete_cookie('__Host-user_db', samesite='strict')
        response.delete_cookie('__Host-sessionexpire', samesite='strict')
        response.delete_cookie('__Host-csrftoken', samesite='strict')
        response.delete_cookie('__Host-sessionid', samesite='strict')
        return response


# Triggers automatic logout of user, such as in the case of closing browser session
# PUT support only
# user/logout/auto/
class AutoLogoutUser(LoginRequiredMixin, APIView):

    def get(self, request):
        pass

    def put(self, request):
        try:
            request.session.set_expiry(
                60)  # User will have 60 seconds to make another API request before their session is invalidated
        except Exception as err:
            traceback.print_exc(file=sys.stdout)
            data = "Could not user session expiry age due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


EMAIL_SCHEMA = {
    "properties": {
        "email": {
            "$ref": "COMMON-TYPES.json#/definitions/Email"
        }
    },
    "requirements": {
        "email"
    }
}


# Send Recovery Email
# POST support only
# /sendRecoveryEmail
# This will Send a recovery email to the specified user. Invoked from forgot my password
# Returns error message if it fails, otherwise, nothing
# Good reference on how to properly configure gmail smtp server: https://www.siteground.com/kb/google_free_smtp_server/
class SendRecoveryEmail(APIView):
    def get(self, request):
        pass

    @validate_data(collection="recovery_email", resource_id_param=None, validate_required=True, schema=EMAIL_SCHEMA)
    def post(self, request, data):
        # Getting email
        userEmail = data['email'].lower()
        # Check if email exists in database
        user_database = database_manager.user_database
        auth_user_coll = user_database['auth_user']
        is_email_registered = auth_user_coll.find_one({"email": userEmail})

        # Will only send recovery code to registered email accounts.
        if is_email_registered:
            # Generate new passcode
            recoveryHash = uuid.uuid4().hex
            # Plain text passcode sent to email
            recoveryCodeToEmail = recoveryHash[:6]
            # print(f"Recovery code: {recoveryCodeToEmail}") # Used for testing
            # Hashed passcode saved in database
            hashedPwdToDb = make_password(recoveryCodeToEmail)
            try:
                # Post recovery codes to the database.
                user_database = database_manager.user_database
                collection = user_database["user_recovery"]
                data = {
                    "_id": userEmail,
                    "Challenge code": hashedPwdToDb,
                    "Time": datetime.now()
                }
                update_document = {"$set": data}
                collection.update_one(filter={"_id": data['_id']}, update=update_document, upsert=True)

                # Subject and body of email
                subject = 'Password Recovery Code'
                body = 'Your code is: ' + recoveryCodeToEmail + ' (Case-Sensitive) This code will expire within 10 minutes of receipt. A new password must be created before the allotted 10 minutes are up.'

                emailFrom = settings.EMAIL_HOST_USER  # Sender. Currently configured in settings.py
                emailTo = [userEmail]  # Who we are sending to.

                # Sending email. We want this to not fail silently. That way, we can catch this and trigger warning to display in index(UI)
                send_mail(subject, body, emailFrom, emailTo, fail_silently=False)
                response = {'Sent': 'True'}
            except Exception as e:
                response = {'Sent': 'False', 'Message': 'Recovery Email Failed To Send'}
                db = dbs['Default']
                db.sys_log({"user": userEmail}, "recovery", False)
                print("Error: Recovery email failed to send.")
        else:
            # To avoid malicious actions, if email account doesn't exist we act like we sent recovery code anyway
            response = {'Sent': 'True'}
        return JsonResponse(response)  # return status


VALIDATE_RECOVERY_CODE_SCHEMA = {
    "properties": {
        "email": {
            "$ref": "COMMON-TYPES.json#/definitions/Email"
        },
        "resetCode": {
            "type": "string",
        }
    },
    "requirements": {
        "email",
        "resetCode"
    }
}


# Validate Recovery Code
# POST support only
# /validateRecoveryCode
# This will validate that a recovery code supplied by user is correct and valid
class ValidateRecoveryCode(APIView):
    @validate_data(collection="recovery_code", resource_id_param=None, validate_required=True,
                   schema=VALIDATE_RECOVERY_CODE_SCHEMA)
    def post(self, request, data):
        userCode = data['resetCode']
        userEmail = data['email'].lower()

        try:
            user_database = database_manager.user_database
            collection = user_database["user_recovery"]
            recoverDoc = collection.find_one({"_id": userEmail})
            storedCode = recoverDoc["Challenge code"]
            userStartTime = recoverDoc["Time"]
            currentDate = datetime.now()

            if check_password(userCode, storedCode):
                if currentDate - timedelta(minutes=PASSCODE_TIME_LIMIT) <= userStartTime <= currentDate:
                    user = User.objects.get(username=userEmail)
                    response_data = {'valid': 'True', 'fname': user.first_name,
                                     'lname': user.last_name}  # If code was valid

                    return JsonResponse(response_data)
                else:
                    collection.delete_one({"_id": userEmail})
                    response_data = {'Sent': 'False', 'Message': 'Passcode has expired.', 'valid': 'False'}
            else:
                response_data = {'valid': 'False'}  # If code was not valid

        except Exception as e:
            pon_manager_logger.error(f"Failed validating recovery code: {e}")
            response_data = {'valid': 'False'}

        return JsonResponse(response_data)


RESET_PASSWORD_SCHEMA = {
    "type": "object",
    "properties": {
        "email": {
            "$ref": "COMMON-TYPES.json#/definitions/Email"
        },
        "resetCode": {
            "type": "string",
        },
        "passwordHash": {
            "type": "string",
            "maxLength": 128,
            "minLength": 1
        }
    },
    "required": [
        "email",
        "resetCode",
        "passwordHash"
    ]
}


######### Reset Password #########################
# POST support only
# Reset Password
# POST support only
# /resetPassword
# This will reset a given users password
class ResetPassword(APIView):
    def get(self, request):
        pass

    @validate_data(collection="password_reset", resource_id_param=None, validate_required=True,
                   schema=RESET_PASSWORD_SCHEMA)
    def put(self, request, data):
        try:
            userEmail = data['email'].lower()
            userCode = data['resetCode']
            user_password = hashlib.md5(data['passwordHash'].encode('utf-8')).hexdigest()

            # Gets data stored for this user in the user_recovery collection
            user_database = database_manager.user_database
            collection = user_database["user_recovery"]
            recoverDoc = collection.find_one({"_id": userEmail})
            storedCode = recoverDoc["Challenge code"]
            userStartTime = recoverDoc["Time"]

            currentDate = datetime.now()
            user = User.objects.get(username=userEmail)

            # Validating password meets minimum requirements
            password_requirement_response = validateMinimumPasswordRequirements(request, data['passwordHash'],
                                                                                userEmail, user.first_name,
                                                                                user.last_name)

            # If password does not meet requirements, return
            if password_requirement_response['password_meets_requirements'] is False:
                return PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             data=password_requirement_response['missing_criteria_message'])

            if check_password(userCode, storedCode):
                if currentDate - timedelta(minutes=PASSCODE_TIME_LIMIT) <= userStartTime <= currentDate:
                    collection.delete_one({"_id": userEmail})
                    user.set_password(user_password)  # Setting the password of that user to the new password hash
                    user.save()  # Saving the state of that user
                    msg = {'valid': 'True', 'user': userEmail}
                else:
                    collection.delete_one({"_id": userEmail})
                    response_data = {'Sent': 'False', 'Message': 'Passcode has expired.', 'valid': 'False'}
                    return PonManagerApiResponse(status=status.HTTP_401_UNAUTHORIZED, data=response_data)
            else:
                response_data = {'valid': 'False'}  # If code was not valid
                return PonManagerApiResponse(status=status.HTTP_403_FORBIDDEN, data=response_data)
        except Exception as e:
            pon_manager_logger.error(f"Failed resetting password: {e}")
            return PonManagerApiResponse(status=status.HTTP_403_FORBIDDEN, data={'valid': 'False'})
        return PonManagerApiResponse(status=status.HTTP_200_OK, data=msg)


#########  Getting Password Requirements #########################
# GET support only
# /user/password/requirements/unauthenticated
# Get/Set the minimum requirements for a valid password
class GetPasswordRequirements(generics.GenericAPIView):

    def get(self, request):
        # Setting requesters email, if it exists
        try:
            requesting_email = request.user.email
        except Exception:
            requesting_email = 'AnonymousUser'

        db = get_client_db("Default", requesting_email)
        response = db.get_minimum_password_requirements(request, requesting_email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


#########  Setting Password Requirements #########################
# PUT support only
# /user/password/requirements/authenticated
# Get/Set the minimum requirements for a valid password
class SetPasswordRequirements(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="PONMGR-CFG", resource_id_param=None, validate_required=False)
    def put(self, request, data):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_accounts_admin" in db.get_user_permissions(request.user.email):
            password_requirements = data["Password Requirements"]
            response = db.set_minimum_password_requirements(request, password_requirements)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


######### Delete User #########################
# DELETE support only
# Delete User
# /user/<email>
# This will take in the email of the user who will be deleted, and removes it
class DeleteUser(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        pass

    def delete(self, request, email):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_delete_accounts_admin" in db.get_user_permissions(request.user.email):
            data = {}

            try:
                user = User.objects.get(username=email)
                if request.user.email == email:
                    return Response(status=status.HTTP_403_FORBIDDEN, data="User cannot delete themselves.")

                user.delete()
                data = {'valid': 'True',
                        'user': request.user.email}

                db.delete_user(request.user.email)

            except Exception:
                data = {'valid': 'False'}
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to update this data. See a system administrator"
        return PonManagerApiResponse(status=status.HTTP_200_OK, data=data)


######### Get Logs For User #########################
# GET support only
# /userLogs/<email>/<since_utc_time>
# This will take in the email of the user and return the logs associated
# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class GetSysLogsForUser(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, email, since_utc_time):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_sys_logs_for_user(email, since_utc_time, request.user.email)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


######### Get Returned Data for log#########################
# GET support only
# /userLogs/<logId>
# This will take in the log id and return the relevent returned log data
# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class GetSysLogReturned(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, logId):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.get_sys_log_returned(logId)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


ADMIN_EDIT_USERS_SCHEMA = {
    "properties": {
        "users_list": {
            "type": "array",
            "additionalItems": True,
            "items": {
                "type": "object",
                "properties": {
                    "email": {
                        "$ref": "COMMON-TYPES.json#/definitions/Email"
                    },
                    "firstName": {
                        "type": "string"
                    },
                    "lastName": {
                        "type": "string"
                    },
                    "roles": {
                        "type": "array",
                        "items": {
                            "type": "string"
                        }
                    },
                    "delete": {
                        "type": "boolean"
                    },
                    "newPassword": {
                        "type": "string"
                    }
                },
                "required": [
                    "email"
                ]
            }
        }
    },
    "required": [
        "users_list"
    ]
}


class AdminEditUsers(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="admin_edit_users", resource_id_param=None, validate_required=False,
                   schema=ADMIN_EDIT_USERS_SCHEMA)
    def put(self, request, body):
        # USER DETAILS CHANGE -> admin>users tab
        status_code = status.HTTP_200_OK
        old_data = {}
        data = {}
        data_change = None

        db = get_client_db(request.user.user_db, request.user.email)
        has_permission = "can_update_accounts_admin" in db.get_user_permissions(request.user.email)

        if has_permission:
            for user_info in body['users_list']:
                user = None
                if use_mongo_engine:
                    user = User.objects.get(email=user_info['email'])
                elif User.objects.filter(email=user_info['email']).exists():
                    user = User.objects.get(email=user_info['email'])

                if user:
                    if 'delete' in user_info and user_info['delete'] is True:
                        user.delete()
                        db.delete_user(request.user.email)
                        data_change = f"deleted user: {user_info['email']}"
                    else:
                        if user.first_name != user_info['firstName'] or user.last_name != user_info['lastName']:
                            if not user_info['email'] in data:
                                old_data[user_info['email']] = {}
                                data[user_info['email']] = {}
                            old_data[user_info['email']]['First Name'] = user.first_name
                            old_data[user_info['email']]['Last Name'] = user.last_name
                            data[user_info['email']]['First Name'] = user_info['firstName']
                            data[user_info['email']]['Last Name'] = user_info['lastName']
                        user.first_name = user_info['firstName']
                        user.last_name = user_info['lastName']

                        if 'newPassword' in user_info:
                            if not user_info['email'] in data:
                                old_data[user_info['email']] = {}
                                data[user_info['email']] = {}
                            old_data[user_info['email']]['Password'] = '******'
                            data[user_info['email']]['Password'] = '*******'
                            user.set_password(hashlib.md5(user_info['newPassword'].encode('utf-8')).hexdigest())
                            if hasattr(user, 'get_session_auth_hash') and request.user == user:
                                # Gets new session if user changed their own password
                                update_session_auth_hash(request, user)
                        if not user_info['email'] in data:
                            old_data[user_info['email']] = {}
                            data[user_info['email']] = {}

                        if use_mongo_engine:
                            old_data[user_info['email']]['Roles'] = user.groups.names()

                            existing_groups = user.groups.names()
                            groups_to_add = set(user_info['roles']).difference(existing_groups)
                            groups_to_remove = set(existing_groups).difference(user_info['roles'])

                            for group_name_to_remove in groups_to_remove:
                                group = Group.objects.get(name=group_name_to_remove)
                                if group:
                                    user.groups.remove(group)

                            for group_name_to_add in groups_to_add:
                                group = Group.objects.get(name=group_name_to_add)
                                if group:
                                    user.groups.add(group)
                            data[user_info['email']]['Roles'] = user.groups.names()
                        else:
                            old_data[user_info['email']]['Roles'] = list(map(lambda u: u['fields']['name'], json.loads(
                                serializers.serialize('json', user.groups.all()))))

                            for role in user_info['roles']:
                                group = Group.objects.get(name=role)
                                if group:
                                    user.groups.add(group)
                            data[user_info['email']]['Roles'] = list(map(lambda u: u['fields']['name'], json.loads(
                                serializers.serialize('json', user.groups.all()))))

                        user.save()
                else:
                    messages.error(self.request, f"User {user_info['email']} was not found")
                    status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
                    data = f"User {user_info['email']} was not found"
                    break
        else:
            status_code = status.HTTP_403_FORBIDDEN
            data = "You are not permitted to update this data. See a system administrator"
        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_data, data_change=data_change,
                                     headers=set_response_headers(request))


class UserAuthType(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        status_code = status.HTTP_200_OK
        data = {}

        try:
            db = get_client_db(request.user.user_db, request.user.email)
            data = db.get_user_database_type(request.user.email)
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
            data = "Could not update user details due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        return Response(status=status_code, data=data, headers=set_response_headers(request))


class UserDetails(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, user_email):
        status_code = status.HTTP_200_OK
        data = {}

        try:
            db = get_client_db(request.user.user_db, request.user.email)
            has_permission = "can_read_accounts_admin" in db.get_user_permissions(request.user.email)
            is_current_user = request.user.email == user_email

            # Update details if they are for the current user OR if they have permission
            if is_current_user or has_permission:
                user = User.objects.get(username=user_email)
                data['email'] = user_email
                data['firstName'] = user.first_name
                data['lastName'] = user.last_name
                data['roles'] = db.get_user_groups(user_email)
                data['permissions'] = db.get_user_permissions(user_email)
                data['lastLogin'] = user.last_login
                data['dateJoined'] = user.date_joined
            else:
                status_code = status.HTTP_403_FORBIDDEN
                data = "You are not permitted to read this data. See a system administrator"
        except Exception as err:
            traceback.print_exc(file=sys.stdout)
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
            data = "Could not update user details due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        return Response(status=status_code, data=data, headers=set_response_headers(request))

    @validate_data(collection="auth_user", resource_id_param="user_email", validate_required=False)
    def put(self, request, user_info, user_email):
        status_code = status.HTTP_200_OK
        old_data = {}
        data = {}

        try:
            user_to_change = user_email
            db = get_client_db(request.user.user_db, request.user.email)
            has_permission = "can_update_accounts_admin" in db.get_user_permissions(request.user.email)
            is_current_user = request.user.email == user_to_change

            # Update details if they are for the current user OR if they have permission
            if is_current_user or has_permission:

                # Validating password meets minimum requirements
                if 'password' in user_info:
                    # Setting first name, if present. Else set based on DB user model
                    if 'first_name' in user_info:
                        first_name = user_info['first_name']
                    else:
                        first_name = request.user.first_name

                    # Setting last name, if present. Else set based on DB user model
                    if 'last_name' in user_info:
                        last_name = user_info['last_name']
                    else:
                        last_name = request.user.last_name

                    # Setting current password if present. Else set to blank
                    if 'current_password' in user_info:
                        current_password = user_info['current_password']
                    else:
                        current_password = ''

                    # Validating password criteria
                    password_requirement_response = validateMinimumPasswordRequirements(request, user_info['password'],
                                                                                        request.user.email,
                                                                                        first_name, last_name,
                                                                                        current_password)
                # If password is not being configured, it passes requirements
                else:
                    password_requirement_response = {'password_meets_requirements': True}
                # END Validating password meets minimum requirements

                # If password meets reqs, continue
                if password_requirement_response['password_meets_requirements'] is True:
                    user = User.objects.get(email=user_to_change)

                    if 'first_name' in user_info:
                        if user_info['first_name'] != user.first_name:
                            old_data['First Name'] = user.first_name
                            data['First Name'] = user_info['first_name']
                            user.first_name = user_info['first_name']
                    if 'last_name' in user_info:
                        if user_info['last_name'] != user.last_name:
                            old_data['Last Name'] = user.last_name
                            data['Last Name'] = user_info['last_name']
                            user.last_name = user_info['last_name']
                    if 'password' in user_info:
                        old_data['Password'] = '******'
                        data['Password'] = '*******'
                        user.set_password(hashlib.md5(user_info['password'].encode('utf-8')).hexdigest())
                        user.save()
                        user = authenticate(username=request.user.email,
                                            password=hashlib.md5(user_info['password'].encode('utf-8')).hexdigest())
                        update_session_auth_hash(request, user)
                    user.save()

                else:
                    status_code = status.HTTP_400_BAD_REQUEST
                    data = password_requirement_response['missing_criteria_message']
            else:
                status_code = status.HTTP_403_FORBIDDEN
                data = "You are not permitted to update this data. See a system administrator"

        except Exception as err:
            traceback.print_exc(file=sys.stdout)
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            status_code = status.HTTP_500_INTERNAL_SERVER_ERROR
            data = "Could not update user details due to {}::{}".format(type(err).__name__, sys.exc_info()[1])

        return PonManagerApiResponse(status=status_code, data=data, new_data=data, old_data=old_data,
                                     headers=set_response_headers(request))


class SwitchAutoCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_automation" in db.get_user_permissions(request.user.email):
            response = db.get_switch_automation_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ControllerCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_controller_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ControllerAutoCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_automation" in db.get_user_permissions(request.user.email):
            response = db.get_cntl_automation_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_olt_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltAutoCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_automation" in db.get_user_permissions(request.user.email):
            response = db.get_olt_automation_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuAutoCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_automation" in db.get_user_permissions(request.user.email):
            response = db.get_onu_automation_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuVcmCounts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_vcm_counts(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class SearchTreeDevices(LoginRequiredMixin, APIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, device_type, filter_str=''):
        db = get_client_db(request.user.user_db, request.user.email)
        response = db.search_tree_device(request.user.email, device_type, filter_str)
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuConfigsUnderOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_configs_for_olt(request.user.email, olt_id)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuStatesUnderOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_states_for_olt(request.user.email, olt_id)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllCntlCfgIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_cntl_cfg_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class SwitchCfgIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_switches" in db.get_user_permissions(request.user.email):
            response = db.get_all_switch_cfg_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOltCfgIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_cfg_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOfflineOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        minutes = request.GET.get('minutes', None)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_offline_olts(request.user.email, minutes)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOfflineOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        minutes = request.GET.get('minutes', None)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_offline_onus(request.user.email, minutes)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetPreProvisionedOnus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_pre_provisioned_onus(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN, "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllAvailableOltPeers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_available_peer_olt_ids(request.user.email, olt_id)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllOnuCfgIds(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_cfg_ids(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


BULK_CHANGES_SCHEMA = {
    "properties": {
        "changesObj": {
            "type": "object",
            "properties": {
                "collection": {
                    "type": "string"
                },
                "attribute": {
                    "type": "string"
                },
                "value": {},
                "devices": {
                    "type": "array",
                    "items": {
                        "type": "string"
                    }
                }
            },
            "required": {
                "collection",
                "attribute",
                "value",
                "devices"
            }
        }
    },
    "required": {
        "changesObj"
    }
}


# Makes a config change for ALL the selected devices
class MakeBulkChange(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="bulk_change", resource_id_param=None, validate_required=True, schema=BULK_CHANGES_SCHEMA)
    def post(self, request, data):
        response = [status.HTTP_200_OK, "", None, {}]
        can_update = False

        try:
            doc_type = data['changesObj']['collection']
            db = get_client_db(request.user.user_db, request.user.email)

            # First check appropriate permisssions
            if doc_type == 'ONU Configurations' and "can_update_network_onus" in db.get_user_permissions(
                    request.user.email):
                can_update = True
            elif doc_type == 'OLT Configurations' and "can_update_network_olts" in db.get_user_permissions(
                    request.user.email):
                can_update = True
            elif doc_type == 'Controller Configurations' and "can_update_network_controllers" in db.get_user_permissions(request.user.email):
                can_update = True
            elif (doc_type == 'Switch Configurations' or doc_type == 'Switch/Router Configurations') and "can_update_network_switches" in db.get_user_permissions(request.user.email):
                can_update = True

            # If one of the above situations is True
            if can_update:
                changes = data
                response = db.make_bulk_change(changes, request.user.email)
            else:
                response = [status.HTTP_403_FORBIDDEN,
                            "You are not permitted to update this data. See a system administrator.", None, {}]
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            response[1] = "Could not get Search State information due to {}::{}".format(type(err).__name__,
                                                                                        sys.exc_info()[1])

        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class GetAllOlts(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olts(request)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# TODO No permissions check on this endpoint as of now. Not sure what an appropriate check(s) is?
class SearchStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, collection):
        response = [status.HTTP_200_OK, ""]

        try:
            try:
                with open(production_search_states[int(collection)], 'r', encoding='utf-8') as state_file:
                    state_data = json.load(state_file)
            except FileNotFoundError:
                with open(development_search_states[int(collection)], 'r', encoding='utf-8') as state_file:
                    state_data = json.load(state_file)

            response[1] = state_data
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            response[1] = "Could not get Search State information due to {}::{}".format(type(err).__name__,
                                                                                        sys.exc_info()[1])

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuDistinctIdentifiers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_distinct_onu_state_identifiers(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltDistinctIdentifiers(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_distinct_olt_state_identifiers(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Handles getting, creating, editing, and deleting PON Automation templates
class AutomationStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, device_type, device_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_automation" in db.get_user_permissions(request.user.email):
            response = db.get_automation_state(device_type, device_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuAutomationStatesForOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_automation" in db.get_user_permissions(request.user.email):
            projection = None
            if request.query_params and request.query_params['project']:
                projection = json.loads(request.query_params["project"])

            response = db.get_onu_automation_states_for_olt(olt_id, projection, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuAutomationConfigsForOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, olt_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_automation" in db.get_user_permissions(request.user.email):
            response = db.get_onu_automation_cfgs_for_olt(olt_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


# Utility function
# Validates that password parameter meets all minimum password requirements
# TODO How do I handle if password requirements in MongoDB do NOT exist?
def validateMinimumPasswordRequirements(request, password, email=None, first_name=None, last_name=None,
                                        current_password=None):
    password_meets_requirements = True
    missing_criteria_message = []

    lowercase_character_count = 0
    uppercase_character_count = 0
    number_character_count = 0
    special_character_count = 0

    # Setting requesters email, if it exists
    try:
        requesting_email = request.user.email
        user_db = request.user.user_db
    except Exception as err:
        requesting_email = 'AnonymousUser'
        user_db = "Default"

    # Retrieve minimum password criteria from MongoDB
    db = get_client_db(user_db, requesting_email)
    minimum_password_criteria = db.get_minimum_password_requirements(request, requesting_email)[1]

    # Minimum Length
    if len(password) < minimum_password_criteria['Minimum Length']:
        password_meets_requirements = False
        missing_criteria_message.append(
            "Password is too short. Password contains only " + str(len(password)) + " character(s); Atleast " + str(
                minimum_password_criteria['Minimum Length']) + " characters are required.")

    # Maximum Length
    if len(password) > minimum_password_criteria['Maximum Length']:
        password_meets_requirements = False
        missing_criteria_message.append(
            "Password is too long. Password contains " + str(len(password)) + " character(s); " + str(
                minimum_password_criteria['Maximum Length']) + " characters are allowed.")

    # Looping through each character in the password
    for char in password:
        # Counting lowercase characters
        if char.islower():
            lowercase_character_count += 1

        # Counting uppercase characters
        if char.isupper():
            uppercase_character_count += 1

        # Counting numeric characters
        if char.isnumeric():
            number_character_count += 1

        # Counting special characters
        if not (char.isnumeric() and char.isspace() and char.isalpha()):
            special_character_count += 1

        # Does not contain excluded Characters
        if char in minimum_password_criteria['Excluded Characters']:
            password_meets_requirements = False
            missing_criteria_message.append(
                "Password contains invalid character(s). Password is not allowed to contain " + char)

    # Verifying lowercase count
    if lowercase_character_count < minimum_password_criteria['Minimum Lowercase Character Count']:
        password_meets_requirements = False
        missing_criteria_message.append("Password requires atleast " + str(
            minimum_password_criteria['Minimum Lowercase Character Count']) + " lowercase character(s); " + str(
            lowercase_character_count) + " lowercase characters are present.")

    # Verifying uppercase count
    if uppercase_character_count < minimum_password_criteria['Minimum Uppercase Character Count']:
        password_meets_requirements = False
        missing_criteria_message.append("Password requires atleast " + str(
            minimum_password_criteria['Minimum Uppercase Character Count']) + " uppercase character(s); " + str(
            uppercase_character_count) + " uppercase characters are present.")

    # Verifying number count
    if number_character_count < minimum_password_criteria['Minimum Number Count']:
        password_meets_requirements = False
        missing_criteria_message.append("Password requires atleast " + str(
            minimum_password_criteria['Minimum Number Count']) + " number(s); " + str(
            number_character_count) + " numbers are present.")

    # Verifying special character count
    if special_character_count < minimum_password_criteria['Minimum Special Character Count']:
        password_meets_requirements = False
        missing_criteria_message.append("Password requires atleast " + str(
            minimum_password_criteria['Minimum Special Character Count']) + " special character(s);" + str(
            special_character_count) + " special characters are present.")

    # ASCII Standard Characters Only
    found_non_ascii_character = False
    if minimum_password_criteria['ASCII Standard Characters Only'] is True:
        try:
            password.encode('ascii')
        except UnicodeEncodeError:
            found_non_ascii_character = True

        if found_non_ascii_character is True:
            password_meets_requirements = False
            missing_criteria_message.append("Invalid password characters.")

    # Does not include email
    if minimum_password_criteria['Exclude Email'] is True and email is not None:
        email_components = email.split('@')
        if len(email_components) < 2 or email_components[0].lower() in password.lower() or email_components[
            1].lower() in password.lower():
            password_meets_requirements = False
            missing_criteria_message.append("Password cannot contain email.")

    # Does not include Name
    # TODO Might want to think about this more. What if users name is a single name that may naturally occur in a password?
    if minimum_password_criteria['Exclude Name'] is True and first_name is not None and last_name is not None:
        if first_name.lower() in password.lower() or last_name.lower() in password.lower():
            password_meets_requirements = False
            missing_criteria_message.append("Password cannot contain name.")

    # Current Password is correct
    if current_password is not None:
        user = authenticate(username=email, password=hashlib.md5(current_password.encode('utf-8')).hexdigest())

        if user is None:
            password_meets_requirements = False
            missing_criteria_message.append("Current password is invalid.")

    # Meets minimum strength
    if minimum_password_criteria['Minimum Password Strength'].lower() != 'none':
        # Password Strength enum
        password_strength_enum = {
            'none': 0,
            'weak': 1,
            'fair': 2,
            'strong': 3,
            'very strong': 4
        }
        password_strength_score = zxcvbn(password)['score']
        minimum_password_strength_as_int = password_strength_enum[
            minimum_password_criteria['Minimum Password Strength'].lower()]

        if password_strength_score < minimum_password_strength_as_int:
            password_meets_requirements = False
            missing_criteria_message.append(
                "Password is not strong enough. A strength of at least " + minimum_password_criteria[
                    'Minimum Password Strength'] + " is required.")

    return {"password_meets_requirements": password_meets_requirements,
            "missing_criteria_message": missing_criteria_message}


DASH_CFG_LIST_SCHEMA = {
    "properties": {
        "type": "array",
        "additionalItems": True,
        "items": {
            "_id": {
                "type": "string",
                "description": "Unique name of the grafana dashboard"
            },
            "URL": {
                "type": "string",
                "description": "The URL string with any query parameters"
            },
            "delete": True
        }

    },
    "required": []
}


class GrafanaOltLinks(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="OLT-DASH-CFG", resource_id_param=None, validate_required=True,
                   schema=DASH_CFG_LIST_SCHEMA)
    def put(self, request, data):
        has_permissions = False
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_network_olts" in db.get_user_permissions(request.user.email):
            has_permissions = True

        if has_permissions:
            response = db.put_grafana_olt_links(data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def get(self, request):
        has_permissions = False
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            has_permissions = True

        if has_permissions:
            response = db.get_grafana_olt_links(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to see this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GrafanaOnuLinks(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="ONU-DASH-CFG", resource_id_param=None, validate_required=True,
                   schema=DASH_CFG_LIST_SCHEMA)
    def put(self, request, data):
        has_permissions = False
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_update_network_onus" in db.get_user_permissions(request.user.email):
            has_permissions = True

        if has_permissions:
            response = db.put_grafana_onu_links(data, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def get(self, request):
        has_permissions = False
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            has_permissions = True

        if has_permissions:
            response = db.get_grafana_onu_links(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to see this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetOnuInventory(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        switch_id = params['switch_id']
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_inventory(request, switch_id)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class SearchFilters(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="SEARCH-CFG", resource_id_param=None, validate_required=True)
    def put(self, request, data, device_type):
        has_permissions = False
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_delete_other_search_filters" and "can_update_other_search_filters" and "can_create_other_search_filters" \
                in db.get_user_permissions(request.user.email):
            has_permissions = True

        if has_permissions:
            response = db.put_search_filters(data, device_type, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))

    def get(self, request, device_type):
        has_permissions = False
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_view_other_search_filters" and "can_read_other_search_filters" \
                in db.get_user_permissions(request.user.email):
            has_permissions = True

        if has_permissions:
            response = db.get_search_filters(device_type, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]

        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetManyMibCurrentStates(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        onus = params["onuIds"]
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_many_mib_current_states(onus, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class ReplaceOnu(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, old_onu_id, new_onu_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_network_olts" in db.get_user_permissions(request.user.email):
            response = db.put_replace_onu(old_onu_id, new_onu_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class ReplaceOlt(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def put(self, request, old_olt_id, new_olt_id):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_update_network_switches" in db.get_user_permissions(
                request.user.email) and "can_update_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.put_replace_olt(old_olt_id, new_olt_id, request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to update this data. See a system administrator.", None, None]
        return PonManagerApiResponse(status=response[0], data=response[1], new_data=response[2], old_data=response[3],
                                     headers=set_response_headers(request))


class AllCntlAlarmHistories(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_all_cntl_alarms_hist(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class AllOltAlarmHistories(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_all_olt_alarms_hist(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class AllOnuAlarmHistories(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_all_onu_alarms_hist(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class CntlAlarmHistoriesCount(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_controllers" in db.get_user_permissions(request.user.email):
            response = db.get_cntl_alarm_hist_count(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OltAlarmHistoriesCount(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_olts" in db.get_user_permissions(request.user.email):
            response = db.get_olt_alarm_hist_count(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class OnuAlarmHistoriesCount(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)
        if "can_read_network_onus" in db.get_user_permissions(request.user.email):
            response = db.get_onu_alarm_hist_count(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


BULK_COMMENT_ALARMS_SCHEMA = {
    "properties": {
        "Alarms": {
            "type": "object",
            "properties": {
                "type": "object",
                "patternproperties": {
                }, "^[0-9]*$": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "Alarm ID": {
                                "type": "number"
                            },
                            "Comment": {
                                "type": "string"
                            },
                            "Comment Operator": {
                                "type": "string"
                            }
                        },
                        "required": {
                            "Alarm ID",
                            "Comment",
                            "Comment Operator"
                        }
                    }
                }
            }
        }
    }, "Device Type": {
        "type": "string"
    },
    "required": {
        "Alarms",
        "Device Type"
    }
}


class BulkCommentAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="bulk_comment_alarms", resource_id_param=None, validate_required=True,
                   schema=BULK_COMMENT_ALARMS_SCHEMA)
    def put(self, request, data):
        response = [status.HTTP_200_OK, "", None, {}]

        try:
            can_update = False
            device_type = data['Device Type']
            db = get_client_db(request.user.user_db, request.user.email)

            if device_type == "ONU" and "can_update_network_onus" in db.get_user_permissions(request.user.email):
                can_update = True
            if device_type == "OLT" and "can_update_network_olts" in db.get_user_permissions(request.user.email):
                can_update = True
            if device_type == "CNTL" and "can_update_network_controllers" in db.get_user_permissions(
                    request.user.email):
                can_update = True

            if can_update:
                response = db.bulk_comment_alarms(request.user.email, data, device_type)
            else:
                response = [status.HTTP_403_FORBIDDEN,
                            "You are not permitted to update this data. See a system administrator."]
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            response[1] = "Could not comment alarms due to {}::{}".format(type(err).__name__,
                                                                          sys.exc_info()[1])
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


BULK_UPDATE_ALARMS_SCHEMA = {
    "properties": {
        "Alarms": {
            "type": "array",
            "items": {
                "type": "object",
                "properties": {
                    "configId": {
                        "type": "number"
                    }, "deviceType": {
                        "type": "string"
                    }, "deviceId": {
                        "type": "string"
                    }, "time": {
                        "type": "string"
                    }, "state": {
                        "type": "string"
                    }, "severity": {
                        "type": "string"
                    }, "alarmID": {
                        "type": "string"
                    }, "instance": {
                        "type": "string"
                    }, "raisedCount": {
                        "type": "number"
                    }, "ack": {
                        "type": "string"
                    }, "lastAckTime": {
                        "type": "string"
                    }, "lastAckUser": {
                        "type": "string"
                    }, "comments": {
                        "type": "string"
                    }, "commentOperator": {
                        "type": "string"
                    }, "lastRaisedTime": {
                        "type": "string"
                    }, "lastClearedTime": {
                        "type": "string"
                    }, "text": {
                        "type": "string"
                    }, "configAckCount": {
                        "type": "number"
                    }, "stateAckCount": {
                        "type": "number"
                    }, "configPurgeCount": {
                        "type": "number"
                    }, "statePurgeCount": {
                        "type": "number"
                    }, "newComment": {
                        "type": "boolean"
                    }
                }
            },
            "required": {
                "configId",
                "deviceType",
                "deviceId",
                "time",
                "state",
                "severity",
                "alarmID",
                "instance",
                "raisedCount",
                "ack",
                "lastAckTime",
                "lastAckUser",
                "comments",
                "commentOperator",
                "lastRaisedTime",
                "lastClearedTime",
                "text",
                "configAckCount",
                "stateAckCount",
                "configPurgeCount",
                "statePurgeCount",
                "newComment"
            }
        }, "Device Type": {
            "type": "string"
        }, "Ack Operator": {
            "type": "string"
        }
    },
    "required": {
        "Alarms",
        "Device Type"
    }
}


class BulkPurgeAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="bulk_purge_alarms", resource_id_param=None, validate_required=True,
                   schema=BULK_UPDATE_ALARMS_SCHEMA)
    def put(self, request, data):
        response = [status.HTTP_200_OK, "", None, {}]

        try:
            can_update = False
            device_type = data['Device Type']

            db = get_client_db(request.user.user_db, request.user.email)

            if device_type == "ONU" and "can_update_network_onus" in db.get_user_permissions(request.user.email):
                can_update = True
            if device_type == "OLT" and "can_update_network_olts" in db.get_user_permissions(request.user.email):
                can_update = True
            if device_type == "CNTL" and "can_update_network_controllers" in db.get_user_permissions(
                    request.user.email):
                can_update = True

            if can_update:
                response = db.bulk_purge_alarms(request.user.email, data, device_type)
            else:
                response = [status.HTTP_403_FORBIDDEN,
                            "You are not permitted to update this data. See a system administrator."]
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            response[1] = "Could not purge alarms due to {}::{}".format(type(err).__name__,
                                                                        sys.exc_info()[1])
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class BulkAcknowledgeAlarms(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    @validate_data(collection="bulk_acknowledge_alarms", resource_id_param=None, validate_required=True,
                   schema=BULK_UPDATE_ALARMS_SCHEMA)
    def put(self, request, data):
        response = [status.HTTP_200_OK, "", None, {}]

        try:
            can_update = False
            device_type = data['Device Type']
            db = get_client_db(request.user.user_db, request.user.email)

            if device_type == "ONU" and "can_update_network_onus" in db.get_user_permissions(request.user.email):
                can_update = True
            if device_type == "OLT" and "can_update_network_olts" in db.get_user_permissions(request.user.email):
                can_update = True
            if device_type == "CNTL" and "can_update_network_controllers" in db.get_user_permissions(
                    request.user.email):
                can_update = True

            if can_update:
                response = db.bulk_acknowledge_alarms(request.user.email, data, device_type)
            else:
                response = [status.HTTP_403_FORBIDDEN,
                            "You are not permitted to update this data. See a system administrator."]
        except Exception as err:
            messages.error(self.request, f"{type(err)}")
            messages.error(self.request, traceback.format_exc(), extra_tags='stacktrace')
            response[0] = status.HTTP_500_INTERNAL_SERVER_ERROR
            response[1] = "Could not acknowledge alarms due to {}::{}".format(type(err).__name__,
                                                                              sys.exc_info()[1])
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetTaskValidationStatus(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        params = request.query_params
        device_ids = params["device_ids"]
        device_type = params["device_type"]
        fw_bank_pointer = params["fw_bank_pointer"]
        fw_versions = params["fw_versions"]
        fw_files = params["fw_files"]

        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_global_config_tasks" in db.get_user_permissions(request.user.email):
            response = db.get_task_validation_status(request.user.email, device_ids, device_type, fw_bank_pointer, fw_versions, fw_files)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetTaskDetailedStatesByPage(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request, task_id):
        params = request.query_params
        filter_value = params["filter_value"]
        sort_active = params["sort_active"]
        sort_direction = params["sort_direction"]
        page_index = params["page_index"]
        page_size = params["page_size"]

        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_global_config_tasks" in db.get_user_permissions(request.user.email):
            response = db.get_task_detailed_states_by_page(request.user.email, task_id, filter_value, sort_active, sort_direction, page_index, page_size)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))


class GetAllPonAutoEnabledTasks(LoginRequiredMixin, generics.GenericAPIView):
    # Setting unauthenticated behavior
    raise_exception = True

    def get(self, request):
        db = get_client_db(request.user.user_db, request.user.email)

        if "can_read_automation" in db.get_user_permissions(request.user.email):
            response = db.get_all_pon_auto_enabled_tasks(request.user.email)
        else:
            response = [status.HTTP_403_FORBIDDEN,
                        "You are not permitted to read this data. See a system administrator."]
        return Response(status=response[0], data=response[1], headers=set_response_headers(request))
