"""
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import copy
import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField, ChoiceField, BooleanField
from drf_spectacular.utils import extend_schema, OpenApiParameter, inline_serializer, OpenApiResponse
from drf_spectacular.types import OpenApiTypes
from rest_framework import status
from rest_framework.generics import GenericAPIView

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.serializers import RequestSerializer
from utils.tools import get_nested_value, PonManagerApiResponse, validate_query_params, validate_data, \
    permission_required_any_of, load_mongo_query_parameter, validate_device_id, permission_required
from utils.serializers import schema, get_schema, RequestSerializer, OkResponseSerializer


class CascConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CASC-CFG')
    @extend_schema(
        operation_id="get_casc_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['cascading'],
        summary="Get all cascading config",
        description=" "
    )
    @method_decorator(permission_required('can_read_cascading_configurations', raise_exception=True))
    @validate_query_params(collection="CASC-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all CASC configurations"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="CASC-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get("database"), collection="CASC-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_casc_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['cascading'],
        summary="Create the provided Cascading Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_cascading_configurations', raise_exception=True))
    @validate_data(collection="CASC-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided CASC config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="CASC-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            casc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details={
                                                 "message": f"CASC configuration with id {casc_id} already exists"})

        return response


class OneCascConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CASC-CFG')

    @extend_schema(
        operation_id="get_one_casc_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['cascading'],
        summary="Get all Cascading Configs",
        description=" "
    )
    @method_decorator(permission_required('can_read_cascading_configurations', raise_exception=True))
    def get(self, request, casc_id, version):
        """ Get the specified CASC configuration """
        res_data = database_manager.find_one(database_id=request.session.get("database"), collection="CASC-CFG",
                                             query={"_id": casc_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                "message": "CASC ID " + str(casc_id) + " has no configuration documents"})

        return response

    @extend_schema(
        operation_id="put_casc_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['cascading'],
        summary="Update the provided Cascading Config",
        description=" "
    )
    @method_decorator(
        permission_required_any_of(['can_update_cascading_configurations', 'can_create_cascading_configurations'],
                                   raise_exception=True))
    @validate_data(collection="CASC-CFG", resource_id_param="casc_id")
    def put(self, request, data, casc_id, version):
        """ Update the provided Cascading Config """
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="CASC-CFG", query={"_id": casc_id},
                                                             new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_casc_config",
        responses=None,
        tags=['cascading'],
        summary="Delete the specified Cascading Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_cascading_configurations', raise_exception=True))
    def delete(self, request, casc_id, version):
        """ Delete the specified Cascading Config """
        database_manager.delete_one(database_id=request.session.get('database'), collection="CASC-CFG",
                                    query={"_id": casc_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
