"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""
import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField, BooleanField, CharField, ListField

from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse, inline_serializer
from drf_spectacular.types import OpenApiTypes
from rest_framework import status
from database_manager import database_manager
from utils.tools import get_nested_value, PonManagerApiResponse, validate_query_params, validate_data, \
    permission_required_any_of, validate_device_id, permission_required
from utils.serializers import schema, get_schema, RequestSerializer, OkResponseSerializer
from utils.schema_helpers import ResponseExample

from docs.json_reference_resolver import generate_swagger_schema_files, PRODUCTION

from api.settings import IN_PRODUCTION
from docs import json_reference_resolver


# from json_validator import JsonSchemaValidator, _get_schema
# validator = JsonSchemaValidator(args.schema_path)

# ==================================================
# ========= One PON Controller State View ==========
# ==================================================
class OneState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-STATE')

    @extend_schema(
        operation_id="get_one_controller_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller states'],
        summary="Get the state for the specified PON Controller",
        description=" "

    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cntl_id, version):
        """Get the state for the specified PON Controller"""
        # Verify validity of cntl_id
        if not validate_device_id(cntl_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid CNTL ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="CNTL-STATE",
                                                 query={"_id": cntl_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "CNTL ID " + str(cntl_id) + " has no state document"})

        return response

    @extend_schema(
        operation_id="delete_one_controller_state",
        responses=None,
        tags=['controller states'],
        summary="Delete the state of the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_controllers', raise_exception=True))
    def delete(self, request, cntl_id, version):
        """Delete the state of the specified PON Controller"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="CNTL-STATE",
                                    query={"_id": cntl_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== PON Controller States View ============
# ==================================================
class States(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-STATE')

    @extend_schema(
        operation_id="get_controller_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller states'],
        summary="Get the states for all PON Controllers",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    @validate_query_params(collection="CNTL-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the states for all PON Controllers"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="CNTL-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="CNTL-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ===== One PON Controller Configuration View ======
# ==================================================
class OneConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-CFG')

    @extend_schema(
        operation_id="get_one_controller_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller config'],
        summary="Get the config for the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cntl_id, version):
        """Get the config for the specified PON Controller"""
        # Verify validity of cntl_id
        if not validate_device_id(cntl_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid CNTL ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="CNTL-CFG",
                                                 query={"_id": cntl_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "CNTL ID " + str(cntl_id) + " has no config document"})

        return response

    swaggerSchema = get_schema('CNTL-CFG')

    @extend_schema(
        operation_id="put_one_controller_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['controller config'],
        summary="Update the config for the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_controllers', 'can_create_network_controllers'],
                                                 raise_exception=True))
    @validate_data(collection="CNTL-CFG", resource_id_param="cntl_id")
    def put(self, request, data, cntl_id, version):
        """Update the config for the specified PON Controller"""
        data['CNTL']['CFG Change Count'] += 1
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="CNTL-CFG", query={"_id": cntl_id},
                                                             new_document=data)
        # Conceal Shared Secret in the log output
        data['RADIUS']['Shared Secret'] = '******'
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            old_document['RADIUS']['Shared Secret'] = '******'
            status_code = status.HTTP_200_OK
        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_controller_config",
        responses=None,
        tags=['controller config'],
        summary="Delete the config of the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_controllers', raise_exception=True))
    def delete(self, request, cntl_id, version):
        """Delete the config of the specified PON Controller"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="CNTL-CFG",
                                    query={"_id": cntl_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ====== PON Controller Configurations View ========
# ==================================================
class Configurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-CFG')

    @extend_schema(
        operation_id="get_controller_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller config'],
        summary="Get the configs for all PON Controllers",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    @validate_query_params(collection="CNTL-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the configs for all PON Controllers"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="CNTL-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="CNTL-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('CNTL-CFG')

    @extend_schema(
        operation_id="post_controller_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['controller config'],
        summary="Create the provided PON Controller config",
        description=" "
    )
    @method_decorator(permission_required('can_create_network_controllers', raise_exception=True))
    @validate_data(collection="CNTL-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided PON Controller config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="CNTL-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            cntl_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={
                "message": f"PON Controller configuration with id {cntl_id} already exists"})

        return response


# ==================================================
# == One PON Controller Alarm Configuration View ===
# ==================================================
class OneAlarmConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-ALARM-CFG')

    @extend_schema(
        operation_id="get_one_controller_alarm_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller alarm config'],
        summary="Get the specified PON Controller Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_alarms', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified PON Controller Alarm Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="CNTL-ALARM-CFG",
                                             query={"_id": cfg_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                "message": "CFG ID " + str(cfg_id) + " has no alarm config documents"})

        return response

    swaggerSchema = get_schema('CNTL-ALARM-CFG')

    @extend_schema(

        operation_id="put_one_controller_alarm_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['controller alarm config'],
        summary="Update the config for the specified PON Controller Alarm Config",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_global_config_alarms', 'can_create_global_config_alarms'],
                                                 raise_exception=True))
    @validate_data(collection="CNTL-ALARM-CFG", resource_id_param="cfg_id")
    def put(self, request, data, cfg_id, version):
        """Update the config for the specified PON Controller Alarm Config"""
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="CNTL-ALARM-CFG", query={"_id": cfg_id},
                                                             new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK
        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_controller_alarm_config",
        responses=None,
        tags=['controller alarm config'],
        summary="Delete the specified PON Controller Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_alarms', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified PON Controller Alarm Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="CNTL-ALARM-CFG",
                                    query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ==== PON Controller Alarm Configurations View ====
# ==================================================
class AlarmConfigurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-ALARM-CFG')

    @extend_schema(
        operation_id="get_controller_alarm_configs",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller alarm config'],
        summary="Get all PON Controller Alarm Configs",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_alarms', raise_exception=True))
    @validate_query_params(collection="CNTL-ALARM-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all PON Controller Alarm Configs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="CNTL-ALARM-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="CNTL-ALARM-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('CNTL-ALARM-CFG')

    @extend_schema(
        operation_id="post_controller_alarm_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            200: inline_serializer(name='CNTL-ALARM-CFG', fields=schema(swaggerSchema)),
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['controller alarm config'],
        summary="Create the provided PON Controller Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_global_config_alarms', raise_exception=True))
    @validate_data(collection="CNTL-ALARM-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided PON Controller Alarm Config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="CNTL-ALARM-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={
                "message": f"PON Controller alarm configuration with id {doc_id} already exists"})

        return response


# ==================================================
# === One PON Controller Alarm History State View ==
# ==================================================
class OneAlarmHistoryState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-ALARM-HIST-STATE')

    @extend_schema(
        operation_id="get_one_controller_alarm_history",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=["controller alarm history"],  # , 'alarm-history', 'get'
        summary='Get the specified PON Controller Alarm History State',
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cntl_id, version):
        """Get the specified PON Controller Alarm History State"""
        # Verify validity of cntl_id
        if not validate_device_id(cntl_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid CNTL ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="CNTL-ALARM-HIST-STATE", query={"_id": cntl_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "CNTL ID " + str(cntl_id) + " has no alarm history document"})

        return response

    @extend_schema(
        operation_id="delete_one_controller_alarm_history",
        responses=None,
        tags=["controller alarm history"],
        summary="Delete the specified PON Controller Alarm Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_controllers', raise_exception=True))
    def delete(self, request, cntl_id, version):
        """Delete the specified PON Controller Alarm Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="CNTL-ALARM-HIST-STATE",
                                    query={"_id": cntl_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ==== PON Controller Alarm History States View ====
# ==================================================
class AlarmHistoryStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-ALARM-HIST-STATE')

    @extend_schema(
        operation_id="get_controller_alarm_history",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=["controller alarm history"],
        summary="Get all PON Controller Alarm History States",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    @validate_query_params(collection="CNTL-ALARM-HIST-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all PON Controller Alarm History States"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="CNTL-ALARM-HIST-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'),
                                             collection="CNTL-ALARM-HIST-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ======= One PON Controller Statistics View =======
# ==================================================
class Statistics(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-STATE')

    @extend_schema(
        operation_id="get_one_controller_stats",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller stats'],
        summary="Get the statistics of the specified PON Controller between the start and end times",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cntl_id, version):
        """Get the statistics of the specified PON Controller between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             data="Parameter 'start-time' is required")
        # Verify validity of cntl_id
        elif not validate_device_id(cntl_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid CNTL ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))

            try:
                state_data = database_manager.find_one(database_id=request.session.get('database'),
                                                       collection="CNTL-STATE",
                                                       query={"_id": cntl_id}, projection={"_id": 0, "CNTL.Version": 1})
                cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
                sub_three_one_version = False

                if float(cntl_version[1:4]) < 3.1:
                    sub_three_one_version = True

                if sub_three_one_version:
                    collection = database.get_collection("STATS-CNTL-{}".format(cntl_id.replace(":", "")))
                    if end_time is None:
                        res_data = list(collection.find({"_id": {"$gte": start_time}}).limit(10000))
                    else:
                        res_data = list(collection.find({"_id": {"$gte": start_time, "$lte": end_time}}).limit(10000))
                else:
                    # For new versions of the DB
                    collection = database.get_collection("STATS-CNTL")
                    if end_time is None:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": cntl_id},
                                {"valid": True},
                                {"_id": {"$gte": start_time}},
                            ]
                        }).limit(10000))
                    else:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": cntl_id},
                                {"valid": True},
                                {"_id": {"$gte": start_time, "$lte": end_time}}
                            ]
                        }).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            if res_data:
                # Add PON Controller ID to response format for easier handling in UI
                for block in res_data:
                    block['mac_address'] = cntl_id
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "CNTL ID " + str(cntl_id) + " has no statistics document"})

        return response

    @extend_schema(
        operation_id="delete_one_controller_stats",
        responses=None,
        tags=['controller stats'],
        summary="Delete the Statistics of the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_controllers', raise_exception=True))
    def delete(self, request, cntl_id, version):
        """Delete the Statistics of the specified PON Controller"""
        database = database_manager.get_database(request.session.get('database'))

        try:
            state_data = database_manager.find_one(database_id=request.session.get('database'), collection="CNTL-STATE",
                                                   query={"_id": cntl_id}, projection={"_id": 0, "CNTL.Version": 1})
            cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
            sub_three_one_version = False

            if float(cntl_version[1:4]) < 3.1:
                sub_three_one_version = True

            if sub_three_one_version:
                collection = database.get_collection("STATS-CNTL-{}".format(cntl_id.replace(":", "")))
                collection.drop()
            else:
                # For new versions of the DB
                collection = database.get_collection("STATS-CNTL")
                collection.update_many({"device ID": cntl_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========= One PON Controller Logs View ===========
# ==================================================
class Logs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-STATE')

    @extend_schema(
        operation_id="get_one_controller_logs",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller logs'],
        summary="Get the logs of the specified PON Controller between the start and end times",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cntl_id, version):
        """Get the logs of the specified PON Controller between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             data="Parameter 'start-time' is required")
        # Verify validity of cntl_id
        elif not validate_device_id(cntl_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid CNTL ID format"})
        else:
            database = database_manager.get_database(request.session.get('database'))

            try:
                state_data = database_manager.find_one(database_id=request.session.get('database'),
                                                       collection="CNTL-STATE",
                                                       query={"_id": cntl_id}, projection={"_id": 0, "CNTL.Version": 1})
                cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
                sub_three_one_version = False

                if float(cntl_version[1:4]) < 3.1:
                    sub_three_one_version = True

                if sub_three_one_version:
                    collection = database.get_collection("SYSLOG-CNTL-{}".format(cntl_id.replace(":", "")))
                    if end_time is None:
                        res_data = list(collection.find({"time": {"$gte": start_time}},
                                                        {"_id": 0, "device ID": 0}))
                    else:
                        res_data = list(collection.find({"time": {"$gte": start_time, "$lte": end_time}},
                                                        {"_id": 0, "device ID": 0}))
                else:
                    collection = database.get_collection("SYSLOG-CNTL")
                    if end_time is None:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": cntl_id},
                                {"valid": True},
                                {"time": {"$gte": start_time}},
                            ]
                        }, {"_id": 0, "device ID": 0}).limit(10000))
                    else:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": cntl_id},
                                {"valid": True},
                                {"time": {"$gte": start_time, "$lte": end_time}}
                            ]
                        }, {"_id": 0, "device ID": 0}).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "CNTL ID " + str(cntl_id) + " has no log documents"})

        return response

    @extend_schema(
        operation_id="delete_one_controller_logs",
        responses=None,
        tags=['controller logs'],
        summary="Delete the Logs of the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_controllers', raise_exception=True))
    def delete(self, request, cntl_id, version):
        """Delete the Logs of the specified PON Controller"""
        database = database_manager.get_database(request.session.get('database'))

        try:
            state_data = database_manager.find_one(database_id=request.session.get('database'), collection="CNTL-STATE",
                                                   query={"_id": cntl_id}, projection={"_id": 0, "CNTL.Version": 1})
            cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
            sub_three_one_version = False

            if float(cntl_version[1:4]) < 3.1:
                sub_three_one_version = True

            if sub_three_one_version:
                collection = database.get_collection("SYSLOG-CNTL-{}".format(cntl_id.replace(":", "")))
                collection.drop()
            else:
                # For new versions of the DB
                collection = database.get_collection("SYSLOG-CNTL")
                collection.update_many({"device ID": cntl_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ==== One PON Controller Automation State View ====
# ==================================================
class OneAutomationState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-AUTO-STATE')

    @extend_schema(
        operation_id="get_one_controller_automation_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller automation state'],
        summary="Get the Automation State of the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cntl_id, version):
        """Get the Automation State of the specified PON Controller"""
        # Verify validity of cntl_id
        if not validate_device_id(cntl_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid CNTL ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="CNTL-AUTO-STATE", query={"_id": cntl_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "CNTL ID " + str(cntl_id) + " has no automation state document"})

        return response

    @extend_schema(
        operation_id="delete_one_controller_automation_state",
        responses=None,
        tags=['controller automation state'],
        summary="Delete the specified PON Controller Automation State",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_controllers', raise_exception=True))
    def delete(self, request, cntl_id, version):
        """Delete the specified PON Controller Automation State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="CNTL-AUTO-STATE",
                                    query={"_id": cntl_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ====== PON Controller Automation States View =====
# ==================================================
class AutomationStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-AUTO-STATE')

    @extend_schema(
        operation_id="get_controller_automation_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller automation state'],
        summary="Get the Automation States of all PON Controllers",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    # TODO @validate_query_params(collection="CNTL-AUTO-STATE")
    def get(self, request, version):
        """Get the Automation States of all PON Controllers"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="CNTL-AUTO-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ==== One PON Controller Automation Config View ===
# ==================================================
class OneAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-AUTO-CFG')

    @extend_schema(
        operation_id="get_one_controller_automation_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller automation config'],
        summary="Get the specified PON Controller Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified PON Controller Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="CNTL-AUTO-CFG",
                                             query={"_id": cfg_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                "message": "CFG ID " + str(cfg_id) + " has no automation documents"})

        return response

    swaggerSchema = get_schema('CNTL-AUTO-CFG')

    @extend_schema(
        operation_id="put_one_controller_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['controller automation config'],
        summary="Update the config for the specified PON Controller Automation Config",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_network_controllers', 'can_create_network_controllers'],
                                                 raise_exception=True))
    @validate_data(collection="CNTL-AUTO-CFG", resource_id_param="cfg_id")
    def put(self, request, data, cfg_id, version):
        """Update the config for the specified PON Controller Automation Config"""
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={
                "message": "Request body must be of format '{ data: <CNTL-AUTO-CFG> }'"})
        else:
            data['_id'] = cfg_id
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                                 collection="CNTL-AUTO-CFG", query={"_id": cfg_id},
                                                                 new_document=data)

        if old_document is None:
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="delete_one_controller_automation_config",
        responses=None,
        tags=['controller automation config'],
        summary="Delete the specified PON Controller Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_controllers', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified PON Controller Automation Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="CNTL-AUTO-CFG",
                                    query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ===== PON Controller Automation Configs View =====
# ==================================================
class AutomationConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-AUTO-CFG')

    @extend_schema(
        auth=None,
        operation_id="get_controller_automation_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller automation config'],
        summary="Get the Automation Configs of all PON Controllers",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    # TODO @validate_query_params(collection="CNTL-AUTO-CFG")
    def get(self, request, version):
        """Get the Automation Configs of all PON Controllers"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="CNTL-AUTO-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('CNTL-AUTO-CFG')

    @extend_schema(
        operation_id="post_controller_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['controller automation config'],
        summary="Create the provided PON Controller Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_network_controllers', raise_exception=True))
    # TODO @validate_data(collection="CNTL-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided PON Controller Automation Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="CNTL-AUTO-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={
                "message": f"PON Controller Automation configuration with id {doc_id} already exists"})

        return response


# ==================================================
# == Global PON Controller Automation Config View ==
# ==================================================
class GlobalAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-AUTO-CFG')

    @extend_schema(
        operation_id="get_global_controller_automation_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller automation config'],
        summary="Get the Global PON Controller Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the Global PON Controller Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="CNTL-AUTO-CFG",
                                             query={"_id": "Global"})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('CNTL-AUTO-CFG')

    @extend_schema(
        operation_id="put_global_controller_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['controller automation config'],
        summary="Update the Global PON Controller Automation Config",
        description=" "
    )
    @method_decorator(
        permission_required_any_of(['can_update_automation', 'can_create_automation'], raise_exception=True))
    @validate_data(collection="CNTL-AUTO-CFG", resource_id_param=None)
    def put(self, request, data, version):
        """Update the Global PON Controller Automation Config"""
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={
                "message": "Request body must be of format '{ data: <CNTL-AUTO-CFG> }'"})
        else:
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                                 collection="CNTL-AUTO-CFG", query={"_id": "Global"},
                                                                 new_document=data)

        if old_document is None:
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    swaggerSchema = get_schema('CNTL-AUTO-CFG')

    @extend_schema(
        operation_id="post_global_controller_automation_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['controller automation config'],
        summary="Create the provided PON Controller Automation Global Config",
        description=" "
    )
    @method_decorator(permission_required('can_create_automation', raise_exception=True))
    # TODO @validate_data(collection="CNTL-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided PON Controller Automation Global Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="CNTL-AUTO-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={
                "message": f"PON Controller Automation configuration with id Global already exists"})

        return response

    @extend_schema(
        operation_id="delete_global_controller_automation_config",
        parameters=[
            OpenApiParameter(name="step", description="Automation step to delete template from",
                             type=OpenApiTypes.STR, required=True),
            OpenApiParameter(name="name", description="Automation template to delete from the specified step",
                             type=OpenApiTypes.STR, required=True)
        ],
        responses=None,
        tags=['controller automation config'],
        summary="Delete the Global PON Controller Automation Config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_automation', raise_exception=True))
    def delete(self, request, version):
        """Delete the Global PON Controller Automation Config"""
        step = request.GET.get('step', None)
        name = request.GET.get('name', None)

        # Return missing parameter response if step or name are undefined
        if step is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'step' is required")
        elif name is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'name' is required")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("CNTL-AUTO-CFG")
            try:
                if step.upper() == "IDENTIFY":
                    collection.update_one({'_id': "Global"},
                                          {"$pull": {"IDENTIFY.Mapping": {"[CNTL-STATE][CNTL][Version]": name}}})
                else:
                    collection.update_one({'_id': "Global"}, {"$unset": {f"{step.upper()}.{name}": ""}})
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

        return response


# ============================================
# ==== One PON Controller Auth State View ====
# ============================================
class CntlOneAuthState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-AUTH-STATE')

    @extend_schema(
        operation_id="get_one_controller_auth_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller auth'],
        summary="Get the Auth State of the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cntl_id, version):
        """Get the Auth State of the specified PON Controller"""
        # Verify validity of cntl_id
        if not validate_device_id(cntl_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid CNTL ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="CNTL-AUTH-STATE", query={"_id": cntl_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "CNTL ID " + str(cntl_id) + " has no auth state document"})

        return response


# ==========================================
# ====== PON Controller Auth State View ====
# ==========================================
class CntlAuthStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-AUTH-STATE')

    @extend_schema(
        operation_id="get_controller_auth_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller auth'],
        summary="Get the Auth States of all PON Controllers",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    # TODO @validate_query_params(collection="CNTL-AUTH-STATE")
    def get(self, request, version):
        """Get the Auth States of all PON Controllers"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="CNTL-AUTH-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# =======================================================
# ==== One PON Controller Host Processing State View ====
# =======================================================
class OneEngineState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-ENGINE-STATE')

    @extend_schema(
        operation_id="get_one_controller_engine_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller engine'],
        summary="Get the Engine of the specified PON Controller",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    def get(self, request, cntl_id, version):
        """Get the Engine of the specified PON Controller"""
        # Verify validity of cntl_id
        if not validate_device_id(cntl_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Invalid CNTL ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'),
                                                 collection="CNTL-ENGINE-STATE", query={"_id": cntl_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "CNTL ID " + str(cntl_id) + " has no engine state document"})

        return response


# =====================================================
# ====== PON Controller Host Processing State View ====
# =====================================================
class EngineStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CNTL-ENGINE-STATE')

    @extend_schema(
        operation_id="get_controller_engine_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['controller engine'],
        summary="Get the Engine States of all PON Controllers",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_controllers', raise_exception=True))
    # TODO @validate_query_params(collection="CNTL-ENGINE-STATE")
    def get(self, request, version):
        """Get the Engine States of all PON Controllers"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="CNTL-ENGINE-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
