"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import PonManagerApiResponse, validate_query_params, validate_mac, permission_required
from utils.serializers import schema, get_schema


# ==================================================
# =============== One CPE State View ===============
# ==================================================
class OneCpeState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CPE-STATE')

    @extend_schema(
        operation_id="get_one_cpe_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['cpe states'],
        summary="Get the specified CPE State",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    def get(self, request, cpe_id, version):
        """Get the specified CPE State"""
        # Verify validity of cpe_id
        if not validate_mac(cpe_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={"message": "Invalid CPE ID format"})
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="CPE-STATE", query={"_id": cpe_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message": "CPE ID "+str(cpe_id)+" has no state document"})

        return response

    @extend_schema(
        operation_id="delete_one_cpe_state",
        responses=None,
        tags=['cpe states'],
        summary="Delete the specified CPE State",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_onus', raise_exception=True))
    def delete(self, request, cpe_id, version):
        """Delete the specified CPE State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="CPE-STATE", query={"_id": cpe_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ================= CPE States View ================
# ==================================================
class CpeStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('CPE-STATE')

    @extend_schema(
        operation_id="get_cpe_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['cpe states'],
        summary="Get all CPE States",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_onus', raise_exception=True))
    @validate_query_params(collection="CPE-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all CPE States"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="CPE-STATE",
                                                 query=query,
                                                 distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="CPE-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
