"""
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""
import argparse
import json
import os
import sys
from pathlib import Path
from jsonref import replace_refs

try:
    build_env = os.environ['BUILDING']
    PRODUCTION = True
except KeyError:
    PRODUCTION = False


def load_schema(file_path):
    """ Return the opened schema file, read only"""
    with open(file_path, 'r') as schema_file:
        return json.load(schema_file)


def resolve_references(original_schema_dir):
    """ Recursively replaces all references in the file"""
    schema_as_dict = replace_refs(
        load_schema(original_schema_dir),
        merge_props=True,
        base_uri=Path(original_schema_dir).absolute().as_uri(),
    )
    return schema_as_dict


def generate_swagger_schema_files(original_schema_path, swagger_schema_path):
    """ Generates a new swagger friendly schema file for regular schema file"""

    # creating the new files for the versioned schema files
    for file in os.listdir(original_schema_path):
        # load the original schema file
        schema_file = os.path.join(original_schema_path, file)

        # resolve the references
        swagger_schema = resolve_references(schema_file)

        # make the swagger schema file path name
        swagger_schema_file = os.path.join(swagger_schema_path, "swagger-" + file)

        # make the directory if it doesn't already exist
        os.makedirs(os.path.dirname(swagger_schema_file), exist_ok=True)

        # write the new schema to the swagger file
        with open(swagger_schema_file, 'w') as swagger_file:
            json.dump(swagger_schema, swagger_file, indent=4)


def main():
    # parsing arguments from build_deb.sh
    print("Creating SwaggerUI files for production")
    parser = argparse.ArgumentParser(add_help=True, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("--src", type=str, help="Schema file source", required=True)
    parser.add_argument("--dst", type=str, help="Swagger file destination", required=True)
    parser.add_argument("--version", type=str, help="PON Manager version", required=True)

    parser.parse_args()
    args = parser.parse_args()

    source = args.src
    # ex: "/tmp/pon_manager/api/schema_files/"
    destination = args.dst
    # ex: "/tmp/pon_manager/build/deb/tibit-ponmgr/api/docs/swagger_schema_files"
    version = args.version
    # ex: "R5.1.0-rcXXXX"

    # cleaning up parseargs
    if "rc" in version or "ma" in version:
        strsplit = version.split("-")
        version = strsplit[0]

    if "," in source:
        strsplit = source.split(",")
        source = strsplit[0]

    # setting paths
    original_schema_path = source + "/" + version
    # ex: "/tmp/pon_manager/api/schema_files/R5.1.0"
    swagger_schema_path = destination
    # ex: "/tmp/pon_manager/build/deb/tibit-ponmgr/api/docs/swagger_schema_files"
    original_unversioned_schema_path = source + "/unversioned_schema_files"

    # create the schema files for versioned schema
    generate_swagger_schema_files(original_schema_path, swagger_schema_path)

    # create the schema files for unversioned schema
    generate_swagger_schema_files(original_unversioned_schema_path, swagger_schema_path)
    print(f"Created swagger schema files for development: {swagger_schema_path}")

def swagger_dev():
    print("Creating SwaggerUI files for development")
    # setting necessary path variables
    from version import VERSION
    version = VERSION
    version = version.replace("E", "R")

    # setting paths
    original_schema_path = "schema_files/" + version
    swagger_schema_path = "docs/swagger_schema"
    original_unversioned_schema_path = "schema_files/unversioned_schema_files"

    # create the schema files for versioned schema
    generate_swagger_schema_files(original_schema_path, swagger_schema_path)

    # create the schema files for unversioned schema
    generate_swagger_schema_files(original_unversioned_schema_path, swagger_schema_path)


if __name__ == '__main__':
    main()
