"""
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import copy
import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField, ChoiceField, BooleanField
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse
from drf_spectacular.types import OpenApiTypes
from rest_framework import status
from rest_framework.generics import GenericAPIView

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.serializers import RequestSerializer
from utils.tools import get_nested_value, PonManagerApiResponse, validate_query_params, validate_data, \
    permission_required_any_of, load_mongo_query_parameter, validate_device_id, permission_required
from utils.serializers import schema, get_schema



# ==================================================
# ============== One OLT State View ================
# ==================================================
class OneDPGState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_one_dpg_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt state'],
        summary="Get the Automation State of the specified OLT",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the Automation State of the specified OLT"""
        # Verify validity of olt_id
        if not validate_device_id(olt_id):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Invalid OLT ID format")
        else:
            res_data = database_manager.find_one(database_id=request.session.get('database'), collection="DPG-STATE",
                                                 query={"_id": olt_id})
            if res_data:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": "OLT ID " + str(olt_id) + " has no DPG state document"})

        return response

    @extend_schema(
        operation_id="delete_one_dpg_state",
        responses=None,
        tags=['olt state'],
        summary="Delete the specified OLT Automation State",
        description=" "
    )
    @method_decorator(permission_required('can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the specified OLT Automation State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="DPG-STATE",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


class DPGStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('OLT-STATE')

    @extend_schema(
        operation_id="get_dpg_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['olt state'],
        summary="Get the states for all OLTs",
        description=" "
    )
    @method_decorator(permission_required('can_read_network_olts', raise_exception=True))
    @validate_query_params(collection="DPG-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the states for all OLTs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="DPG-STATE",
                                                 query=query, distinct=distinct)
        # Filter out '_internal' field
        else:
            if projection:
                keys = list(projection.keys())
                if '_id' not in keys:
                    if any(projection[key] == 0 for key in keys):
                        projection['_internal'] = 0
                else:
                    keys.remove('_id')
                    if len(keys) == 0 and projection['_id'] == 0:
                        projection['_internal'] = 0
                    elif any(projection[key] == 0 for key in keys):
                        projection['_internal'] = 0
            else:
                projection = {'_internal': 0}

            res_data = database_manager.find(database_id=request.session.get('database'), collection="DPG-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
