"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo
from pymongo import monitoring

from log.PonManagerLogger import pon_manager_logger
from utils.interval_timer import database_stats_timer

class MongoServerHeartbeatHandler(monitoring.ServerHeartbeatListener):
    """ Custom handler for pymongo.monitoring.ServerHeartbeatListener responses """

    is_alive: bool = False
    do_log = False  # Makes the first time the sever comes online not create a log message
    stop_listening = False
    status: str = ""
    details: str = "MongoDB server awaiting Heartbeat response"
    name: str = ""
    connection_id : str
    failure_count: int
    primary: dict = {} # Track the current primary replica set member

    recovery_failure_count = 0

    def __init__(self, is_replica_set=False, name=""):
        """ Initializes the heartbeat listener """
        self.is_replica_set = is_replica_set
        if self.is_replica_set:
            self.hosts = {}
        self.name = name
        self.connection_id = ''
        self.failure_count = 0

    def stop(self):
        """ Stop listening for heartbeat to prevent unintended failed message when closing MongoClient """
        self.stop_listening = True

    def started(self, event: monitoring.ServerHeartbeatStartedEvent):
        """ Handle a heart beat started signal """
        if self.is_replica_set:
            if event.connection_id not in self.hosts:
                self.hosts[event.connection_id] = MongoServerHost(event.connection_id)
        else:
            self.connection_id = event.connection_id

    def succeeded(self, event: monitoring.ServerHeartbeatSucceededEvent):
        """ Handle a successful heart beat response """
        if not self.stop_listening:
            host = event.connection_id[0]
            port = event.connection_id[1]
            self.status = "Online"
            if self.is_replica_set:
                is_primary = True
                if event.reply.primary:
                    is_primary = event.reply.primary[0] == host and event.reply.primary[1] == port
                    if self.primary is None: # There is no current primary, this server is the new primary
                        pon_manager_logger.info(f"MongoDB server at {host}:{port} is now primary replica set member")
                elif not event.reply.primary and not self.primary:
                    is_primary = False

                if not self.hosts[event.connection_id].alive and self.do_log:
                    if is_primary:
                        pon_manager_logger.info(f"MongoDB server at {host}:{port} has come online as primary replica set member")
                    else:
                        pon_manager_logger.info(f"MongoDB server at {host}:{port} has come online as a secondary replica set member")

                    self.hosts[event.connection_id].success()
                    # Immediately update database state in the case of status change
                    self.write_statistics()

                else:
                    self.hosts[event.connection_id].success()
                self.primary = event.reply.primary
            elif not self.is_alive and self.do_log:
                pon_manager_logger.info(f"MongoDB server at {host}:{port} has come online")
                self.write_statistics()

            self.is_alive = True
            self.do_log = True

            # Handle excessive logging for recovery of databases
            if self.recovery_failure_count > 0:
                self.recovery_failure_count = self.recovery_failure_count - 1
                if self.recovery_failure_count > 0:
                    self.do_log = False

                # Once the counter reaches 0, set is_alive to false so the message will be logged
                else:
                    # Check logging for replica sets on hosts that may still be active
                    if self.is_replica_set:
                        self.is_alive = any(self.hosts.values())
                    else:
                        self.is_alive = False

    def failed(self, event: monitoring.ServerHeartbeatFailedEvent):
        """ Handle a failed heart beat response """
        if not self.stop_listening:
            host = event.connection_id[0]
            port = event.connection_id[1]
            log_message = f"MongoDB server at {host}:{port} has gone offline due to Heartbeat failure"
            self.status = "Offline"
            self.failure_count = self._incr64(self.failure_count)

            if not isinstance(event.reply, pymongo.errors.NotPrimaryError):
                log_message += f": {str(event.reply)}"

            self.details = log_message

            if self.is_replica_set:
                # If using a replica set the mongo database is still online as long as
                # at least one of the hosts is accessible. Each replica set host server
                # has its own heartbeat, so they must be tracked separately to not
                # override one another.
                is_primary = True
                if self.primary:
                    is_primary = self.primary[0] == host and self.primary[1] == port

                if self.hosts[event.connection_id].alive and self.do_log:
                    if is_primary:
                        pon_manager_logger.error(f"MongoDB primary replica set server at {host}:{port} has gone offline due to Heartbeat failure")
                    else:
                        pon_manager_logger.error(f"MongoDB secondary replica set server at {host}:{port} has gone offline due to Heartbeat failure")

                    self.hosts[event.connection_id].failure()
                    # Immediately update in the case of status change
                    self.write_statistics()
                else:
                    self.hosts[event.connection_id].failure()

                self.is_alive = any(self.hosts.values())
            else:
                if self.is_alive and self.do_log:
                    pon_manager_logger.error(log_message)
                    self.write_statistics()

                self.is_alive = False

            # After a failure, set do_log to false to prevent excessive logging
            self.do_log = False
            self.recovery_failure_count = 3

    def _incr64(self, count):
        """ Check if counter has reached max signed 64-bit integer value. If so, rollover to 1 """
        if not abs(count) > (1 << 63) - 2:  # sub 2 so we rotate 1 number before exceeding max signed 64 bit int
            count += 1
        else:
            count = 1
        return count

    def write_statistics(self):
        """ override the timer and write a group of stats to the DB immediately """
        database_stats_timer.do_function()

    def get_statistics(self):
        """ Gets all heartbeat statistics
        Returns:
            dict: All heartbeat statistics in DB format
        """
        stats = []
        if self.is_replica_set:
            for host, server in self.hosts.items():
                stats.append(self.hosts[host].get_statistics())
        else:
            host_name = str(self.connection_id[0]) + ":" + str(self.connection_id[1])
            # Return as a list to match replica set/multiple host format
            stats.append({
                "Host": host_name,
                "Status": self.status,
                "Failure Count": self.failure_count
            })
        self.clear_statistics()
        return stats

    def clear_statistics(self):
        if self.is_replica_set:
            for connection_id in self.hosts:
                self.hosts[connection_id].failure_count = 0
        else:
            self.failure_count = 0

class MongoServerHost:
    def __init__(self, connection_id):
        self.connection_id = connection_id
        self.status = "Offline"
        self.state = ''
        self.alive = False
        self.failure_count = 0

    def success(self):
        self.status = "Online"
        self.alive = True

    def failure(self):
        self.status = "Offline"
        self.alive = False
        self.failure_count = self._incr64(self.failure_count)

    def get_statistics(self):
        format_host = str(self.connection_id[0]) + ":" + str(self.connection_id[1])
        return {
            "Host": format_host,
            "Status": self.status,
            "Failure Count": self.failure_count
        }

    def _incr64(self, count):
        """ Check if counter has reached max signed 64-bit integer value. If so, rollover to 1 """
        if not abs(count) > (1 << 63) - 2:  # sub 2 so we rotate 1 number before exceeding max signed 64 bit int
            count += 1
        else:
            count = 1
        return count