"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, OpenApiResponse
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import PonManagerApiResponse, get_nested_value, validate_data, validate_query_params, permission_required_any_of, permission_required
from utils.serializers import schema, get_schema


# ==================================================
# ======== All Service Configurations View =========
# ==================================================
class ServiceConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SRV-CFG')

    @extend_schema(
        operation_id="get_service_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['service configs'],
        summary="Get all Service configurations",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_services', raise_exception=True))
    @validate_query_params(collection="SRV-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """ Get all Service configurations """
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="SRV-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get("database"), collection="SRV-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip, next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    swaggerSchema = get_schema('SRV-CFG')

    @extend_schema(
        operation_id="post_service_config",
        request={
            "application/json": schema(swaggerSchema),
        },        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['service configs'],
        summary="Create the provided Service config",
        description=" "
    )
    @method_decorator(permission_required('can_create_global_config_services', raise_exception=True))
    @validate_data(collection="SRV-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided Service config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="SRV-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            srv_cfg_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details={"message":f"Service configuration with id {srv_cfg_id} already exists"})

        return response


# ==================================================
# ========= One Service Configuration View =========
# ==================================================
class OneServiceConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('SRV-CFG')

    @extend_schema(
        operation_id="get_service_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['service configs'],
        summary="Get the specified Service configuration",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_services', raise_exception=True))
    def get(self, request, srv_cfg_id, version):
        """ Get the specified Service configuration """
        res_data = database_manager.find_one(database_id=request.session.get("database"), collection="SRV-CFG", query={"_id": srv_cfg_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"Service configs with ID "+str(srv_cfg_id)+" were not found"})
        return response

    swaggerSchema = get_schema('SRV-CFG')

    @extend_schema(
        operation_id="put_service_config",
        request={
            "application/json": schema(swaggerSchema),
        },        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['service configs'],
        summary="Update the provided Service config",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_global_config_services', 'can_create_global_config_services'], raise_exception=True))
    @validate_data(collection="SRV-CFG", resource_id_param="srv_cfg_id", strict_validation=True)
    def put(self, request, data, srv_cfg_id, version):
        """ Update the provided Service configuration """
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="SRV-CFG",
                                                             query={"_id": srv_cfg_id}, new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)


    @extend_schema(
        operation_id="delete_service_config",
        responses=None,
        tags=['service configs'],
        summary="Delete the provided Service config",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_services', raise_exception=True))
    def delete(self, request, srv_cfg_id, version):
        """ Delete the specified SERVICE configuration """
        database_manager.delete_one(database_id=request.session.get("database"), collection="SRV-CFG", query={"_id": srv_cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
