"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors
import re

from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.fields import JSONField
from drf_spectacular.utils import extend_schema, inline_serializer, OpenApiResponse
from rest_framework import status
from rest_framework.generics import GenericAPIView

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import get_nested_value, PonManagerApiResponse, validate_query_params, validate_data, \
    permission_required_any_of, load_mongo_query_parameter, validate_device_id, permission_required
from utils.serializers import schema, get_schema, RequestSerializer, OkResponseSerializer


# ==================================================
# ========= One Automation Task State View =========
# ==================================================
class OneTaskState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('AUTO-TASK-STATE')
    @extend_schema(
        operation_id="get_one_task_state",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['task automation state'],
        summary="Get the state for the specified Automation Task",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_tasks', raise_exception=True))
    def get(self, request, task_id, version):
        """Get the Automation Task State for the specified ID"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="AUTO-TASK-STATE", query={"_id": task_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"ID "+str(task_id)+" has no automation task state document"})

        return response

    @extend_schema(
        operation_id="delete_one_task_state",
        responses=None,
        tags=['task automation state'],
        summary="Delete the state of the specified Automation Task",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_tasks', raise_exception=True))
    def delete(self, request, task_id, version):
        """Delete the specified Automation Task State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="AUTO-TASK-STATE",
                                    query={"_id": task_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== Automation Task States View ===========
# ==================================================
class TaskStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('AUTO-TASK-STATE')
    @extend_schema(
        operation_id="get_task_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['task automation state'],
        summary="Get the states for all Automation Tasks",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_tasks', raise_exception=True))
    @validate_query_params(collection="AUTO-TASK-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all Automation Task States"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="AUTO-TASK-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ======== One Automation Task Config View =========
# ==================================================
class OneTaskConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('AUTO-TASK-CFG')
    @extend_schema(
        operation_id="get_one_task_config",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['task automation config'],
        summary="Get the config for the specified Automation Task",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_tasks', raise_exception=True))
    def get(self, request, task_id, version):
        """Get the specified Automation Task Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="AUTO-TASK-CFG", query={"_id": task_id})
        if res_data:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={"message":"ID "+str(task_id)+" has no automation task configuration document"})

        return response

    @extend_schema(
        operation_id="put_one_task_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['task automation config'],
        summary="Update the config for the specified Automation Task",
        description=" "
    )
    @method_decorator(permission_required_any_of(['can_update_global_config_tasks', 'can_create_global_config_tasks'], raise_exception=True))
    @validate_data(collection="AUTO-TASK-CFG", resource_id_param="task_id")
    def put(self, request, data, task_id, version):
        """Update the config for the specified Automation Task Config"""
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="AUTO-TASK-CFG", query={"_id": task_id},
                                                             new_document=data)

        if old_document is None:
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="delete_one_task_config",
        responses=None,
        tags=['task automation config'],
        summary="Delete the config of the specified Automation Task",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_tasks', raise_exception=True))
    def delete(self, request, task_id, version):
        """Delete the specified Automation Task Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="AUTO-TASK-CFG",
                                    query={"_id": task_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========= Automation Task Configs View ===========
# ==================================================
class TaskConfigurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('AUTO-TASK-CFG')
    @extend_schema(
        operation_id="get_task_configs",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['task automation config'],
        summary="Get the configs for all Automation Tasks",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_tasks', raise_exception=True))
    @validate_query_params(collection="AUTO-TASK-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all Automation Task Configs"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="AUTO-TASK-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_task_config",
        request={
            "application/json": schema(swaggerSchema),
        },
        responses={
            201: OpenApiResponse(response=schema(swaggerSchema),
                                 description='Created'),
        },
        tags=['task automation config'],
        summary="Create the provided Automation Task config",
        description=" "
    )
    @method_decorator(permission_required('can_create_global_config_tasks', raise_exception=True))
    @validate_data(collection="AUTO-TASK-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided ID Automation Task Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="AUTO-TASK-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details={"message":f"Automation task configuration with id {doc_id} already exists"})

        return response


# ==================================================
# ====== Automation Detailed Task States View ======
# ==================================================
class DetailedTaskStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    swaggerSchema = get_schema('AUTO-TASK-DETAILED-STATE')
    @extend_schema(
        operation_id="get_detailed_task_states",
        responses={
            200: OpenApiResponse(response=schema(swaggerSchema),
                                 description='OK'),
        },
        tags=['task automation detailed state'],
        summary="Get the detailed states for all specified Automation Tasks",
        description=" "
    )
    @method_decorator(permission_required('can_read_global_config_tasks', raise_exception=True))
    @validate_query_params(collection="AUTO-TASK-DETAILED-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all Automation Task Detailed States """
        query_params = request.query_params
        task_ids = query_params['task_ids'].split('%2C,')

        for index, task in enumerate(task_ids):
            # Check if the last 3 digits are %2C
            if task[len(task) - 3:] == "%2C":
                task_ids[index] = task[:len(task) - 3]

        details_query = {'Task ID': {'$in': task_ids}}

        task_filter = ''
        if 'filter' in query_params:
            task_filter = query_params['filter']
            details_query = {
                'Task ID': {'$in': task_ids},
                '$or': [
                        {'Device ID': {'$regex': re.compile(r"(?i)" + task_filter)}},
                        {'Task.Status': {'$regex': re.compile(r"(?i)" + task_filter)}}]
            }
        task_limit = 0
        if 'limit' in query_params:
            task_limit = int(query_params['limit'])
        task_skip = 0
        if 'skip' in query_params:
            task_skip = query_params['skip']


        res_data = database_manager.find(database_id=request.session.get('database'),
                                         collection="AUTO-TASK-DETAILED-STATE",
                                         query=details_query,
                                         skip=task_skip,
                                         limit=task_limit)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="delete_detailed_task_states",
        responses=None,
        tags=['task automation detailed state'],
        summary="Delete the detailed states for all specified Automation Tasks",
        description=" "
    )
    @method_decorator(permission_required('can_delete_global_config_tasks', raise_exception=True))
    def delete(self, request, version):
        """Delete the specified Automation Task Config"""
        query_params = request.query_params
        task_id = query_params['task_id']
        query_doc = {'Task ID': task_id}
        device_ids = []
        if 'device_ids' in query_params and query_params['device_ids'] != '':
            device_ids = query_params['device_ids'].split(',')
            query_doc = {'Task ID': task_id, 'Device ID': {'$in': device_ids}}

        database_manager.delete_many(database_id=request.session.get('database'), collection="AUTO-TASK-DETAILED-STATE",
                                    query=query_doc)

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
