import os
import keyring
from keyrings.cryptfile.cryptfile import CryptFileKeyring
from log.PonManagerLogger import pon_manager_logger

FILE_ENCODING = 'utf-8'

def lookup_indices_value_in_dict(the_dict, indices) -> any:
    """
    Lookup the value in the dictionary using the list of index values

    Args:
        the_dict: Dictionary to look up value in

        indices: List of index values for the lookup

    Returns:
        Value at the location or None if not found
    """
    if isinstance(the_dict, dict):
        if indices and not isinstance(indices, list):
            indices = list(indices)
        for index in indices:
            if isinstance(the_dict, dict) and index in the_dict:
                # Note on the final index, the_dict is the value
                the_dict = the_dict[index]
            else:
                return None
    return the_dict


def delete_keyring_username(keyring_cryptfile_path: str, keyring_key_path: str, username: str):
    if os.path.isfile(keyring_key_path):
        key_file = open(keyring_key_path, "r", encoding=FILE_ENCODING)
        keyring_key = key_file.read()
    else:
        raise ValueError("Keyring Key File Not Found: {}".format(keyring_key_path))

    if os.path.isfile(keyring_cryptfile_path):
        kr = CryptFileKeyring()
        kr.keyring_key = keyring_key
        keyring.set_keyring(kr)

        keyring.delete_password('ponmgr', username)


def get_mongo_password(database_dict: dict, username = 'user_database.password') -> str:
    """
    Gets MongoDB password utilizing the correct mechanism (.json, Python Keyring)

    Args:
        database_dict (dict) : document with password and password_opts arguments
        **username (string) : username to retrieve from the keyring data file

    Returns:
        (str) MongoDB password
    """
    password = ""
    password_type = lookup_indices_value_in_dict(database_dict, ["password_opts", "type"])

    # Get MongoDB password from .json or python keyring. If type not specified, use cleartext mechanism
    if password_type is None or ((isinstance(password_type, str) and password_type.lower().strip()) == "password") or ("auth_enable" in database_dict and database_dict["auth_enable"] == False):
        password = lookup_indices_value_in_dict(database_dict, ["password"])

    elif isinstance(password_type, str) and password_type.lower().strip() == "keyring":
        keyring_path = lookup_indices_value_in_dict(database_dict, ["password_opts", "keyring_path"])
        keyring_key_path = lookup_indices_value_in_dict(database_dict,
                                                        ["password_opts", "keyring_key_path"])
        if isinstance(keyring_path, str) and os.path.isfile(keyring_path):
            # Set keyring path env var
            os.environ["KEYRING_CRYPTFILE_PATH"] = keyring_path

            # Read keyring passphrase
            if isinstance(keyring_key_path, str) and os.path.isfile(keyring_key_path):

                try:
                    phrase = open(f"{keyring_key_path}", "r").read()

                    # Instantiating keyring object
                    kr = CryptFileKeyring()
                    kr.keyring_key = phrase

                    # Unlocking Keyring
                    keyring.set_keyring(kr)

                    # Getting Mongo auth credential
                    keyring_value = keyring.get_password("ponmgr", username)
                except Exception:
                    keyring_value = ''
                    pon_manager_logger.error(f"{username} - Unable to unlock python keyring")

                if isinstance(keyring_value, str) and keyring_value:
                    password = keyring_value
                else:  # Bad value stored
                    pon_manager_logger.error(f"{username} password keyring value not a valid value: {keyring_value}")
            else:  # Bad keyring key path
                pon_manager_logger.error(f"{username} password keyring key path is not a valid file path: {keyring_key_path}")
        else:  # Bad keyring path
            pon_manager_logger.error(f"{username} password keyring path is not a valid file path: {keyring_path}")
    else:  # Bad type value
        pon_manager_logger.error(f"{username} password type is not a valid value: {password_type}")

    return password