#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
""" Activate firmware for an ONU.

This MCMS REST API example script activates firmware for an ONU by updating
the firmware bank pointer. The firmware bank pointer is used to select the
firmware image for the ONU to boot from. Set the bank to 65535 to disable
firmware upgrade.

Example:

  ./activate_onu_firmware.py --url https://10.2.10.29/api --user <email> --password <password> --onu TBITc84c0083 --bank 1

usage: activate_onu_firmware.py --bank BANK [-d DATABASE] [-f FILTER] [-h]
                                [-l URL] --onu ONU [-p PASSWORD] [-u USER]
                                [-v]

optional arguments:
  --bank BANK           ONU firmware bank to activate (0, 1, or 65535 to
                        disable firmware download) (default: None)
  -d DATABASE, --db DATABASE
                        Name of the database. (default: Default)
  -f FILTER, --filter FILTER
                        Filter by ONU state (e.g., Dregistered, Disabled,
                        Disallowed-Admin, Disallowed-Error, Dying-Gasp,
                        Registered, Unprovisioned, Unspecified) (default:
                        None)
  -h, --help            Show this help message and exit.
  -l URL, --url URL     URL of the MCMS API server (e.g.,
                        https://10.2.10.29/api). (default:
                        https://10.2.10.29/api)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: tibit)
  -u USER, --user USER  User email to authenticate with. (default:
                        tibit@tibitcom.com)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys
from api_client import ApiClient

FW_UPGRADE_DISABLED = 65535

def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("--bank", action="store", dest="bank", default=None, required=True, help="ONU firmware bank to activate (0, 1, or 65535 to disable firmware download)")
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("-f", "--filter", action="store", dest="filter", default=None, required=False, help="Filter by ONU state (e.g., Dregistered, Disabled, Disallowed-Admin, Disallowed-Error, Dying-Gasp, Registered, Unprovisioned, Unspecified)")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Validate the bank pointer value
    fw_bank_ptr = int(args.bank)
    if fw_bank_ptr not in (0, 1, FW_UPGRADE_DISABLED):
        print(f"ERROR: Invalid value for firmware bank {fw_bank_ptr}.")
        sys.exit(1)

    # Get the ONU-CFG for this ONU
    status, onu_cfg = api_client.request("GET", f"/v1/onus/configs/{args.onu}/")
    if status != 200 or not onu_cfg:
        print(f"ERROR: Failed to read state for ONU {args.onu}.")
        sys.exit(1)

    # Set the ONU Firmware Bank pointer
    onu_cfg["ONU"]["FW Bank Ptr"] = fw_bank_ptr

    # Update the ONU configuration
    status, _ = api_client.request("PUT", f"/v1/onus/configs/{args.onu}/", data={"data" : onu_cfg})
    if status not in (200, 201):
        print(f"ERROR: Configuration failed for ONU {args.onu}.")
        sys.exit(1)

    # Logout of the web server to terminate the session
    api_client.logout()

    # Success
    if fw_bank_ptr == FW_UPGRADE_DISABLED:
        print(f"Firmware upgrade disabled for ONU {args.onu}.")
    else:
        print(f"Active firmware bank set to {fw_bank_ptr} for ONU {args.onu}")

if __name__ == '__main__':
    main()
