#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
""" Configure an Add CTag service on an ONU.

This MCMS REST API example script configures double-tagged VLAN service on
an ONU, where the OLT adds/removes an outer STag (0x88A8) and the ONU
adds/removes an inner CTag (0x8100). The script includes options for
configuring DHCP Relay and SLA.

As part of configuring the service for the ONU, the script determines the
XGS-PON ONU ID and TCONT ALLOC ID/XGem Port to use for this service. If a
value has not already been configured for this ONU, the script finds the
next free value.

Example:

  ./config_addctag_service.py --url https://10.2.10.29/api --user <email> --password <password> \
     --olt e8:b4:70:70:0c:9c --olt_tag 200 --onu ALPHe30cadcf --onu_tag 25

usage: config_addctag_service.py [-d DATABASE]
                                 [--dhcp-circuit-id DHCP_CIRCUIT_ID]
                                 [--dhcp-remote-id DHCP_REMOTE_ID] [--dhcpv4]
                                 [--dhcpv6] [-h] [-l URL] --olt OLT --olt_tag
                                 OLT_TAG --onu ONU --onu_tag ONU_TAG
                                 [-p PASSWORD] [--sla SLA] [-u USER] [-v]

optional arguments:
  -d DATABASE, --db DATABASE
                        Name of the database. (default: Default)
  --dhcp-circuit-id DHCP_CIRCUIT_ID
                        DHCP Relay Option 82, Circuit ID (default: )
  --dhcp-remote-id DHCP_REMOTE_ID
                        DHCP Relay Option 82, Remote ID (default: )
  --dhcpv4              Enable DHCPv4 Relay (default: False)
  --dhcpv6              Enable DHCPv6 Relay (default: False)
  -h, --help            Show this help message and exit.
  -l URL, --url URL     URL of the MCMS API server (e.g.,
                        https://10.2.10.29/api). (default:
                        https://10.2.10.29/api)
  --olt OLT             OLT MAC Address (e.g., e8:b4:70:70:0c:9c) (default:
                        None)
  --olt_tag OLT_TAG     Tag to be added by the OLT (default: None)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  --onu_tag ONU_TAG     Tag to be added by the ONU (default: None)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: tibit)
  --sla SLA             SLA (default: Max)
  -u USER, --user USER  User email to authenticate with. (default:
                        tibit@tibitcom.com)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import datetime
import sys
from typing import Any, List, Optional
from api_client import ApiClient
from api_utilities import dict_read, dict_to_list, list_to_dict

# OLT ONU ID Constants
ONU_ID_MIN = 1
ONU_ID_MAX = 128

# OLT Alloc ID Constants
ALLOC_ID_MIN = 1154
ALLOC_ID_MAX = 1534
ALLOC_IDS_RESERVED = [0x4FF, 0x5FF]

# OLT Flooding ID Constants
PON_FLOOD_ID_MIN = 1919
PON_FLOOD_ID_MAX = 2044

def get_onu_ids(onu_inventory: dict) -> List[str]:
    """ Get the list of ONU IDs configured in ONU Inventory.

    Args:
        onu_inventory: ONU Inventory.

    Returns:
        onu_ids: List of ONU IDs configured in Inventory.
    """
    onu_ids = {}
    for onu_serial_number, onu in onu_inventory.items():
        if 'ALLOC ID (OMCC)' in onu:
            onu_id = onu['ALLOC ID (OMCC)']
            onu_ids[onu_id] = onu_serial_number

    return onu_ids

def get_next_onu_id(onu_inventory: dict) -> int:
    """ Get the next available ONU ID not configured in ONU Inventory.

    Args:
        onu_inventory: ONU Inventory.

    Returns:
        next_onu_id: The next available ONU ID not configured in Inventory.
    """
    next_onu_id = None
    onu_ids = get_onu_ids(onu_inventory)
    if len(onu_ids.keys()) == 0:
        next_onu_id = ONU_ID_MIN
    else:
        for value in range(ONU_ID_MIN, ONU_ID_MAX+1):
            if value not in onu_ids.keys():
                next_onu_id = value
                break

    return next_onu_id

def get_alloc_ids(onu_inventory: dict)-> List[str]:
    """ Get the list of ALLOC IDs configured in ONU Inventory.

    Args:
        onu_inventory: ONU Inventory.

    Returns:
        onu_ids: List of ALLOC IDs configured in Inventory.
    """
    alloc_ids = {}
    for onu_serial_number, onu in onu_inventory.items():
        for key, value in onu.items():
            if key.startswith("OLT-Service"):
                alloc_ids[value] = onu_serial_number

    return alloc_ids

def get_next_alloc_id(onu_inventory: dict) -> int:
    """ Get the next available ALLOC ID not configured in ONU Inventory.

    Args:
        onu_inventory: ONU Inventory.

    Returns:
        next_onu_id: The next available ALLOC ID not configured in Inventory.
    """
    next_alloc_id = None
    alloc_ids = get_alloc_ids(onu_inventory)
    if len(alloc_ids.keys()) == 0:
        next_alloc_id = ALLOC_ID_MIN
    else:
        # Need to exclude values reserved by HW: 0x4FF, 0x5FF (already excluded from the range above)
        for value in range(ALLOC_ID_MIN, ALLOC_ID_MAX+1):
            if value not in alloc_ids:
                if value not in ALLOC_IDS_RESERVED:
                    next_alloc_id = value
                    break

    return next_alloc_id

def get_next_flooding_id(nni_networks: dict) -> int:
    """ Get the next available PON Flooding ID not configured in NNI Networks.

    Args:
        onu_inventory: NNI Networks.

    Returns:
        next_onu_id: The next available PON Flooding ID not configured in NNI Networks.
    """
    next_flooding_id = None

    flooding_ids = {}
    for tag_match, nni_network in nni_networks.items():
        if 'PON FLOOD ID' in nni_network:
            flood_id = nni_network['PON FLOOD ID']
            flooding_ids[flood_id] = tag_match

    if len(flooding_ids.keys()) == 0:
        next_flooding_id = PON_FLOOD_ID_MIN
    else:
        for value in range(PON_FLOOD_ID_MIN, PON_FLOOD_ID_MAX+1):
            if value not in flooding_ids.keys():
                next_flooding_id = value
                break

    return next_flooding_id


def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("--dhcp-circuit-id", action="store", dest="dhcp_circuit_id", default="", required=False, help="DHCP Relay Option 82, Circuit ID")
    parser.add_argument("--dhcp-remote-id", action="store", dest="dhcp_remote_id", default="", required=False, help="DHCP Relay Option 82, Remote ID")
    parser.add_argument("--dhcpv4", action="store_true", dest="dhcpv4_enable", default=False, required=False, help="Enable DHCPv4 Relay")
    parser.add_argument("--dhcpv6", action="store_true", dest="dhcpv6_enable", default=False, required=False, help="Enable DHCPv6 Relay")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., e8:b4:70:70:0c:9c)")
    parser.add_argument("--olt_tag", action="store", dest="olt_tag", default=None, required=True, help="Tag to be added by the OLT")
    parser.add_argument("--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("--onu_tag", action="store", dest="onu_tag", default=None, required=True, help="Tag to be added by the ONU")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("--sla", action="store", dest="sla", default="Max", required=False, help="SLA")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Construct the NNI and PON Tags for this service
    nni_tags = f"s{args.olt_tag}.c{args.onu_tag}.c0"
    pon_tags = f"s0.c{args.onu_tag}.c0"

    # Determine Protocol Filters based on the arguments passed to the script.
    if args.dhcpv4_enable:
        dhcpv4_protocol_filter = "UMT"
    else:
        dhcpv4_protocol_filter = "pass"

    if args.dhcpv6_enable:
        dhcpv6_protocol_filter = "UMT"
    else:
        dhcpv6_protocol_filter = "pass"

    #
    # Configure the OLT
    #

    # Get the OLT configuration that this ONU is attached to
    status, olt_cfg = api_client.request("GET", f"/v1/olts/configs/{args.olt}/")
    if status != 200 or not olt_cfg:
        print(f"ERROR: Configuration for OLT {args.olt} does not exist.")
        sys.exit(1)

    # Configure ONU Inventory
    onu_inventory = dict_read(olt_cfg, "ONUs")

    # If the ONU ID is not already configured, get the next available ONU ID
    onu_id = dict_read(onu_inventory, f"{args.onu}.ALLOC ID (OMCC)")
    if not onu_id:
        onu_id = get_next_onu_id(onu_inventory)
    if not onu_id:
        print(f"ERROR: No available ONU IDs for OLT {args.olt}.")
        sys.exit(1)

    # If the Alloc ID/XGem Port is not already configured, get the next available Alloc ID
    alloc_id = dict_read(onu_inventory, f"{args.onu}.OLT-Service 0")
    if not alloc_id:
        alloc_id = get_next_alloc_id(onu_inventory)
    if not alloc_id:
        print(f"ERROR: No available Alloc IDs/XGem Ports for OLT {args.olt}.")
        sys.exit(1)

    # Add or update the ONU Inventory entry for this ONU
    if args.onu not in olt_cfg["ONUs"]:
        # Create a new entry with default values
        olt_cfg["ONUs"][args.onu] = {}
        olt_cfg["ONUs"][args.onu]["Disable"] = False
    olt_cfg["ONUs"][args.onu]["ALLOC ID (OMCC)"] = onu_id
    olt_cfg["ONUs"][args.onu]["OLT-Service 0"] = alloc_id

    # Get the list of NNI Networks
    nni_networks = dict_read(olt_cfg, "NNI Networks", default=[])
    nni_networks = list_to_dict(nni_networks, "TAG MATCH")

    # If this is a Shared VLAN, determine the next available flooding ID. Otherwise, use the XGem Port value as the flooding ID.
    pon_flood_id = alloc_id

    # Add or update the NNI Network entry for this service.
    if nni_tags not in nni_networks:
        # Create a new entry with default values
        nni_networks[nni_tags] = {}
        nni_networks[nni_tags]["Filter"] = {"EAPOL" : "pass", "PPPoE" : "pass"}
        nni_networks[nni_tags]["Learning Limit"] = 2046
        nni_networks[nni_tags]["SLA-CFG"] = {"Source" : "N/A"}
    nni_networks[nni_tags]["TAG MATCH"] = nni_tags
    nni_networks[nni_tags]["PON FLOOD ID"] = pon_flood_id
    nni_networks[nni_tags]["Filter"]["DHCPv4"] = dhcpv4_protocol_filter
    nni_networks[nni_tags]["Filter"]["DHCPv6"] = dhcpv6_protocol_filter
    olt_cfg["NNI Networks"] = dict_to_list(nni_networks, "TAG MATCH")

    # Update the OLT configuration
    status, result = api_client.request("PUT", f"/v1/olts/configs/{args.olt}/", data={"data" : olt_cfg})
    if status != 200:
        print(f"ERROR: Configuration failed for OLT {args.olt}.")
        sys.exit(1)


    #
    # Configure the ONU
    #

    # Get the configuration for this ONU
    status, onu_cfg = api_client.request("GET", f"/v1/onus/configs/{args.onu}/")
    if status != 200 or not onu_cfg:
        # If this is a pre-provisioned ONU, there is no existing ONU
        # configuration. Start with the default configuration.
        status, onu_cfg = api_client.request("GET", f"/v1/onus/configs/Default/")
        if status != 200 or not onu_cfg:
            print(f"ERROR: Failed to read Default configuration for ONU {args.onu}.")
            sys.exit(1)

        # Update the default configuration with the resource ID for this ONU
        onu_cfg["_id"] = args.onu
        onu_cfg["NETCONF"]["Name"] = args.onu
        onu_cfg["ONU"]["Create Date"] = str(datetime.datetime.utcnow())

    # Set the NNI Network, PON Network, and SLA on OLT-Service 0
    onu_cfg["OLT-Service 0"]["Enable"] = True
    onu_cfg["OLT-Service 0"]["DHCP"]["Circuit ID"] = args.dhcp_circuit_id
    onu_cfg["OLT-Service 0"]["DHCP"]["Remote ID"] = args.dhcp_remote_id
    onu_cfg["OLT-Service 0"]["Filter"]["DHCPv4"] = dhcpv4_protocol_filter
    onu_cfg["OLT-Service 0"]["Filter"]["DHCPv6"] = dhcpv6_protocol_filter
    onu_cfg["OLT-Service 0"]["NNI Network"] = [nni_tags]
    onu_cfg["OLT-Service 0"]["PON Network"] = [pon_tags]
    onu_cfg["OLT-Service 0"]["SLA-CFG"] = args.sla

    # Set the ONU Service Configuration (Unmodified or AddCTag)
    onu_cfg["ONU"]["SRV-CFG"] = "Add CTag"
    onu_cfg["ONU"]["CVID"] = int(args.onu_tag)

    # Configure the allowed OLT for this ONU
    onu_cfg["OLT"]["MAC Address"] = [args.olt]

    # Update the ONU configuration
    status, result = api_client.request("PUT", f"/v1/onus/configs/{args.onu}/", data={"data" : onu_cfg})
    if status not in (200, 201):
        print(f"ERROR: Configuration failed for ONU {args.onu}.")
        sys.exit(1)

    # Logout of the web server to terminate the session
    api_client.logout()

if __name__ == '__main__':
    main()
