#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
""" List OLT devices.

This MCMS REST API example script lists all OLT devices, switch port
information, and online status.

Example:

  ./get_olt_status.py --url http://10.2.10.29/api --user <email> --password <password>


usage: get_olt_status.py [-d DATABASE] [-h] [-l URL] [-p PASSWORD] [-u USER]
                         [-v]

optional arguments:
  -d DATABASE, --db DATABASE
                        Name of the database. (default: Default)
  -h, --help            Show this help message and exit.
  -l URL, --url URL     URL of the MCMS API server (e.g.,
                        https://10.2.10.29/api). (default:
                        https://10.2.10.29/api)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: tibit)
  -u USER, --user USER  User email to authenticate with. (default:
                        tibit@tibitcom.com)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import datetime
from api_client import ApiClient
from api_utilities import natural_sort_key


def controller_is_online(timestamp):
    """ Returns True if the PON Controller is online, and False otherwise. """
    seconds_in_day = 60 * 60 * 24
    cntl_time = datetime.datetime.strptime(timestamp, '%Y-%m-%d %H:%M:%S.%f')
    time_delta = datetime.datetime.utcnow() - cntl_time
    # The controller is considered online if the timestamp has been updated within six minutes
    return divmod(time_delta.days * seconds_in_day + time_delta.seconds, 60)[0] <= 6


def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Get the PON Controller states
    controller_states = []
    status, response_data = api_client.request("GET", f"/v1/controllers/states/?projection=_id=1,System Status=1,Time=1")
    if status == 200 and response_data:
        controller_states = response_data

    # Determine the current OLT online/offline status
    # Loop through the PON Controller states sorted by Time (oldest first)
    # for mac, state in sorted(controller_states, key=lambda item: item['Time']):
    online_olts = []
    for controller_state in sorted(controller_states, key=lambda state: state['Time']):
        if controller_is_online(controller_state['Time']):
            for olt_mac, status in controller_state['System Status'].items():
                if status['OLT State'] == "Primary" or status['OLT State'] == "Secondary":
                    online_olts.append(olt_mac)

    # Get OLT state data
    olt_states = []
    status, response_data = api_client.request("GET", f"/v1/olts/states/")
    if status == 200 and response_data:
        olt_states = response_data

    olt_status = {}
    for olt_state in olt_states:
        olt_mac = olt_state['_id']
        olt_status[olt_mac] = {}
        olt_status[olt_mac]['id'] = olt_mac
        if olt_mac not in online_olts:
            olt_status[olt_mac]['status'] = "offline"
        elif olt_state['OLT']['Laser Shutdown'] != "Laser ON":
            olt_status[olt_mac]['status'] = olt_state['OLT']['Laser Shutdown']
        else:
            olt_status[olt_mac]['status'] = "online"
        olt_status[olt_mac]['firmware_version'] = olt_state['OLT']['FW Version']
        olt_status[olt_mac]['switch_id'] = olt_state['Switch']['Chassis ID']
        olt_status[olt_mac]['switch_port'] = olt_state['Switch']['Port ID']

    # Get the OLT Configurations and append the list of pre-provisioned OLTs
    status, olt_configs = api_client.request("GET", f"/v1/olts/configs/?projection=_id=1,Switch=1")
    if status == 200 and olt_configs:
        for olt_config in olt_configs:
            olt_mac = olt_config['_id']
            if olt_mac == "Default":
                continue
            if olt_mac not in olt_status:
                olt_status[olt_mac] = {}
                olt_status[olt_mac]['id'] = olt_mac
                olt_status[olt_mac]['status'] = "provisioned"
                olt_status[olt_mac]['firmware_version'] = ""
                if 'Chassis ID' in olt_config['Switch']:
                    olt_status[olt_mac]['switch_id'] = olt_config['Switch']['Chassis ID']
                else:
                    olt_status[olt_mac]['switch_id'] = ""
                if 'Port ID' in olt_config['Switch']:
                    olt_status[olt_mac]['switch_port'] = olt_config['Switch']['Port ID']
                else:
                    olt_status[olt_mac]['switch_port'] = ""

    # Logout of the web server to terminate the session
    api_client.logout()

    # Display the ONU registration status of for the OLT
    print(f"\nOLT Devices:")
    print("{:<17}  {:<8}  {:<17}  {:<12}  {}".format('Switch', 'Port', 'OLT MAC', 'Status', 'FW Version'))
    for status in sorted(sorted(sorted(list(olt_status.values()), key=lambda k: k['id']), key=lambda k: natural_sort_key(k['switch_port'])), key=lambda k: k['switch_id']):
        print("{:<17}  {:<8}  {:<17}  {:<12}  {}".format(
            status['switch_id'],
            status['switch_port'],
            status['id'],
            status['status'],
            status['firmware_version']))

if __name__ == '__main__':
    main()
