/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.netconf.notifications;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.text.ParsePosition;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.util.Date;
import java.util.function.Function;
import org.opendaylight.netconf.api.NetconfMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public final class NetconfNotification
extends NetconfMessage {
    private static final Logger LOG = LoggerFactory.getLogger(NetconfNotification.class);
    public static final String NOTIFICATION = "notification";
    public static final String NOTIFICATION_NAMESPACE = "urn:ietf:params:netconf:capability:notification:1.0";
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ISO_DATE_TIME;
    public static final Function<Date, String> RFC3339_DATE_FORMATTER = date -> DATE_TIME_FORMATTER.format(date.toInstant().atOffset(ZoneOffset.UTC));
    public static final Function<String, Date> RFC3339_DATE_PARSER = time -> {
        try {
            ZonedDateTime localDateTime = ZonedDateTime.parse(time, DATE_TIME_FORMATTER);
            boolean startAt = false;
            TemporalAccessor parsed = DATE_TIME_FORMATTER.parse((CharSequence)time, new ParsePosition(0));
            int nanoOfSecond = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.NANO_OF_SECOND);
            long reminder = nanoOfSecond % 1000000;
            if (reminder != 0L) {
                StringBuilder reminderBuilder = new StringBuilder(String.valueOf(reminder));
                while (reminderBuilder.length() < 6) {
                    reminderBuilder.insert(0, '0');
                }
                while (reminderBuilder.charAt(reminderBuilder.length() - 1) == '0') {
                    reminderBuilder.deleteCharAt(reminderBuilder.length() - 1);
                }
                LOG.warn("Fraction of second is cut to three digits. Value that was cut {}", (Object)reminderBuilder.toString());
            }
            return Date.from(Instant.from(localDateTime));
        }
        catch (DateTimeParseException exception) {
            Date res = NetconfNotification.handlePotentialLeapSecond(time);
            if (res != null) {
                return res;
            }
            throw exception;
        }
    };
    public static final String EVENT_TIME = "eventTime";
    public static final Date UNKNOWN_EVENT_TIME = new Date(0L);
    private final Date eventTime;

    private static Date handlePotentialLeapSecond(String time) {
        boolean offset = false;
        TemporalAccessor parsed = DATE_TIME_FORMATTER.parseUnresolved(time, new ParsePosition(0));
        if (parsed == null) {
            return null;
        }
        int secondOfMinute = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.SECOND_OF_MINUTE);
        int hourOfDay = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.HOUR_OF_DAY);
        int minuteOfHour = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.MINUTE_OF_HOUR);
        if (secondOfMinute != 60 || minuteOfHour != 59 || hourOfDay != 23) {
            return null;
        }
        LOG.trace("Received time contains leap second, adjusting by replacing the second-of-minute of 60 with 59 {}", (Object)time);
        secondOfMinute = 59;
        int year = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.YEAR);
        int monthOfYear = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.MONTH_OF_YEAR);
        int dayOfMonth = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.DAY_OF_MONTH);
        int nanoOfSecond = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.NANO_OF_SECOND);
        int offsetSeconds = NetconfNotification.getFieldFromTemporalAccessor(parsed, ChronoField.OFFSET_SECONDS);
        LocalDateTime currentTime = LocalDateTime.of(year, monthOfYear, dayOfMonth, hourOfDay, minuteOfHour, secondOfMinute, nanoOfSecond);
        OffsetDateTime dateTimeWithZoneOffset = currentTime.atOffset(ZoneOffset.ofTotalSeconds(offsetSeconds));
        return RFC3339_DATE_PARSER.apply(dateTimeWithZoneOffset.toString());
    }

    private static int getFieldFromTemporalAccessor(TemporalAccessor accessor, ChronoField field) {
        return accessor.isSupported(field) ? (int)accessor.getLong(field) : 0;
    }

    public NetconfNotification(Document notificationContent) {
        this(notificationContent, new Date());
    }

    @SuppressFBWarnings(value={"EI_EXPOSE_REP2"})
    public NetconfNotification(Document notificationContent, Date eventTime) {
        super(NetconfNotification.wrapNotification(notificationContent, eventTime));
        this.eventTime = eventTime;
    }

    @SuppressFBWarnings(value={"EI_EXPOSE_REP"})
    public Date getEventTime() {
        return this.eventTime;
    }

    private static Document wrapNotification(Document notificationContent, Date eventTime) {
        Preconditions.checkNotNull((Object)notificationContent);
        Preconditions.checkNotNull((Object)eventTime);
        Element baseNotification = notificationContent.getDocumentElement();
        Element entireNotification = notificationContent.createElementNS(NOTIFICATION_NAMESPACE, NOTIFICATION);
        entireNotification.appendChild(baseNotification);
        Element eventTimeElement = notificationContent.createElementNS(NOTIFICATION_NAMESPACE, EVENT_TIME);
        eventTimeElement.setTextContent(NetconfNotification.getSerializedEventTime(eventTime));
        entireNotification.appendChild(eventTimeElement);
        notificationContent.appendChild(entireNotification);
        return notificationContent;
    }

    private static String getSerializedEventTime(Date eventTime) {
        return RFC3339_DATE_FORMATTER.apply(eventTime);
    }
}

