/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.raft;

import akka.persistence.RecoveryCompleted;
import akka.persistence.SnapshotOffer;
import com.google.common.base.Stopwatch;
import java.util.Collections;
import org.opendaylight.controller.cluster.PersistentDataProvider;
import org.opendaylight.controller.cluster.raft.RaftActorContext;
import org.opendaylight.controller.cluster.raft.RaftActorRecoveryCohort;
import org.opendaylight.controller.cluster.raft.ReplicatedLog;
import org.opendaylight.controller.cluster.raft.ReplicatedLogEntry;
import org.opendaylight.controller.cluster.raft.ReplicatedLogImpl;
import org.opendaylight.controller.cluster.raft.base.messages.ApplySnapshot;
import org.opendaylight.controller.cluster.raft.persisted.ApplyJournalEntries;
import org.opendaylight.controller.cluster.raft.persisted.DeleteEntries;
import org.opendaylight.controller.cluster.raft.persisted.EmptyState;
import org.opendaylight.controller.cluster.raft.persisted.MigratedSerializable;
import org.opendaylight.controller.cluster.raft.persisted.ServerConfigurationPayload;
import org.opendaylight.controller.cluster.raft.persisted.Snapshot;
import org.opendaylight.controller.cluster.raft.persisted.UpdateElectionTerm;
import org.opendaylight.controller.cluster.raft.protobuff.client.messages.PersistentPayload;
import org.slf4j.Logger;

class RaftActorRecoverySupport {
    private final RaftActorContext context;
    private final RaftActorRecoveryCohort cohort;
    private int currentRecoveryBatchCount;
    private boolean dataRecoveredWithPersistenceDisabled;
    private boolean anyDataRecovered;
    private boolean hasMigratedDataRecovered;
    private Stopwatch recoveryTimer;
    private final Logger log;

    RaftActorRecoverySupport(RaftActorContext context, RaftActorRecoveryCohort cohort) {
        this.context = context;
        this.cohort = cohort;
        this.log = context.getLogger();
    }

    boolean handleRecoveryMessage(Object message, PersistentDataProvider persistentProvider) {
        this.log.trace("{}: handleRecoveryMessage: {}", (Object)this.context.getId(), message);
        boolean bl = this.anyDataRecovered = this.anyDataRecovered || !(message instanceof RecoveryCompleted);
        if (RaftActorRecoverySupport.isMigratedSerializable(message)) {
            this.hasMigratedDataRecovered = true;
        }
        boolean recoveryComplete = false;
        if (message instanceof UpdateElectionTerm) {
            this.context.getTermInformation().update(((UpdateElectionTerm)message).getCurrentTerm(), ((UpdateElectionTerm)message).getVotedFor());
        } else if (message instanceof SnapshotOffer) {
            this.onRecoveredSnapshot((SnapshotOffer)message);
        } else if (message instanceof ReplicatedLogEntry) {
            this.onRecoveredJournalLogEntry((ReplicatedLogEntry)message);
        } else if (message instanceof ApplyJournalEntries) {
            this.onRecoveredApplyLogEntries(((ApplyJournalEntries)message).getToIndex());
        } else if (message instanceof DeleteEntries) {
            this.onDeleteEntries((DeleteEntries)message);
        } else if (message instanceof ServerConfigurationPayload) {
            this.context.updatePeerIds((ServerConfigurationPayload)message);
        } else if (message instanceof RecoveryCompleted) {
            recoveryComplete = true;
            this.onRecoveryCompletedMessage(persistentProvider);
        }
        return recoveryComplete;
    }

    private void possiblyRestoreFromSnapshot() {
        Snapshot restoreFromSnapshot = this.cohort.getRestoreFromSnapshot();
        if (restoreFromSnapshot == null) {
            return;
        }
        if (this.anyDataRecovered) {
            this.log.warn("{}: The provided restore snapshot was not applied because the persistence store is not empty", (Object)this.context.getId());
            return;
        }
        this.log.debug("{}: Restore snapshot: {}", (Object)this.context.getId(), (Object)restoreFromSnapshot);
        this.context.getSnapshotManager().apply(new ApplySnapshot(restoreFromSnapshot));
    }

    private ReplicatedLog replicatedLog() {
        return this.context.getReplicatedLog();
    }

    private void initRecoveryTimer() {
        if (this.recoveryTimer == null) {
            this.recoveryTimer = Stopwatch.createStarted();
        }
    }

    private void onRecoveredSnapshot(SnapshotOffer offer) {
        this.log.debug("{}: SnapshotOffer called.", (Object)this.context.getId());
        this.initRecoveryTimer();
        Snapshot snapshot = (Snapshot)offer.snapshot();
        for (ReplicatedLogEntry entry : snapshot.getUnAppliedEntries()) {
            if (!RaftActorRecoverySupport.isMigratedPayload(entry)) continue;
            this.hasMigratedDataRecovered = true;
        }
        if (!this.context.getPersistenceProvider().isRecoveryApplicable()) {
            snapshot = Snapshot.create(EmptyState.INSTANCE, Collections.emptyList(), -1L, -1L, -1L, -1L, snapshot.getElectionTerm(), snapshot.getElectionVotedFor(), snapshot.getServerConfiguration());
        }
        this.context.setReplicatedLog(ReplicatedLogImpl.newInstance(snapshot, this.context));
        this.context.setLastApplied(snapshot.getLastAppliedIndex());
        this.context.setCommitIndex(snapshot.getLastAppliedIndex());
        this.context.getTermInformation().update(snapshot.getElectionTerm(), snapshot.getElectionVotedFor());
        Stopwatch timer = Stopwatch.createStarted();
        if (!(snapshot.getState() instanceof EmptyState)) {
            this.cohort.applyRecoverySnapshot(snapshot.getState());
        }
        if (snapshot.getServerConfiguration() != null) {
            this.context.updatePeerIds(snapshot.getServerConfiguration());
        }
        timer.stop();
        this.log.info("Recovery snapshot applied for {} in {}: snapshotIndex={}, snapshotTerm={}, journal-size={}", new Object[]{this.context.getId(), timer.toString(), this.replicatedLog().getSnapshotIndex(), this.replicatedLog().getSnapshotTerm(), this.replicatedLog().size()});
    }

    private void onRecoveredJournalLogEntry(ReplicatedLogEntry logEntry) {
        if (this.log.isDebugEnabled()) {
            this.log.debug("{}: Received ReplicatedLogEntry for recovery: index: {}, size: {}", new Object[]{this.context.getId(), logEntry.getIndex(), logEntry.size()});
        }
        if (RaftActorRecoverySupport.isServerConfigurationPayload(logEntry)) {
            this.context.updatePeerIds((ServerConfigurationPayload)logEntry.getData());
        }
        if (RaftActorRecoverySupport.isMigratedPayload(logEntry)) {
            this.hasMigratedDataRecovered = true;
        }
        if (this.context.getPersistenceProvider().isRecoveryApplicable()) {
            this.replicatedLog().append(logEntry);
        } else if (!RaftActorRecoverySupport.isPersistentPayload(logEntry)) {
            this.dataRecoveredWithPersistenceDisabled = true;
        }
    }

    private void onRecoveredApplyLogEntries(long toIndex) {
        if (!this.context.getPersistenceProvider().isRecoveryApplicable()) {
            this.dataRecoveredWithPersistenceDisabled = true;
            return;
        }
        long lastUnappliedIndex = this.context.getLastApplied() + 1L;
        if (this.log.isDebugEnabled()) {
            this.log.debug("{}: Received apply journal entries for recovery, applying to state: {} to {}", new Object[]{this.context.getId(), lastUnappliedIndex, toIndex});
        }
        long lastApplied = lastUnappliedIndex - 1L;
        for (long i = lastUnappliedIndex; i <= toIndex; ++i) {
            ReplicatedLogEntry logEntry = this.replicatedLog().get(i);
            if (logEntry != null) {
                ++lastApplied;
            } else {
                this.log.error("{}: Log entry not found for index {}", (Object)this.context.getId(), (Object)i);
                break;
            }
            this.batchRecoveredLogEntry(logEntry);
        }
        this.context.setLastApplied(lastApplied);
        this.context.setCommitIndex(lastApplied);
    }

    private void onDeleteEntries(DeleteEntries deleteEntries) {
        if (this.context.getPersistenceProvider().isRecoveryApplicable()) {
            this.replicatedLog().removeFrom(deleteEntries.getFromIndex());
        } else {
            this.dataRecoveredWithPersistenceDisabled = true;
        }
    }

    private void batchRecoveredLogEntry(ReplicatedLogEntry logEntry) {
        this.initRecoveryTimer();
        int batchSize = this.context.getConfigParams().getJournalRecoveryLogBatchSize();
        if (!RaftActorRecoverySupport.isServerConfigurationPayload(logEntry)) {
            if (this.currentRecoveryBatchCount == 0) {
                this.cohort.startLogRecoveryBatch(batchSize);
            }
            this.cohort.appendRecoveredLogEntry(logEntry.getData());
            if (++this.currentRecoveryBatchCount >= batchSize) {
                this.endCurrentLogRecoveryBatch();
            }
        }
    }

    private void endCurrentLogRecoveryBatch() {
        this.cohort.applyCurrentLogRecoveryBatch();
        this.currentRecoveryBatchCount = 0;
    }

    private void onRecoveryCompletedMessage(PersistentDataProvider persistentProvider) {
        if (this.currentRecoveryBatchCount > 0) {
            this.endCurrentLogRecoveryBatch();
        }
        String recoveryTime = "";
        if (this.recoveryTimer != null) {
            this.recoveryTimer.stop();
            recoveryTime = " in " + this.recoveryTimer.toString();
            this.recoveryTimer = null;
        }
        this.log.info("{}: Recovery completed {} - Switching actor to Follower - last log index = {}, last log term = {}, snapshot index = {}, snapshot term = {}, journal size = {}", new Object[]{this.context.getId(), recoveryTime, this.replicatedLog().lastIndex(), this.replicatedLog().lastTerm(), this.replicatedLog().getSnapshotIndex(), this.replicatedLog().getSnapshotTerm(), this.replicatedLog().size()});
        if (this.dataRecoveredWithPersistenceDisabled || this.hasMigratedDataRecovered && !this.context.getPersistenceProvider().isRecoveryApplicable()) {
            if (this.hasMigratedDataRecovered) {
                this.log.info("{}: Saving snapshot after recovery due to migrated messages", (Object)this.context.getId());
            } else {
                this.log.info("{}: Saving snapshot after recovery due to data persistence disabled", (Object)this.context.getId());
            }
            Snapshot snapshot = Snapshot.create(EmptyState.INSTANCE, Collections.emptyList(), -1L, -1L, -1L, -1L, this.context.getTermInformation().getCurrentTerm(), this.context.getTermInformation().getVotedFor(), this.context.getPeerServerInfo(true));
            persistentProvider.saveSnapshot((Object)snapshot);
            persistentProvider.deleteMessages(persistentProvider.getLastSequenceNumber());
        } else if (this.hasMigratedDataRecovered) {
            this.log.info("{}: Snapshot capture initiated after recovery due to migrated messages", (Object)this.context.getId());
            this.context.getSnapshotManager().capture(this.replicatedLog().last(), -1L);
        } else {
            this.possiblyRestoreFromSnapshot();
        }
    }

    private static boolean isServerConfigurationPayload(ReplicatedLogEntry repLogEntry) {
        return repLogEntry.getData() instanceof ServerConfigurationPayload;
    }

    private static boolean isPersistentPayload(ReplicatedLogEntry repLogEntry) {
        return repLogEntry.getData() instanceof PersistentPayload;
    }

    private static boolean isMigratedPayload(ReplicatedLogEntry repLogEntry) {
        return RaftActorRecoverySupport.isMigratedSerializable(repLogEntry.getData());
    }

    private static boolean isMigratedSerializable(Object message) {
        return message instanceof MigratedSerializable && ((MigratedSerializable)message).isMigrated();
    }
}

