/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.netconf.nettyutil;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.handler.ssl.SslHandler;
import io.netty.util.Timeout;
import io.netty.util.Timer;
import io.netty.util.TimerTask;
import io.netty.util.concurrent.Promise;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.opendaylight.netconf.api.NetconfDocumentedException;
import org.opendaylight.netconf.api.NetconfMessage;
import org.opendaylight.netconf.api.NetconfSessionListener;
import org.opendaylight.netconf.api.NetconfSessionPreferences;
import org.opendaylight.netconf.api.messages.NetconfHelloMessage;
import org.opendaylight.netconf.nettyutil.AbstractNetconfSession;
import org.opendaylight.netconf.nettyutil.NetconfSessionNegotiator;
import org.opendaylight.netconf.nettyutil.handler.FramingMechanismHandlerFactory;
import org.opendaylight.netconf.nettyutil.handler.NetconfChunkAggregator;
import org.opendaylight.netconf.nettyutil.handler.NetconfMessageToXMLEncoder;
import org.opendaylight.netconf.nettyutil.handler.NetconfXMLToHelloMessageDecoder;
import org.opendaylight.netconf.nettyutil.handler.NetconfXMLToMessageDecoder;
import org.opendaylight.netconf.util.messages.FramingMechanism;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

public abstract class AbstractNetconfSessionNegotiator<P extends NetconfSessionPreferences, S extends AbstractNetconfSession<S, L>, L extends NetconfSessionListener<S>>
extends ChannelInboundHandlerAdapter
implements NetconfSessionNegotiator<S> {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractNetconfSessionNegotiator.class);
    public static final String NAME_OF_EXCEPTION_HANDLER = "lastExceptionHandler";
    protected final P sessionPreferences;
    protected final Channel channel;
    private final Promise<S> promise;
    private final L sessionListener;
    private Timeout timeout;
    private State state = State.IDLE;
    private final Timer timer;
    private final long connectionTimeoutMillis;

    protected AbstractNetconfSessionNegotiator(P sessionPreferences, Promise<S> promise, Channel channel, Timer timer, L sessionListener, long connectionTimeoutMillis) {
        this.channel = Objects.requireNonNull(channel);
        this.promise = Objects.requireNonNull(promise);
        this.sessionPreferences = sessionPreferences;
        this.timer = timer;
        this.sessionListener = sessionListener;
        this.connectionTimeoutMillis = connectionTimeoutMillis;
    }

    protected final void startNegotiation() {
        if (this.ifNegotiatedAlready()) {
            LOG.debug("Negotiation on channel {} already started", (Object)this.channel);
        } else {
            Optional<SslHandler> sslHandler = AbstractNetconfSessionNegotiator.getSslHandler(this.channel);
            if (sslHandler.isPresent()) {
                ((SslHandler)sslHandler.get()).handshakeFuture().addListener(future -> {
                    Preconditions.checkState((boolean)future.isSuccess(), (Object)"Ssl handshake was not successful");
                    LOG.debug("Ssl handshake complete");
                    this.start();
                });
            } else {
                this.start();
            }
        }
    }

    protected final synchronized boolean ifNegotiatedAlready() {
        return this.state != State.IDLE;
    }

    private static Optional<SslHandler> getSslHandler(Channel channel) {
        SslHandler sslHandler = (SslHandler)channel.pipeline().get(SslHandler.class);
        return sslHandler == null ? Optional.absent() : Optional.of((Object)sslHandler);
    }

    public P getSessionPreferences() {
        return this.sessionPreferences;
    }

    private void start() {
        NetconfHelloMessage helloMessage = this.sessionPreferences.getHelloMessage();
        this.channel.pipeline().addLast(NAME_OF_EXCEPTION_HANDLER, (ChannelHandler)new ExceptionHandlingInboundChannelHandler());
        this.sendMessage((NetconfMessage)helloMessage);
        this.replaceHelloMessageOutboundHandler();
        this.changeState(State.OPEN_WAIT);
        this.timeout = this.timer.newTimeout(new TimerTask(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void run(Timeout timeout) {
                1 var2_2 = this;
                synchronized (var2_2) {
                    if (AbstractNetconfSessionNegotiator.this.state != State.ESTABLISHED) {
                        LOG.debug("Connection timeout after {}, session is in state {}", (Object)timeout, (Object)AbstractNetconfSessionNegotiator.this.state);
                        if (!this.isPromiseFinished()) {
                            LOG.warn("Netconf session was not established after {}", (Object)AbstractNetconfSessionNegotiator.this.connectionTimeoutMillis);
                            AbstractNetconfSessionNegotiator.this.changeState(State.FAILED);
                            AbstractNetconfSessionNegotiator.this.channel.close().addListener(future -> {
                                if (future.isSuccess()) {
                                    LOG.debug("Channel {} closed: success", (Object)future.channel());
                                } else {
                                    LOG.warn("Channel {} closed: fail", (Object)future.channel());
                                }
                            });
                        }
                    } else if (AbstractNetconfSessionNegotiator.this.channel.isOpen()) {
                        AbstractNetconfSessionNegotiator.this.channel.pipeline().remove(AbstractNetconfSessionNegotiator.NAME_OF_EXCEPTION_HANDLER);
                    }
                }
            }

            private boolean isPromiseFinished() {
                return AbstractNetconfSessionNegotiator.this.promise.isDone() || AbstractNetconfSessionNegotiator.this.promise.isCancelled();
            }
        }, this.connectionTimeoutMillis, TimeUnit.MILLISECONDS);
    }

    private void cancelTimeout() {
        if (this.timeout != null) {
            this.timeout.cancel();
        }
    }

    protected final S getSessionForHelloMessage(NetconfHelloMessage netconfMessage) throws NetconfDocumentedException {
        Document doc = netconfMessage.getDocument();
        if (this.shouldUseChunkFraming(doc)) {
            this.insertChunkFramingToPipeline();
        }
        this.changeState(State.ESTABLISHED);
        return this.getSession(this.sessionListener, this.channel, netconfMessage);
    }

    private void insertChunkFramingToPipeline() {
        AbstractNetconfSessionNegotiator.replaceChannelHandler(this.channel, "frameEncoder", FramingMechanismHandlerFactory.createHandler(FramingMechanism.CHUNK));
        AbstractNetconfSessionNegotiator.replaceChannelHandler(this.channel, "aggregator", (ChannelHandler)new NetconfChunkAggregator());
    }

    private boolean shouldUseChunkFraming(Document doc) {
        return AbstractNetconfSessionNegotiator.containsBase11Capability(doc) && AbstractNetconfSessionNegotiator.containsBase11Capability(this.sessionPreferences.getHelloMessage().getDocument());
    }

    protected final void replaceHelloMessageInboundHandler(S session) {
        ChannelHandler helloMessageHandler = AbstractNetconfSessionNegotiator.replaceChannelHandler(this.channel, "netconfMessageDecoder", (ChannelHandler)new NetconfXMLToMessageDecoder());
        Preconditions.checkState((boolean)(helloMessageHandler instanceof NetconfXMLToHelloMessageDecoder), (String)"Pipeline handlers misplaced on session: %s, pipeline: %s", session, (Object)this.channel.pipeline());
        Iterable<NetconfMessage> netconfMessagesFromNegotiation = ((NetconfXMLToHelloMessageDecoder)helloMessageHandler).getPostHelloNetconfMessages();
        for (NetconfMessage message : netconfMessagesFromNegotiation) {
            ((AbstractNetconfSession)((Object)session)).handleMessage(message);
        }
    }

    private void replaceHelloMessageOutboundHandler() {
        AbstractNetconfSessionNegotiator.replaceChannelHandler(this.channel, "netconfMessageEncoder", (ChannelHandler)new NetconfMessageToXMLEncoder());
    }

    private static ChannelHandler replaceChannelHandler(Channel channel, String handlerKey, ChannelHandler decoder) {
        return channel.pipeline().replace(handlerKey, handlerKey, decoder);
    }

    protected abstract S getSession(L var1, Channel var2, NetconfHelloMessage var3) throws NetconfDocumentedException;

    private synchronized void changeState(State newState) {
        LOG.debug("Changing state from : {} to : {} for channel: {}", new Object[]{this.state, newState, this.channel});
        Preconditions.checkState((boolean)AbstractNetconfSessionNegotiator.isStateChangePermitted(this.state, newState), (String)"Cannot change state from %s to %s for chanel %s", (Object)((Object)this.state), (Object)((Object)newState), (Object)this.channel);
        this.state = newState;
    }

    private static boolean containsBase11Capability(Document doc) {
        NodeList nList = doc.getElementsByTagNameNS("urn:ietf:params:xml:ns:netconf:base:1.0", "capability");
        for (int i = 0; i < nList.getLength(); ++i) {
            if (!nList.item(i).getTextContent().contains("urn:ietf:params:netconf:base:1.1")) continue;
            return true;
        }
        return false;
    }

    private static boolean isStateChangePermitted(State state, State newState) {
        if (state == State.IDLE && newState == State.OPEN_WAIT) {
            return true;
        }
        if (state == State.OPEN_WAIT && newState == State.ESTABLISHED) {
            return true;
        }
        if (state == State.OPEN_WAIT && newState == State.FAILED) {
            return true;
        }
        LOG.debug("Transition from {} to {} is not allowed", (Object)state, (Object)newState);
        return false;
    }

    protected final void negotiationSuccessful(S session) {
        LOG.debug("Negotiation on channel {} successful with session {}", (Object)this.channel, session);
        this.channel.pipeline().replace((ChannelHandler)this, "session", session);
        this.promise.setSuccess(session);
    }

    protected void negotiationFailed(Throwable cause) {
        LOG.debug("Negotiation on channel {} failed", (Object)this.channel, (Object)cause);
        this.channel.close();
        this.promise.setFailure(cause);
    }

    protected final void sendMessage(NetconfMessage msg) {
        this.channel.writeAndFlush((Object)msg).addListener(f -> {
            if (!f.isSuccess()) {
                LOG.warn("Failed to send message on channel {}. {}", new Object[]{this.channel, msg, f.cause()});
                this.negotiationFailed(f.cause());
            } else {
                LOG.warn("Session negotiation started with hello message on channel {}. {}", (Object)this.channel, (Object)msg);
            }
        });
    }

    public final void channelActive(ChannelHandlerContext ctx) {
        LOG.debug("Starting session negotiation on channel {}", (Object)this.channel);
        try {
            this.startNegotiation();
        }
        catch (Exception e) {
            LOG.warn("Unexpected negotiation failure", (Throwable)e);
            this.negotiationFailed(e);
        }
    }

    public final void channelRead(ChannelHandlerContext ctx, Object msg) {
        LOG.debug("Negotiation read invoked on channel {}", (Object)this.channel);
        try {
            this.handleMessage((NetconfHelloMessage)msg);
        }
        catch (Exception e) {
            LOG.debug("Unexpected error while handling negotiation message {}", msg, (Object)e);
            this.negotiationFailed(e);
        }
    }

    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        LOG.info("Unexpected error during negotiation", cause);
        this.negotiationFailed(cause);
    }

    protected abstract void handleMessage(NetconfHelloMessage var1) throws Exception;

    private final class ExceptionHandlingInboundChannelHandler
    extends ChannelInboundHandlerAdapter {
        private ExceptionHandlingInboundChannelHandler() {
        }

        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
            LOG.warn("An exception occurred during negotiation with {}", (Object)AbstractNetconfSessionNegotiator.this.channel.remoteAddress(), (Object)cause);
            AbstractNetconfSessionNegotiator.this.cancelTimeout();
            AbstractNetconfSessionNegotiator.this.negotiationFailed(cause);
            AbstractNetconfSessionNegotiator.this.changeState(State.FAILED);
        }
    }

    protected static enum State {
        IDLE,
        OPEN_WAIT,
        FAILED,
        ESTABLISHED;

    }
}

