/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.aaa.cert.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Base64;
import java.util.Date;
import javax.xml.bind.DatatypeConverter;
import org.bouncycastle.asn1.x509.X509Name;
import org.bouncycastle.jce.PKCS10CertificationRequest;
import org.bouncycastle.jce.X509Principal;
import org.bouncycastle.x509.X509V3CertificateGenerator;
import org.opendaylight.aaa.cert.impl.KeyStoreConstant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ODLKeyTool {
    private static final Logger LOG = LoggerFactory.getLogger(ODLKeyTool.class);
    private final String workingDir;

    protected ODLKeyTool() {
        this.workingDir = KeyStoreConstant.KEY_STORE_PATH;
        KeyStoreConstant.createDir(this.workingDir);
    }

    public ODLKeyTool(String workingDirectory) {
        this.workingDir = workingDirectory;
        KeyStoreConstant.createDir(this.workingDir);
    }

    public KeyStore addCertificate(KeyStore keyStore, String certificate, String alias, boolean deleteOld) {
        try {
            X509Certificate newCert = this.getCertificate(certificate);
            if (keyStore.isCertificateEntry(alias) && deleteOld) {
                keyStore.deleteEntry(alias);
            }
            if (newCert == null) {
                LOG.warn("{} Not a valid certificate {}", (Object)alias, (Object)certificate);
                return null;
            }
            keyStore.setCertificateEntry(alias, newCert);
            return keyStore;
        }
        catch (KeyStoreException e) {
            LOG.error("failed to add certificate", (Throwable)e);
            return null;
        }
    }

    public byte[] convertKeystoreToBytes(KeyStore keyStore, String keystorePassword) {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        try {
            keyStore.store(byteArrayOutputStream, keystorePassword.toCharArray());
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOG.error("Fatal error convert keystore to bytes", (Throwable)e);
        }
        return byteArrayOutputStream.toByteArray();
    }

    public KeyStore createKeyStoreWithSelfSignCert(String keyStoreName, String keystorePassword, String distinguishedName, String keyAlias, int validity) {
        return this.createKeyStoreWithSelfSignCert(keyStoreName, keystorePassword, distinguishedName, keyAlias, validity, "RSA", 2048, "SHA1WithRSAEncryption");
    }

    public KeyStore createKeyStoreWithSelfSignCert(String keyStoreName, String keystorePassword, String distinguishedName, String keyAlias, int validity, String keyAlg, int keySize, String signAlg) {
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(keyAlg);
            keyPairGenerator.initialize(keySize);
            KeyPair keyPair = keyPairGenerator.generateKeyPair();
            X509V3CertificateGenerator x509V3CertGen = new X509V3CertificateGenerator();
            x509V3CertGen.setSerialNumber(this.getSecureRandomeInt());
            x509V3CertGen.setIssuerDN((X509Name)new X509Principal(distinguishedName));
            x509V3CertGen.setNotBefore(new Date(System.currentTimeMillis()));
            x509V3CertGen.setNotAfter(new Date(System.currentTimeMillis() + 86400000L * (long)validity));
            x509V3CertGen.setSubjectDN((X509Name)new X509Principal(distinguishedName));
            x509V3CertGen.setPublicKey(keyPair.getPublic());
            x509V3CertGen.setSignatureAlgorithm(signAlg);
            X509Certificate x509Cert = x509V3CertGen.generateX509Certificate(keyPair.getPrivate());
            KeyStore ctlKeyStore = KeyStore.getInstance("JKS");
            ctlKeyStore.load(null, keystorePassword.toCharArray());
            ctlKeyStore.setKeyEntry(keyAlias, keyPair.getPrivate(), keystorePassword.toCharArray(), new Certificate[]{x509Cert});
            LOG.info("{} is created", (Object)keyStoreName);
            return ctlKeyStore;
        }
        catch (IOException | SecurityException | InvalidKeyException | KeyStoreException | NoSuchAlgorithmException | SignatureException | CertificateException e) {
            LOG.error("Fatal error creating keystore", (Throwable)e);
            return null;
        }
    }

    public KeyStore createEmptyKeyStore(String keystorePassword) {
        try {
            KeyStore trustKeyStore = KeyStore.getInstance("JKS");
            trustKeyStore.load(null, keystorePassword.toCharArray());
            return trustKeyStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOG.error("Failed to create trust keystore", (Throwable)e);
            return null;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean exportKeystore(KeyStore keystore, String keystorePassword, String fileName) {
        if (keystore == null) {
            return false;
        }
        try (FileOutputStream fOutputStream = new FileOutputStream(this.workingDir + fileName);){
            keystore.store(fOutputStream, keystorePassword.toCharArray());
            boolean bl = true;
            return bl;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOG.error("Fatal error export keystore", (Throwable)e);
            return false;
        }
    }

    public String generateCertificateReq(KeyStore keyStore, String keystorePassword, String keyAlias, String signAlg, boolean withTag) {
        try {
            if (keyStore.containsAlias(keyAlias)) {
                X509Certificate odlCert = (X509Certificate)keyStore.getCertificate(keyAlias);
                PublicKey pubKey = odlCert.getPublicKey();
                PrivateKey privKey = (PrivateKey)keyStore.getKey(keyAlias, keystorePassword.toCharArray());
                String subject = odlCert.getSubjectDN().getName();
                X509Name xname = new X509Name(subject);
                String signatureAlgorithm = signAlg;
                PKCS10CertificationRequest csr = new PKCS10CertificationRequest(signatureAlgorithm, xname, pubKey, null, privKey);
                String certReq = DatatypeConverter.printBase64Binary((byte[])csr.getEncoded());
                if (withTag) {
                    StringBuilder sb = new StringBuilder();
                    sb.append("-----BEGIN CERTIFICATE REQUEST-----");
                    sb.append("\n");
                    sb.append(certReq);
                    sb.append("\n");
                    sb.append("-----END CERTIFICATE REQUEST-----");
                    return sb.toString();
                }
                return certReq;
            }
            LOG.info("KeyStore does not contain alias {}", (Object)keyAlias);
            return "";
        }
        catch (InvalidKeyException | KeyStoreException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException | UnrecoverableKeyException e) {
            LOG.error("Failed to generate certificate request", (Throwable)e);
            return "";
        }
    }

    public String getCertificate(KeyStore keyStore, String certAlias, boolean withTag) {
        try {
            if (keyStore.containsAlias(certAlias)) {
                X509Certificate odlCert = (X509Certificate)keyStore.getCertificate(certAlias);
                String cert = DatatypeConverter.printBase64Binary((byte[])odlCert.getEncoded());
                if (withTag) {
                    StringBuilder sb = new StringBuilder();
                    sb.append("-----BEGIN CERTIFICATE-----");
                    sb.append("\n");
                    sb.append(cert);
                    sb.append("\n");
                    sb.append("-----END CERTIFICATE-----");
                    return sb.toString();
                }
                return cert;
            }
            LOG.info("KeyStore does not contain alias {}", (Object)certAlias);
            return "";
        }
        catch (KeyStoreException | CertificateException e) {
            LOG.error("Failed to get Certificate", (Throwable)e);
            return "";
        }
    }

    private X509Certificate getCertificate(String certificate) {
        if (certificate.isEmpty()) {
            return null;
        }
        if (certificate.contains("-----BEGIN CERTIFICATE-----")) {
            int fIdx = certificate.indexOf("-----BEGIN CERTIFICATE-----") + "-----BEGIN CERTIFICATE-----".length();
            int sIdx = certificate.indexOf("-----END CERTIFICATE-----");
            certificate = certificate.substring(fIdx, sIdx);
        }
        byte[] byteCert = Base64.getDecoder().decode(certificate);
        ByteArrayInputStream inputStreamCert = new ByteArrayInputStream(byteCert);
        try {
            CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
            X509Certificate newCert = (X509Certificate)certFactory.generateCertificate(inputStreamCert);
            newCert.checkValidity();
            return newCert;
        }
        catch (CertificateException e) {
            LOG.error("Failed to get certificate", (Throwable)e);
            return null;
        }
    }

    private BigInteger getSecureRandomeInt() {
        SecureRandom secureRandom = new SecureRandom();
        BigInteger bigInt = BigInteger.valueOf(secureRandom.nextInt());
        return new BigInteger(1, bigInt.toByteArray());
    }

    public KeyStore loadKeyStore(byte[] keyStoreBytes, String keystorePassword) {
        try {
            KeyStore keyStore = KeyStore.getInstance("JKS");
            keyStore.load(new ByteArrayInputStream(keyStoreBytes), keystorePassword.toCharArray());
            return keyStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOG.error("Fatal error load keystore", (Throwable)e);
            return null;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public KeyStore loadKeyStore(String keyStoreName, String keystorePassword) {
        try (FileInputStream fInputStream = new FileInputStream(this.workingDir + keyStoreName);){
            KeyStore keyStore2 = KeyStore.getInstance("JKS");
            keyStore2.load(fInputStream, keystorePassword.toCharArray());
            KeyStore keyStore = keyStore2;
            return keyStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOG.error("failed to get keystore {}", (Object)e.getMessage());
            return null;
        }
    }
}

