/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.persistence;

import akka.actor.ExtendedActorSystem;
import akka.dispatch.Futures;
import akka.persistence.SelectedSnapshot;
import akka.persistence.SnapshotMetadata;
import akka.persistence.SnapshotSelectionCriteria;
import akka.persistence.serialization.Snapshot;
import akka.persistence.serialization.SnapshotSerializer;
import akka.persistence.snapshot.japi.SnapshotStore;
import akka.serialization.JavaSerializer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.io.ByteStreams;
import com.typesafe.config.Config;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.jdt.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.concurrent.ExecutionContext;
import scala.concurrent.Future;

public class LocalSnapshotStore
extends SnapshotStore {
    private static final Logger LOG = LoggerFactory.getLogger(LocalSnapshotStore.class);
    private static final int PERSISTENCE_ID_START_INDEX = "snapshot-".length();
    private final ExecutionContext executionContext;
    private final int maxLoadAttempts;
    private final File snapshotDir;

    public LocalSnapshotStore(Config config) {
        this.executionContext = this.context().system().dispatchers().lookup(config.getString("stream-dispatcher"));
        this.snapshotDir = new File(config.getString("dir"));
        int localMaxLoadAttempts = config.getInt("max-load-attempts");
        this.maxLoadAttempts = localMaxLoadAttempts > 0 ? localMaxLoadAttempts : 1;
        LOG.debug("LocalSnapshotStore ctor: snapshotDir: {}, maxLoadAttempts: {}", (Object)this.snapshotDir, (Object)this.maxLoadAttempts);
    }

    public void preStart() throws Exception {
        if (!(this.snapshotDir.isDirectory() || this.snapshotDir.mkdirs() || this.snapshotDir.isDirectory())) {
            throw new IOException("Failed to create snapshot directory " + this.snapshotDir.getCanonicalPath());
        }
        super.preStart();
    }

    public Future<Optional<SelectedSnapshot>> doLoadAsync(String persistenceId, SnapshotSelectionCriteria criteria) {
        LOG.debug("In doLoadAsync - persistenceId: {}, criteria: {}", (Object)persistenceId, (Object)criteria);
        Deque metadatas = this.getSnapshotMetadatas(persistenceId, criteria).stream().sorted(LocalSnapshotStore::compare).collect(LocalSnapshotStore.reverse()).stream().limit(this.maxLoadAttempts).collect(Collectors.toCollection(ArrayDeque::new));
        if (metadatas.isEmpty()) {
            return Futures.successful(Optional.empty());
        }
        LOG.debug("doLoadAsync - found: {}", (Object)metadatas);
        return Futures.future(() -> this.doLoad(metadatas), (ExecutionContext)this.executionContext);
    }

    private Optional<SelectedSnapshot> doLoad(Deque<SnapshotMetadata> metadatas) throws IOException {
        SnapshotMetadata metadata = metadatas.removeFirst();
        File file = this.toSnapshotFile(metadata);
        LOG.debug("doLoad {}", (Object)file);
        try {
            Object data = this.deserialize(file);
            LOG.debug("deserialized data: {}", data);
            return Optional.of(new SelectedSnapshot(metadata, data));
        }
        catch (IOException e) {
            LOG.error("Error loading snapshot file {}, remaining attempts: {}", new Object[]{file, metadatas.size(), e});
            if (metadatas.isEmpty()) {
                throw e;
            }
            return this.doLoad(metadatas);
        }
    }

    private Object deserialize(File file) throws IOException {
        return JavaSerializer.currentSystem().withValue((ExtendedActorSystem)this.context().system(), () -> {
            try (ObjectInputStream in = new ObjectInputStream(new FileInputStream(file));){
                Object object = in.readObject();
                return object;
            }
            catch (ClassNotFoundException e) {
                throw new IOException("Error loading snapshot file " + file, e);
            }
            catch (IOException e) {
                LOG.debug("Error loading snapshot file {}", (Object)file, (Object)e);
                return this.tryDeserializeAkkaSnapshot(file);
            }
        });
    }

    private Object tryDeserializeAkkaSnapshot(File file) throws IOException {
        LOG.debug("tryDeserializeAkkaSnapshot {}", (Object)file);
        SnapshotSerializer snapshotSerializer = new SnapshotSerializer((ExtendedActorSystem)this.context().system());
        try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(file));){
            Object object = ((Snapshot)snapshotSerializer.fromBinary(ByteStreams.toByteArray((InputStream)in))).data();
            return object;
        }
    }

    public Future<Void> doSaveAsync(SnapshotMetadata metadata, Object snapshot) {
        LOG.debug("In doSaveAsync - metadata: {}, snapshot: {}", (Object)metadata, snapshot);
        return Futures.future(() -> this.doSave(metadata, snapshot), (ExecutionContext)this.executionContext);
    }

    private Void doSave(SnapshotMetadata metadata, Object snapshot) throws IOException {
        File actual = this.toSnapshotFile(metadata);
        File temp = File.createTempFile(actual.getName(), null, this.snapshotDir);
        LOG.debug("Saving to temp file: {}", (Object)temp);
        try (ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream(temp));){
            out.writeObject(snapshot);
        }
        catch (IOException e) {
            LOG.error("Error saving snapshot file {}. Deleting file..", (Object)temp, (Object)e);
            if (!temp.delete()) {
                LOG.error("Failed to successfully delete file {}", (Object)temp);
            }
            throw e;
        }
        LOG.debug("Renaming to: {}", (Object)actual);
        try {
            Files.move(temp.toPath(), actual.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            LOG.warn("Failed to move {} to {}. Deleting {}..", new Object[]{temp, actual, temp, e});
            if (!temp.delete()) {
                LOG.error("Failed to successfully delete file {}", (Object)temp);
            }
            throw e;
        }
        return null;
    }

    public Future<Void> doDeleteAsync(SnapshotMetadata metadata) {
        LOG.debug("In doDeleteAsync - metadata: {}", (Object)metadata);
        return Futures.future(() -> this.doDelete(metadata), (ExecutionContext)this.executionContext);
    }

    public Future<Void> doDeleteAsync(String persistenceId, SnapshotSelectionCriteria criteria) {
        LOG.debug("In doDeleteAsync - persistenceId: {}, criteria: {}", (Object)persistenceId, (Object)criteria);
        return Futures.future(() -> this.doDelete(persistenceId, criteria), (ExecutionContext)this.executionContext);
    }

    private Void doDelete(String persistenceId, SnapshotSelectionCriteria criteria) {
        List<File> files = this.getSnapshotMetadatas(persistenceId, criteria).stream().flatMap(md -> Stream.of(this.toSnapshotFile((SnapshotMetadata)md))).collect(Collectors.toList());
        LOG.debug("Deleting files: {}", files);
        files.forEach(File::delete);
        return null;
    }

    private Void doDelete(SnapshotMetadata metadata) {
        Collection<File> files = this.getSnapshotFiles(metadata);
        LOG.debug("Deleting files: {}", files);
        files.forEach(File::delete);
        return null;
    }

    private Collection<File> getSnapshotFiles(String persistenceId) {
        String encodedPersistenceId = LocalSnapshotStore.encode(persistenceId);
        Object[] files = this.snapshotDir.listFiles((dir, name) -> {
            int persistenceIdEndIndex = name.lastIndexOf(45, name.lastIndexOf(45) - 1);
            return PERSISTENCE_ID_START_INDEX + encodedPersistenceId.length() == persistenceIdEndIndex && name.startsWith(encodedPersistenceId, PERSISTENCE_ID_START_INDEX) && !name.endsWith(".tmp");
        });
        if (files == null) {
            return Collections.emptyList();
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("getSnapshotFiles for persistenceId: {}, found files: {}", (Object)encodedPersistenceId, (Object)Arrays.toString(files));
        }
        return Arrays.asList(files);
    }

    private Collection<File> getSnapshotFiles(SnapshotMetadata metadata) {
        return this.getSnapshotFiles(metadata.persistenceId()).stream().filter(file -> {
            SnapshotMetadata possible = LocalSnapshotStore.extractMetadata(file);
            return possible != null && possible.sequenceNr() == metadata.sequenceNr() && (metadata.timestamp() == 0L || possible.timestamp() == metadata.timestamp());
        }).collect(Collectors.toList());
    }

    private Collection<SnapshotMetadata> getSnapshotMetadatas(String persistenceId, SnapshotSelectionCriteria criteria) {
        return this.getSnapshotFiles(persistenceId).stream().flatMap(file -> LocalSnapshotStore.toStream(LocalSnapshotStore.extractMetadata(file))).filter(arg_0 -> ((SnapshotSelectionCriteria)criteria).matches(arg_0)).collect(Collectors.toList());
    }

    private static Stream<SnapshotMetadata> toStream(@Nullable SnapshotMetadata md) {
        return md != null ? Stream.of(md) : Stream.empty();
    }

    private static @Nullable SnapshotMetadata extractMetadata(File file) {
        int sequenceNumberEndIndex;
        String name = file.getName();
        int persistenceIdEndIndex = name.lastIndexOf(45, (sequenceNumberEndIndex = name.lastIndexOf(45)) - 1);
        if (PERSISTENCE_ID_START_INDEX >= persistenceIdEndIndex) {
            return null;
        }
        try {
            String persistenceId = LocalSnapshotStore.decode(name.substring(PERSISTENCE_ID_START_INDEX, persistenceIdEndIndex));
            long sequenceNumber = Long.parseLong(name.substring(persistenceIdEndIndex + 1, sequenceNumberEndIndex));
            long timestamp = Long.parseLong(name.substring(sequenceNumberEndIndex + 1));
            return new SnapshotMetadata(persistenceId, sequenceNumber, timestamp);
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    private File toSnapshotFile(SnapshotMetadata metadata) {
        return new File(this.snapshotDir, String.format("snapshot-%s-%d-%d", LocalSnapshotStore.encode(metadata.persistenceId()), metadata.sequenceNr(), metadata.timestamp()));
    }

    private static <T> Collector<T, ?, List<T>> reverse() {
        return Collectors.collectingAndThen(Collectors.toList(), list -> {
            Collections.reverse(list);
            return list;
        });
    }

    private static String encode(String str) {
        try {
            return URLEncoder.encode(str, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            LOG.warn("Error encoding {}", (Object)str, (Object)e);
            return str;
        }
    }

    private static String decode(String str) {
        try {
            return URLDecoder.decode(str, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            LOG.warn("Error decoding {}", (Object)str, (Object)e);
            return str;
        }
    }

    @VisibleForTesting
    static int compare(SnapshotMetadata m1, SnapshotMetadata m2) {
        return (int)(!m1.persistenceId().equals(m2.persistenceId()) ? (long)m1.persistenceId().compareTo(m2.persistenceId()) : (m1.sequenceNr() != m2.sequenceNr() ? m1.sequenceNr() - m2.sequenceNr() : (m1.timestamp() != m2.timestamp() ? m1.timestamp() - m2.timestamp() : 0L)));
    }
}

